/* 
 *   Creation Date: <1999/06/26 02:48:56 samuel>
 *   Time-stamp: <1999/10/30 18:24:19 samuel>
 *   
 *	<mol_semaphore.h>
 *	
 *	Semaphore implementation based upon pthread conditions
 *	(semaphores are broken in glibc 1.99)
 *   
 *   Copyright (C) 1999 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#ifndef _H_MOL_SEMAPHORE
#define _H_MOL_SEMAPHORE

#include <pthread.h>

/* #define USE_GLIBC_SEM */

#ifdef USE_GLIBC_SE
#define		mol_sem_init(x,y,z)	sem_init((x),(y),(z))
#define		mol_sem_wait(x)		sem_wait((x))
#define		mol_sem_trywait(x)	sem_trywait((x))
#define		mol_sem_post(x)		sem_post((x))
#define		mol_sem_getvalue(x,y)	sem_getvalue((x),(y))
#define		mol_sem_destroy(x)	sem_destroy((x))
#else

/* This is a somewhat naive implementation of semaphores
 * build upon pthread. Note that it is NOT safe to call mol_sem_post
 * asynchronously (it is with a sem_post and working pthread).
 *
 */

typedef struct
{
	int		_val;
	pthread_cond_t	_cond;
	pthread_mutex_t	_mutex;
} mol_sem_t;

inline static int mol_sem_init( mol_sem_t *x, int dummy_pshared, int value ) \
{ 	
	if( dummy_pshared ) {
		errno = ENOSYS;
		return -1;
	}
	(x)->_val=(value);
	pthread_mutex_init( &(x)->_mutex, NULL );
	pthread_cond_init( &(x)->_cond, NULL );
	return 0;
}

inline static int mol_sem_post( mol_sem_t *x )
{
	pthread_mutex_lock( &(x)->_mutex );
	(x)->_val++;
	if( (x)->_val > 0 )
		pthread_cond_signal( &(x)->_cond );
	pthread_mutex_unlock( &(x)->_mutex );
	return 0;
}

inline static int mol_sem_wait( mol_sem_t *x )
{
	pthread_cleanup_push( (void*)pthread_mutex_unlock, &(x)->_mutex );
	pthread_mutex_lock( &(x)->_mutex );

	while( (x)->_val <=0 ){
		/* if we get canceled, then the mutex guaranteed to be locked */
		pthread_cond_wait( &(x)->_cond, &(x)->_mutex );
	}
	(x)->_val--;
	pthread_cleanup_pop(1);
	return 0;
}

inline static int mol_sem_trywait( mol_sem_t *x )
{
	int ret=0;
	pthread_mutex_lock( &(x)->_mutex );
	if( (x)->_val <= 0 ) {
		errno = EAGAIN;
		ret = -1;
	} else {
		(x)->_val--;
	}
	pthread_mutex_unlock( &(x)->_mutex );
	return ret;
}

inline static int mol_sem_destroy( mol_sem_t *x )
{
	/* XXX: Set error EBUSY if a thread is blocked */

	pthread_mutex_destroy( &x->_mutex );
	pthread_cond_destroy( &x->_cond );
	return 0;
}


#define mol_sem_getvalue( x )	( (x)->_val )

#endif	/* USE_GLIBC_SEM */

#endif   /* _H_MOL_SEMAPHORE */
