/* 
 *   Creation Date: <2001/01/31 21:05:23 samuel>
 *   Time-stamp: <2001/06/24 13:27:40 samuel>
 *   
 *	<cpu.c>
 *	
 *	
 *   
 *   Copyright (C) 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"
#include "molcpu.h"
#include "mac_registers.h"
#include "wrapper.h"
#include "memory.h"
#include "session.h"
#include "timer.h"
#include "res_manager.h"
#include "rvec.h"
#include "os_interface.h"
#include "constants.h"
#include "io.h"
#include "promif.h"
#include "mol_assert.h"

#include <sys/mman.h>


/* Assembler exports */
extern char	_macprog_start[];
extern char	_macprog_end[];

/* Statics */
static void	init_mregs( void );
static int	save_mregs( void );
static void	load_mregs( void );

static void	fix_cpu_node( void );

static int 	emulate_illegal_601( ulong inst, int *done );

static int	rvec_isi_trap( int dummy_rvec, ulong nip, ulong srr1 );
static int	rvec_dsi_trap( int dummy_rvec, ulong ea, ulong dsisr );
static int	rvec_alignment_trap( int dummy_rvec, ulong dar, ulong dsisr );
static int	rvec_trace_trap( int dummy_rvec );
static int	rvec_altivec_trap( int dummy_rvec );
static int	rvec_altivec_assist( int dummy_rvec, ulong srr1 );

static int	rvec_unexpected_mmu_cond( int dummy_rvec, int param1, int param2 );
static int	rvec_priv_inst( int dummy_rvec, ulong opcode );
static int	rvec_illegal_inst( int dummy_rvec, ulong opcode );
static int	rvec_spr_read( int dummy_rvec, int sprnum, int gprnum );
static int	rvec_spr_write( int dummy_rvec, int sprnum, ulong value );
static int	rvec_unusual_program_excep( int dummy_rvec, ulong opcode, ulong srr1 );
static int	rvec_break( int dummy_rvec, int break_flag );
static int	rvec_msr_pow( int dummy_rvec );

static int	rvec_io_read( int dummy_rvec, ulong mphys_ioaddr, void *usr_data );
static int	rvec_io_write( int dummy_rvec, ulong mphys_ioaddr, void *usr_data );

static int 	cmd_copyprog( int, char ** );
static int 	cmd_loadprog( int, char ** );
static int 	cmd_dt( int, char ** );
static int 	cmd_dtc( int, char ** );

static struct {
	int	break_flags;
} molcpu;


/************************************************************************/
/*	Initialization / Cleanup					*/
/************************************************************************/

void 
molcpu_init( void ) 
{
	memset( &molcpu, 0, sizeof(molcpu) );

	session_save_proc( save_mregs, NULL, kDynamicChunk );
	if( !loading_session() ) {
		char 	*src, *dest;
		ulong	size;
		/* move part of this to the debugger? */
		src = _macprog_start;
		dest = ram.lvbase;
		size = _macprog_end - _macprog_start;
		size = (size>ram.size)? ram.size : size;
		memmove( dest, src, size );

		init_mregs();
	} else {
		load_mregs();
	}
	fix_cpu_node();

	// Exceptions
	set_rvector( RVEC_DSI_TRAP, rvec_dsi_trap, "Soft DSI Trap", kFPU_Safe );
	set_rvector( RVEC_ISI_TRAP, rvec_isi_trap, "Soft ISI Trap", kFPU_Safe );
	set_rvector( RVEC_TRACE_TRAP, rvec_trace_trap, "Trace", kFPU_Safe );
	set_rvector( RVEC_ALIGNMENT_TRAP, rvec_alignment_trap, "Alignment Trap", kFPU_Safe );
	set_rvector( RVEC_ALTIVEC_UNAVAIL_TRAP, rvec_altivec_trap, "Altivec Unavail Trap", kFPU_Safe );
	set_rvector( RVEC_ALTIVEC_ASSIST, rvec_altivec_assist, "Altivec Assist", kFPU_Safe );

	// MMU Vectors
	set_rvector( RVEC_UNUSUAL_DSISR_BITS, rvec_unexpected_mmu_cond, "Unusual DSISR Bits", kFPU_Unsafe );
	set_rvector( RVEC_MMU_IO_SEG_ACCESS, rvec_unexpected_mmu_cond, "MMU IO Segment Access", kFPU_Unsafe );
	set_rvector( RVEC_BAD_NIP, rvec_unexpected_mmu_cond, "Bad NIP", kFPU_Unsafe );

	// Emulation Helpers
	set_rvector( RVEC_PRIV_INST, rvec_priv_inst, "Privileged Instruction", kFPU_Safe /*XXX*/ );
	set_rvector( RVEC_ILLEGAL_INST, rvec_illegal_inst, "Illegal Instruction", kFPU_Safe /*XXX*/ );
	set_rvector( RVEC_SPR_READ, rvec_spr_read, "SPR Read", kFPU_Safe /*XXX*/ );
	set_rvector( RVEC_SPR_WRITE, rvec_spr_write, "SPR Write", kFPU_Safe /*XXX*/ );
	set_rvector( RVEC_UNUSUAL_PROGRAM_EXCEP, rvec_unusual_program_excep, "Unusual Program Excep", kFPU_Safe );

	// I/O READ WRITE
	set_rvector( RVEC_IO_READ, rvec_io_read, "IO Read", kFPU_Unsafe /*XX*/ );
	set_rvector( RVEC_IO_WRITE, rvec_io_write, "IO Write", kFPU_Unsafe /*XX*/ );

	// MSR_POW (DOZE)
	set_rvector( RVEC_MSR_POW, rvec_msr_pow, "MSR POW (DOZE)", kFPU_Unsafe );

	// Debugger
	set_rvector( RVEC_BREAK, rvec_break, "Break", kFPU_Unsafe );

	add_cmd( "copyprog", "copyprog \nCopy macprog to address 0x0\n", -1, cmd_copyprog );
	add_cmd( "loadprog", "loadprog file\nLoad macprog to address 0x0\n", -1, cmd_loadprog );
	add_cmd( "dt", "dt\nShow kernel debug trace\n", -1, cmd_dt );
	add_cmd( "dtc", "dtc\nClear debug trace\n", -1, cmd_dtc );
}

void
molcpu_cleanup( void )
{
}

static int
save_mregs( void )
{
	return write_session_data( "REGS", 0, (char*)mregs, sizeof(mac_regs_t) );
}

static void
load_mregs( void )
{
	mac_regs_t old = *mregs;
	if( read_session_data( "REGS", 0, (char*)mregs, sizeof(mac_regs_t) ))
		session_failure("Could not read registers\n");
	
	// There are some fields we should not touch...
	mregs->timer_stamp = old.timer_stamp;
	mregs->flag_bits = old.flag_bits;
	mregs->interrupt = 1;
}


static void 
init_mregs( void ) 
{
	static struct { int id, alias; char *salias; int altivec, rev, hid0; char *name; } cpu_info[] = {
		{8,	750,	"G3",	0, 0x202,	0,	"PowerPC G3"},	/* default */
		{1,	601,	NULL,	0, 1,	    0x80010080,	"PowerPC 601"},
		{3,	603,	NULL,	0, 1,		0,	"PowerPC 603 - UNIMPLEMENTED"},
		{4,	604,	NULL,	0, 0x103,	0,	"PowerPC 604"},
		{0xc,	7400,	"G4",	1, 0x209,	0,	"PowerPC G4"},
		{0x800c,7410,	NULL,	1, 0x1101,	0,	"PowerPC 7410"},
		{0x8000,7450,	NULL,	1, 0x201,	0,	"PowerPC 7450 - UNIMPLEMENTED"},
		{0,0,NULL,0,0,0,NULL}
	}, *ci;
	int 	rev, val, id=0;
	ulong	pvr;
	char	*pname;

	_get_pvr( &pvr );
	pvr >>= 16;

	// Don't clear mregs here. Some fields (in particular mregs->interrupt)
	// are already in use at this point.

	mregs->no_altivec = get_bool_res("disable_altivec")==1;
	for(ci=&cpu_info[0]; ci->id && ci->id != pvr; ci++ )
		break;
	if( ci->id )
		mregs->no_altivec |= !ci->altivec;
	else {
		mregs->no_altivec = 1;
		printm("Unknown processor id (%04lX)\n", pvr );
	}
	
	id = get_numeric_res("processor");
	if( !(pname=get_str_res("processor")) ) {
		id = 8;
		if( is_a_601() )
			id = 1;
		else if( !mregs->no_altivec )
			id = 0xc;
	}
	for(ci=&cpu_info[0]; ci->id; ci++ )
		if( ci->id == id || ci->alias == id || (pname && ci->salias && !strcasecmp(pname,ci->salias) ))
			break;
	if( !ci->id )
		ci=&cpu_info[0];
	mregs->no_altivec |= !ci->altivec;
	if( ci->altivec && mregs->no_altivec ) {
		printm("AltiVec unavailable or disabled by user - reverting to G3 mode\n");
		ci=&cpu_info[0];
	}

	mregs->processor = ci->id;
	rev = ((val=get_numeric_res("processor_rev"))!= -1) ? val : ci->rev;

	mregs->spr[S_PVR] = (mregs->processor<<16) | (rev & 0xffff);
	mregs->spr[S_HID0] = ci->hid0;
	mregs->nip = 0xfff00100;
	mregs->msr = MSR_ME + MSR_IP;
	mregs->_msr = MSR_ME | MSR_FE0 | MSR_FE1 | MSR_IR | MSR_DR | MSR_PR | MSR_EE;

	printm("Running in %s mode\n", ci->name );
}

static void
fix_cpu_node( void )
{
	static char *pnames[] = { "timebase-frequency", "bus-frequency", "clock-frequency", NULL };
	static struct { int id; char *resname; } ntab[] = {
		{1,	"ppc601_cpu"}, 
		{3,	"ppc603_cpu"}, 
		{4,	"ppc604_cpu"},
		{8,	"ppc750_cpu"}, 
		{0xc,	"ppc7400_cpu"},
		{0x800c,"ppc7410_cpu"},
		{0,NULL}
	}, *ci;
	mol_device_node_t *hw_dn, *dn, *par;
	int i, len;
	char *p, *filename;

	for(ci=&ntab[0]; ci->id && ci->id != mregs->processor; ci++ )
		;
	assert(ci->resname);

	// Add the correct CPU node to the device tree
	par = (dn = prom_find_type("cpu"))? dn->parent : prom_find_type("cpus");
	assert(par);
	while( (dn=prom_find_type("cpu")) )
		prom_delete_node( dn );
	if( !(filename=get_str_res(ci->resname)) || !prom_import_node( par, filename ) ){
		printm("Failed to import cpu device tree node from '%s' (%s).\n", filename, ci->resname );
		exit(1);
	}
	
	// Copy some CPU properties from the hardware tree	
	if( !(hw_dn = prom_find_type_hw( "cpu" )) )
		printm("Warning: CPU node missing from hardware tree!\n");
	if( !(dn = prom_find_type( "cpu" )) )
		printm("Warning: CPU node missing from oftree!\n");
	if( !dn || !hw_dn ) {
		printm("The calibration will probably be wrong... will try running anyway.\n");
		printm("Please contact your favorite MOL developer about this problem.\n");
		return;
	}
	for( i=0; pnames[i]; i++ ){
		if( !(p = prom_get_property( hw_dn, pnames[i], &len )) ) {
			printm("Property 'cpu/%s' is missing!\n", pnames[i] );
			continue;
		}
		prom_add_property( dn, pnames[i], p, len );
	}
}


/************************************************************************/
/*	Exceptions							*/
/************************************************************************/

static inline void 
mac_exception( int vector, int sbits ) 
{
	int msr = mregs->msr;

	mregs->spr[S_SRR0] = mregs->nip;
	mregs->spr[S_SRR1] = (msr & (0xffff|MSR_VEC)) | (sbits & ~(0xffff|MSR_VEC));
	msr &= (MSR_ME | MSR_IP);
	mregs->nip = (msr & MSR_IP) ? (0xfff00000 | vector) : vector;
	mregs->msr = msr;
	_msr_changed();
}

static int
rvec_dsi_trap( int dummy_rvec, ulong dar, ulong dsisr )
{
	// printm("DSI: DAR %08lX, DSISR %08lX\n", dar, dsisr );

	mregs->spr[S_DAR] = dar;
	mregs->spr[S_DSISR] = dsisr;
	mac_exception( 0x300, 0 );
	return 0;
}

static int
rvec_isi_trap( int dummy_rvec, ulong nip, ulong srr1 )
{
	// printm("ISI: NIP %08lX, SRR1 %08lX\n", nip, srr1 );
	mac_exception( 0x400, srr1 );
	return 0;
}

static int
rvec_alignment_trap( int dummy_rvec, ulong dar, ulong dsisr )
{
	// printm("Alignement Trap: NIP %08lX, SRR1 %08lX\n", dar, dsisr );
	mregs->spr[S_DAR] = dar;
	mregs->spr[S_DSISR] = dsisr;
	mac_exception( 0x600, 0 );
	return 0;
}


void
irq_exception( void )
{
	// This function is called from mainloop.c
	mac_exception( 0x500, 0 );
}

void
dec_exception( void )
{
	// This function is called from mainloop.c
	mregs->flag_bits &= ~fb_DecINT;
	mac_exception( 0x900, 0 );
}

static int
rvec_altivec_trap( int dummy_rvec )
{
	if( mregs->no_altivec ) {
		printm("AltiVec is disabled\n");
		mac_exception( 0x700, BIT(12) );
	} else {
		mac_exception( 0xf20, 0 );
	}
	return 0;
}

static int
rvec_altivec_assist( int dummy_rvec, ulong srr1 )
{
	/* 7400 */
	mac_exception( 0x1600, srr1 );
	return 0;
}

static int
rvec_trace_trap( int dummy_rvec )
{
	// This might be a debugger-trace
	if( molcpu.break_flags & BREAK_SINGLE_STEP ) {
		rvec_break( RVEC_BREAK, BREAK_SINGLE_STEP );
		return 0;
	}
#if 1
	printm("MAC-TRACE (!)\n");
//	stop_emulation();
#endif
	mac_exception( 0xd00, 0 );
	return 0;
}


/************************************************************************/
/*	POW 1->0 (DOZE)							*/
/************************************************************************/

static int 
rvec_msr_pow( int dummy_rvec )
{
#if 1
	mregs->msr &= ~MSR_POW;
#endif
	if( (mregs->processor >= 8 || mregs->processor == 3) 
	    && (mregs->spr[S_HID0] & (BIT(8)|BIT(9)|BIT(10))) )
		doze();

//	printm("MSR_POW 0->1, Unimplemented\n");
	return 0;
}


/************************************************************************/
/*	Emulation (most is done in the kernel)				*/
/************************************************************************/

static int
rvec_priv_inst( int dummy_rvec, ulong inst )
{
	int	op, op_ext, b1, b2, b3;

	// Unhandled privileged instruction in supervisor mode

	op = OPCODE( inst );
	op_ext = OPCODE_EXT( inst );
	b1 = B1( inst );	/* bit 6-10 */
	b2 = B2( inst );	/* bit 11-15 */
	b3 = B3( inst );	/* bit 16-20 */

	switch( SW_OP(op,op_ext) ) {
	case SW_OP( 31, 370 ):	/* tlbia (opt.) */
		// not implemented on the 601,603,604,G3 (G4?)
		break;
	case SW_OP( 31, 470 ):  /* dcbi rA,rB  -- rA=b2 rB=b3 */
		printm("dcbi treated as nop\n");
		mregs->nip += 4;
		return 0;
	default:
		printm("Unknown privileged instruction, opcode %lX\n", inst);
		stop_emulation();
		break;
	}
	mac_exception( 0x700, BIT(13) );
	return 0;
}

static int
rvec_illegal_inst( int dummy_rvec, ulong inst )
{
	int ret=0;
	int done=0;

	if( inst == BREAKPOINT_OPCODE ){
		is_stop_breakpoint(mregs->nip);
		stop_emulation();
		return kRVecExit;
	}
	
	// CPU flavor compatibility stuff (not complete)
	if( mregs->processor == 1 ) {
		ret = emulate_illegal_601( inst, &done );
	}
	if( done )
		return ret;

	// printm("ILLEGAL INSTRUCTION %08lX\n", inst );
	// stop_emulation();
	mac_exception( 0x700, BIT(12) );
	return 0;
}

static void
fix_thrm_spr( void )
{
	ulong v, t;
	int i;

	if( !(mregs->spr[SPRN_THRM3] & THRM3_E) )
		return;

	// XXX: Thermal interrupts are unimplemented
	for( i=SPRN_THRM1 ; i<= SPRN_THRM2 ; i++ ) {
		v = mregs->spr[i];
		if( !(v & THRM1_V ))
			continue;
		v |= THRM1_TIV;
		v &= ~THRM1_TIN;
		t = v & THRM1_THRES(127);
		if( (v & THRM1_TID) && t < THRM1_THRES(24) )
			v |= THRM1_TIN;
		if( !(v & THRM1_TID) && t > THRM1_THRES(24) )
			v |= THRM1_TIN;
		mregs->spr[i] = v;
	}
}

static int
rvec_spr_read( int dummy_rvec, int sprnum, int gprnum )
{
	int ret = kRVecGPRsModified;

	// Emulate SPR (in supervisor mode). We must return 
	// kRVecGPRsModified if we touch the gprs.

	switch( sprnum ){
	case SPRN_MMCR0:	// Monitor Mode Control Register 0
	case SPRN_MMCR1:	// Monitor Mode Control Register 1
	case SPRN_PMC1:		// Performance Counter Register 1
	case SPRN_PMC2:		// Performance Counter Register 2
	case SPRN_PMC3:		// Performance Counter Register 3
	case SPRN_PMC4:		// Performance Counter Register 4
	case SPRN_SIA:		// Sampled Instruction Address Register
	case SPRN_SDA:		// Sampled Data Address Register
	case SPRN_L2CR:		// L2 Cache Control Register
	case S_HID0:
	case S_HID1:
	case S_HID2:		// IABR
	case S_HID5:		// DABR
	case S_HID15:
	case SPRN_ZPR:		// Zone Protection Register (?)
	case 951:		// 0x3b7 (?)
	case SPRN_ICCR:		// Instruction Cache Cacheability Register
		mregs->gpr[gprnum] = mregs->spr[sprnum];
		break;
	case SPRN_THRM1:	// Thermal 1
	case SPRN_THRM2:	// Thermal 2
	case SPRN_THRM3:	// Thermal 3
		fix_thrm_spr();
		mregs->gpr[gprnum] = mregs->spr[sprnum];
		break;
	default:
		printm("Read from unimplemented SPR #%d\n", sprnum );
		stop_emulation();
	}
	mregs->nip += 4;
	return ret;
}


static int
rvec_spr_write( int dummy_rvec, int sprnum, ulong value )
{
	switch( sprnum ){
	case SPRN_MMCR0:	// Monitor Mode Control Register 0
	case SPRN_MMCR1:	// Monitor Mode Control Register 1
	case SPRN_PMC1:		// Performance Counter Register 1
	case SPRN_PMC2:		// Performance Counter Register 2
	case SPRN_PMC3:		// Performance Counter Register 3
	case SPRN_PMC4:		// Performance Counter Register 4
	case SPRN_SIA:		// Sampled Instruction Address Register
	case SPRN_SDA:		// Sampled Data Address Register
	case S_HID0:
	case S_HID1:
	case S_HID2:
	case S_HID5:
	case S_HID15:
	case SPRN_ZPR:		// Zone Protection Register (?)
	case 951:		// 0x3b7 (?)
	case SPRN_ICCR:		// Instruction Cache Cacheability Register
		mregs->spr[sprnum] = value;
		break;
	case SPRN_THRM1:	// Thermal 1
	case SPRN_THRM2:	// Thermal 2
	case SPRN_THRM3:	// Thermal 3
		mregs->spr[sprnum] = value;
		fix_thrm_spr();
		break;
	case SPRN_L2CR:			// L2 Cache Control Register
		value &= ~BIT(31);	// Clear L2IP (invalidate in progress)
		mregs->spr[sprnum] = value;
		break;
	default:
		printm("Write to unimplemented SPR #%d (%08lX)\n", sprnum, value );
		stop_emulation();
	}
	mregs->nip += 4;
	return 0;
}
	
static int
rvec_unusual_program_excep( int dummy_rvec, ulong opcode, ulong srr1 )
{
	printm("Unusual program exception occured (SRR1 = %08lX)\n", srr1 );
	stop_emulation();
	return 0;
}


/************************************************************************/
/*	MMU Return Vectors						*/
/************************************************************************/

static int
rvec_unexpected_mmu_cond( int rvec, int param1, int param2 )
{
	switch( rvec ){
	case RVEC_UNUSUAL_DSISR_BITS:	// dar, dsisr
		printm("RVEC_UNUSUAL_DSISR_BITS: dar %08X, dsisr %08X\n", param1, param2 );
		break;
	case RVEC_MMU_IO_SEG_ACCESS:
		printm("RVEC_MMU_IO_SEG_ACCESS\n");
		break;
	case RVEC_BAD_NIP: // nip_phys
		printm("Instruction Pointer not in ROM/RAM (nip %08lX, nip_phys %08X)\n", mregs->nip, param1 );
		stop_emulation();
		break;
	default:
		printm("rvec_unexpected_mmu_cond, unimplemented vector %d\n", rvec );
		break;
	}
	return 0;
}


/************************************************************************/
/*	IO Read/Write							*/
/************************************************************************/

static int
rvec_io_read( int dummy_rvec, ulong mphys_ioaddr, void *usr_data )
{
	ulong	inst = mregs->inst_opcode;
	int	op, op_ext, rD, rA, rB, d;
	int	flag = 0, ret=kRVecGPRsModified;
	ulong	ea, cont;

	enum {	byte=1, half=2, word=4, len_mask=7, indexed=8, update=16, 
		zero=32, reverse=64, nop=128, fpinst=256 };
#if 0
	printm("rvec_io_read: %08lX usr_data: %08lX\n", mphys_ioaddr, (ulong)usr_data );
	stop_emulation();
#endif
	/* break instruction into parts */
	op = OPCODE( inst );	/* bit 0-5 */
	op_ext = OPCODE_EXT( inst );
	rD = B1( inst );	/* bit 6-10 */
	rA = B2( inst );	/* bit 11-15 */
	rB = B3( inst );	/* bit 16-20 */
	d = BD( inst );		/* bit 16-31 sign extended */

	switch( op ) {
	case 34: /* lbz rD,d(rA) */
		flag = byte | zero; break;
	case 35: /* lbzu rD,d(rA) */
		flag = byte | zero | update; break;
	case 40: /* lhz rD,d(rA) */
		flag = half | zero; break;
	case 41: /* lhzu rD,d(rA) */
		flag = half | zero | update; break;
	case 42: /* lha rD,d(rA) */
		flag = half; break;
	case 43: /* lhau rD,d(rA) */
		flag = half | update; break;
	case 32: /* lwz rD,d(rA) */
		flag = word | zero; break;
	case 33: /* lwzu, rD,d(rA) */
		flag = word | zero | update; break;
	case 50: /* lfd frD,d(rA) */			/* FPU */
		flag = word | fpinst | zero; break;
	case 51: /* lfdu frD, d(rA) */			/* FPU */
		flag = word | fpinst | update | zero; break;
	}

	if( !flag && op==31 ) {
		switch( op_ext ) {  /* lxxx rD,rA,rB */
		case 87: /* lbzx rD,rA,rB */
			flag = byte | indexed | zero; break;
		case 119: /* lbzux rD,rA,rB */
			flag = byte | indexed | zero | update; break;
		case 279: /* lhzx rD,rA,rB */
			flag = half | indexed | zero; break;
		case 311: /* lhzux rD,rA,rB */
			flag = half | indexed | zero | update; break;
		case 343: /* lhax rD,rA,rB */
			flag = half | indexed; break;
		case 375: /* lhaux rD,rA,rB */
			flag = half | indexed | update; break;
		case 23: /* lwzx rD,rA,rB */
			flag = word | indexed | zero; break;
		case 55: /* lwzux rD,rA,rB */
			flag = word | indexed | zero | update; break;
		case 790: /* lhbrx rS,rA,rB */
			flag = half | indexed | zero | reverse; break;
		case 534: /* lwbrx rS,rA,rB */
			flag = word | indexed | zero | reverse; break;
		case 599: /* lfdx frD,rA,rB */				/* FPU */
			flag = word | indexed | zero | fpinst; break;
		case 631: /* lfdux frD,rA,rB */				/* FPU */
			flag = word | indexed | zero | update | fpinst; break;
		case 86: /* dcbf rA,rB - cache instruction*/
			/* treat as nop if data-translation is off */
			flag = (mregs->msr & MSR_DR) ? 0 : nop; break;
		}
	}

	if( flag & len_mask) {	/* instruction found */
		if( flag & indexed ) { /* lxxx rD,rA,rB */
			ea = mregs->gpr[rB];
			ea += rA ? mregs->gpr[rA] : 0;
		} else { /* lxxx rD,d(rA) */
			ea = rA ? mregs->gpr[rA] : 0;
			ea += d;
		}

		/* ea is the mac untranslated address, */
		/* mphys_ioaddr is the mac-physical address */

		cont = 0;
		do_io_read( usr_data, mphys_ioaddr, (flag & len_mask), &cont );

		if( flag & byte ){
			cont &= 0xff;
		} else if( flag & half ) {
			cont &= 0xffff;
			if( !(flag & zero) )	/* algebraic */
				cont |= (cont & 0x8000)? 0xffff0000 : 0;
			if( flag & reverse ) {
				cont = ld_le16((ushort*)&cont);
				/* cont = (cont>>8) + ((cont&0xff)<<8); */
			}
		} else if( flag & reverse) {
			cont = ld_le32(&cont);
			/* cont = (cont>>24) + ((cont&0xff0000)>>8)
			   + ((cont&0xff00)<<8) + ((cont&0xff)<<24); */
		}
		if( !(flag & fpinst) )
			mregs->gpr[rD] = cont;
		else {
			/* FPU instruction */
			ret |= kRVecFPRsModified;

			mregs->fpr[rD].h = cont;
			/* check for 4K boundary crossings... */
			if( ((mphys_ioaddr+4) & 0xfff) < 4 )
				printm("emulate_load_data_inst: MMU translation might be bad\n");
			do_io_read( usr_data, mphys_ioaddr+4, 4, &cont );
			mregs->fpr[rD].l = cont;
		}
			
		if( (flag & update) && rA && (rA!=rD || (flag & fpinst)) )
			mregs->gpr[rA] = ea;
	}

	if( flag ) {
		mregs->nip += 4;
	} else {
		/* TODO! */
		/* Integer load/store Multiple instructions not implemented */

		printm("Unimplemented load instruction\n");
		stop_emulation();
	}
	return ret;
}

static int
rvec_io_write( int dummy_rvec, ulong mphys_ioaddr, void *usr_data )
{
	ulong	inst = mregs->inst_opcode;
	int	op, op_ext, rS, rA, rB, d;
	int	flag = 0, ret=0;
	ulong	ea, cont, len;

	enum {	byte=1, half=2, word=4, len_mask=7, indexed=8, update=16, 
		reverse=32, nop=64, fpinst=128 };
#if 0
	printm("rvec_io_write: %08lX usr_data: %08lX\n", mphys_ioaddr, (ulong)usr_data );
	stop_emulation();
#endif

	/* break instruction into parts */
	op = OPCODE( inst );
	op_ext = OPCODE_EXT( inst );
	rS = B1( inst );	/* bit 6-10 */
	rA = B2( inst );	/* bit 11-15 */
	rB = B3( inst );	/* bit 16-20 */
	d = BD( inst );		/* bit 16-31 sign extended */

	switch( op ) {
	case 38: /* stb rS,d(rA) */
		flag = byte ; break;
	case 39: /* stbu rS,d(rA) */
		flag = byte | update; break;
	case 44: /* sth rS,d(rA) */
		flag = half ;  break;
	case 45: /* sthu rS,d(rA) */
		flag = half | update; break;
	case 36: /* stw rS,d(rA) */
		flag = word ; break;
	case 37: /* stwud rS,d(rA) */
		flag = word | update; break;
	case 54: /* stfd frS,d(rA) */			/* FPU */
		/* printm("FPU store inst\n"); */
		flag = word | fpinst; break;
	case 55: /* stfdu frS,d(rA) */			/* FPU */
		/* printm("FPU store inst\n"); */
		flag = word | fpinst | update; break;
	}

	if( !flag && op==31 ) {
		switch( op_ext ) {
		case 215: /* stbx rS,rA,rB */
			flag = byte | indexed ; break;
		case 247: /* stbux rS,rA,rB */
			flag = byte | indexed | update; break;
		case 407: /* sthx rS,rA,rB */
			flag = half | indexed ; break;
		case 439: /* sthux rS,rA,rB */
			flag = half | indexed | update; break;
		case 151: /* stwx rS,rA,rB */
			flag = word | indexed ; break;
		case 183: /* stwux rS,rA,rB */
			flag = word | indexed | update; break;
		case 918: /* sthbrx rS,rA,rB */
			flag = half | indexed | reverse; break;
		case 662: /* stwbrx rS,rA,rB */
			flag = word | indexed | reverse; break;
		case 727: /* stfdx frS,rA,rB */
			/* printm("FPU store inst\n"); */
			flag = word | indexed | fpinst; break;
		case 759: /* stfdux frS,rA,rB */
			/* printm("FPU store inst\n"); */
			flag = word | indexed | update | fpinst; break;
		}
	}

	if( flag & len_mask ) {	/* instruction found */
		if( flag & indexed ) { /* stxxx rS,rA,rB */
			ea = mregs->gpr[rB];
			ea += rA ? mregs->gpr[rA] : 0;
		} else { /* stxxx rS,d(rA) */
			ea = rA ? mregs->gpr[rA] : 0;
			ea += d;
		}
		if( !(flag & fpinst ) )
			cont = mregs->gpr[rS];
		else {
			cont = mregs->fpr[rS].h;
		}
		len = flag & len_mask;
		if( flag & byte ) {
			cont &= 0xff;
		} else if( flag & half ) {
			cont &= 0xffff;
			if( flag & reverse ) {
				cont = ld_le16( (ushort*)&cont );
				/* cont = (cont>>8) + ((cont&0xff)<<8);*/
			}
		} else if( flag & reverse ) { 
			/* must be a word */
			cont = ld_le32(&cont);
			/* cont = (cont>>24) + ((cont&0xff0000)>>8)
			   + ((cont&0xff00)<<8) + ((cont&0xff)<<24); */
		}

		/* ea is the mac untranslated address, */
		/* mregs->mmu_ioaddr holds the translated address */

		do_io_write( usr_data, mphys_ioaddr, cont, len );
		if( flag & fpinst ){
			if( ((mphys_ioaddr+4) & 0xfff) < 4 )
				printm("emulate store data inst: Possible MMU translation error\n");
			do_io_write( usr_data, mphys_ioaddr+4, mregs->fpr[rS].l, 4 );
		}
		
		if( (flag & update) && rA  ) {
			mregs->gpr[rA] = ea;

			/* We have modified a GPR - the kRVecGPRsModified bit
			 * must be set in the return code.
			 */
			ret = kRVecGPRsModified;
		}
	}

	if( flag ) {
		mregs->nip += 4; /* skip over instruction */

	} else {
		/* TODO! */
		/* Integer load/store multiple, semaphore instructions */
		/* not implemented */
		printm("Unimplemented store instruction %08lX\n",inst);
		stop_emulation();
	}
	return ret;
}


/************************************************************************/
/*	Debugger Support						*/
/************************************************************************/

static int
rvec_break( int dummy_rvec, int break_flag )
{
	switch( break_flag ){
	case BREAK_RFI:
		printm("BREAK-RFI (%08lX)\n", mregs->nip );
		break;
	case BREAK_SINGLE_STEP:
		break;
	default:
		printm("rvec_break: Unknown break flag %d\n", break_flag );
		break;
	}
	stop_emulation();
	clear_break_flag( break_flag );
	return 0;
}

void 
set_break_flag( int flag )
{
	molcpu.break_flags |= flag;
	_breakpoint_flags( molcpu.break_flags );
}

void
clear_break_flag( int flag )
{
	molcpu.break_flags &= ~flag;
	_breakpoint_flags( molcpu.break_flags );
}

/************************************************************************/
/*	F U N C T I O N S						*/
/************************************************************************/

void 
soft_trap( int trap_num )
{
	printm("soft_trap unimplemented\n");
}


/************************************************************************/
/*	CPU emulation							*/
/************************************************************************/

static int
emulate_illegal_601( ulong inst, int *done ) 
{
	struct timeval tval;
	int op, op_ext, b1, b2, b3, ret=0;

	op = OPCODE( inst );
	op_ext = OPCODE_EXT( inst );
	b1 = B1( inst );		// bit 6-10
	b2 = B2( inst );		// bit 11-15
	b3 = B3( inst );		// bit 16-20

	switch( SW_OP(op,op_ext) ) {
	case SW_OP( 31, 339 ):	// mfspr b1=r# b3|b2=spr#
		switch( (b3<<5)+b2 ){
		case 0:	// MQ (601)
			printm("mfmq at %08lX\n", mregs->nip);
			mregs->gpr[b1] = 0;
			ret = kRVecGPRsModified;
			*done = 1;
			break;
		case 4: // RTCU, read (601) - counts seconds
			gettimeofday(&tval,NULL);
			mregs->gpr[b1] = tval.tv_sec;
			ret = kRVecGPRsModified;
			*done = 1;
			break;
		case 5: // RTCL, read (601) - counts ns, overflows at 10^10
			gettimeofday(&tval, NULL );
			mregs->gpr[b1] = tval.tv_usec * 1000;
			ret = kRVecGPRsModified;
			*done = 1;
			break;
		}
		break;
	}
	if( *done )
		mregs->nip += 4;
	return ret;
}



/************************************************************************/
/*	Debugger CMDs							*/
/************************************************************************/

static int 
cmd_copyprog( int argc, char **argv )
{
	char 	*src, *dest;
	size_t	size;
	
	if( argc!=1 )
		return 1;

	/* copy macprog to macram (phys addr 0) - used for debugging */
	src = _macprog_start;
	dest = ram.lvbase;
	size = _macprog_end - _macprog_start;
	size = (size>ram.size)? ram.size : size;
	memmove( dest, src, size );
	
	return 0;
}

static int
cmd_loadprog( int argc, char **argv )
{
	int fd;
	if( argc!=2 )
		return 1;

	if( (fd = open( argv[1], O_RDONLY )) < 0 ) {
		perrorm("open");
		return 0;
	}
	if( read( fd, ram.lvbase, ram.size ) < 0 )
		perrorm("read");	
	close( fd );
	return 0;
}

static int
cmd_dt( int argc, char **argv )
{
	int i, j = (mregs->debug_trace & 0xff);
	
	if( argc != 1 )
		return 1;

	for(i=0; i<256; i++ ) {
		int val = mregs->dbg_trace_space[(j-i-1)&0xff ];
		printm("%08x ", val );
		if( i%8==7 )
			printm("\n");
	}
	return 0;
}

static int
cmd_dtc( int argc, char **argv )
{
	int i;
	if( argc != 1 )
		return 1;
	for(i=0; i<256; i++ )
		mregs->dbg_trace_space[i]=0;
	mregs->debug_trace = 0;
	return 0;
}
