/**************************************************************
*   
*   Creation Date: <97/06/29 23:33:42 samuel>
*   Time-stamp: <2001/05/09 22:51:51 samuel>
*   
*	<ioports.c>
*	
*	Memory mapped I/O emulation
*   
*   Copyright (C) 1997, 1999, 2000, 2001 Samuel Rydh
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "mol_config.h"

#include "extralib.h"
#include "molcpu.h"
#include "memory.h"
#include "wrapper.h"
#include "debugger.h"
#include "ioports.h"
#include "debugger.h"
#include "booter.h"

#include "pci.h"
#include "pic.h"
#include "dbdma.h"
#include "promif.h"
#include "oldworld/include/scsi_main.h"
#include "hacks.h"

#include "driver_mgr.h"

/* #define PERFORMANCE_INFO */


/* Define physical reads pass on unmapped hardware reads
 * to the hardware - DANGEROUS but useful for testing
 */
/* #define PHYSICAL_READS */

/* #define IOPORTS_VERBOSE  */

/* #define ADD_OF_RANGES */

typedef struct io_range
{
	struct io_range	*next;		/* list, addition order */
	struct io_range *snext;		/* ordered list */

	char		*name;
	int		id;		/* identifier */

	ulong		mphys;		/* base */
	ulong		size;		/* must be a multiple of 8 */
	int		flags;

	char		*buf;		/* IO-buffer */

	void		*usr;		/* user information */
	io_ops_t	ops;

#ifdef PERFORMANCE_INFO
	ulong		num_reads;	/* performance counter */
	ulong		num_writes;	/* performance counter */
#endif
} io_range_t;

static io_range_t *root=0;
static io_range_t *sroot=0;		/* only intended for verbose output */

static int next_id=1;

static io_range_t *get_range( ulong addr );

static ulong unmapped_read( ulong mphys, int len, void *usr );
static void unmapped_write( ulong mphys, ulong data, int len, void *usr );
static void print_io_access( int isread, ulong addr, ulong value, 
			     int len, void *usr );

static void mem_io_read( ulong ptr, int len, ulong *retvalue );
static void mem_io_write( ulong ptr, ulong data, int len );

/* --- cmds --- */
static int cmd_ios( int, char**);	/* IO show */
static int cmd_iob( int, char**);	/* IO debug break */
static int cmd_iov( int, char**);	/* IO debug verbose */
static int cmd_iocv( int, char**);    	/* IO clear breaks */
static int cmd_iocb( int, char**);    	/* IO clear verbose */
static int cmd_iosv( int, char**);    	/* IO set breaks */
static int cmd_iosb( int, char**);    	/* IO set verbose */


/**************************************************************
*  ioports_init
*    
**************************************************************/

void 
ioports_init( void )
{
#ifdef ADD_OF_RANGES
	mol_device_node_t	*cur;
#endif
	next_id = 0;
	root = 0;

	/* debugger cmds */
	add_cmd( "ios", "ios \nshow IO-ranges\n", -1, cmd_ios );
	add_cmd( "iob", "iob addr \ntoggle IO-range break\n", -1, cmd_iob );
	add_cmd( "iov", "iov addr \ntoggle IO-range verbose flag\n", -1,cmd_iov);
	add_cmd( "iocv", "iocv \nclear all IO-range verbose flags\n",-1,cmd_iocv);
	add_cmd( "iocb", "iocb \nclear all IO-range break flags\n",-1,cmd_iocb);
	add_cmd( "iosv", "iosv \nset all IO-range verbose flags\n",-1,cmd_iosv);
	add_cmd( "iosb", "iosb \nset all IO-range break flags\n",-1,cmd_iosb);

	/* add a complete range */
	add_io_range( 0x80000000, 0x80000000, "IO_unmapped", 0, NULL, NULL );

#ifdef ADD_OF_RANGES
	/* add all mappings from the OF-device tree (info purose only) */
	cur = prom_get_root();
	for( ;cur;cur=cur->allnext) {
		int i;
		for( i=0 ; i<cur->n_addrs; i++ ) {
			if( cur->addrs[i].size == 0 )
				continue;
			if( (cur->addrs[i].size & 0x7)==0 )
				add_io_range( cur->addrs[i].address, cur->addrs[i].size, 
					      cur->name, 0, NULL, NULL );
		}
	}
#endif

	driver_mgr_init();
}


/**************************************************************
*  ioports_cleanup
*    
**************************************************************/

void 
ioports_cleanup( void ) 
{
	io_range_t *cur, *next;
	
#ifdef PERFORMANCE_INFO
	for( cur=root; cur; cur=cur->next ){
		printm("%s R/W (%ld/%ld)\n", cur->name, cur->num_reads, cur->num_writes );
	}
#endif
	driver_mgr_cleanup();

	for( cur=root; cur; cur=next ) {
		if( cur->ops.cleanup_fp )
			cur->ops.cleanup_fp( cur->usr );
		next = cur->next;
		free( cur );
	}
}


/**************************************************************
*  add_io_range
*    
**************************************************************/

int 
add_io_range( ulong mbase, size_t size, char *name, int flags,
	      io_ops_t *ops, void *usr )
{
	io_range_t	*ior, **p;

	if( mbase & 0x7 || size & 0x7 ){
		printm("ERROR: IO-area has incorrect alignment: %08lX %08X\n",
		       mbase, size);
		return 0;
	}

	if( !name )
		name = "NoName";
	if( !(ior = (io_range_t *) calloc(1, sizeof(io_range_t)+strlen(name)+1 )) ){
		printm("Out of memory");
		exit(1);
	}

	/* sorted list */
	for(p=&sroot; *p; p=&(**p).snext )
		if( (**p).mphys >= mbase )
			break;
#ifdef ADD_OF_RANGES
	if( *p && (**p).mphys == mbase && (**p).size == size )
		*p=(**p).snext;
#endif
	ior->snext = *p;
	*p = ior;

	/* ordered by addition time */
	ior->next = root;
	root = ior;

	ior->name = (char*)ior + sizeof(io_range_t);
	strcpy( ior->name, name );

	ior->id = next_id++;
	ior->mphys = mbase;
	ior->size = size;
	if( ops )
		ior->ops = *ops;
	if( !ior->ops.read_fp )
		ior->ops.read_fp = unmapped_read;
	if( !ior->ops.write_fp )
		ior->ops.write_fp = unmapped_write;
	if( !ior->ops.print_fp )
		ior->ops.print_fp = print_io_access;
		
	ior->flags = flags;
	ior->usr = usr;
	
	/* We do the size check to avoid VRAM/ROM etc, 
	 * which should be handled differently.
	 */
#if 1
	if( size <= 0x2000 )
		_add_io_range( mbase, size, ior );
#endif
	return ior->id;
}

int 
remove_io_range( int iorange_id )
{
	io_range_t **pn, *next, **pn2;

	for( pn = &root; *pn; pn = &(**pn).next )
		if( (**pn).id == iorange_id ) {
			/* Tell kernel this mapping is to be removed */
			_remove_io_range( (**pn).mphys, (**pn).size );

			/* We must remove ourselves from the sorted list also... */
			for( pn2 = &sroot; *pn2; pn2 = &(**pn2).snext )
				if( *pn2 == *pn ) {
					*pn2 = (**pn2).snext;
					break;
				}

			/* And cleanup */
			if( (**pn).ops.cleanup_fp )
				(**pn).ops.cleanup_fp( (**pn).usr );
			next = (**pn).next;
			free( *pn );
			*pn = next;
			return 0;
		}
	return 1;
}



/**************************************************************
*  get_range
*    return the first range that contains addr
**************************************************************/

static io_range_t *
get_range( ulong addr ) 
{
	io_range_t 	*cur;
	
	for( cur=root; cur; cur=cur->next )
		if( addr >= cur->mphys && addr <= cur->mphys+cur->size-1 )
			return cur;
	return 0;
}


/**************************************************************
*  mem_io_read
*    ptr 	physical addr
*    len 	1,2,4
*    ret: 1=error
*
*    MUST BE REENTRANT (could be called from DMA)
**************************************************************/

void 
do_io_read( void *_ior, ulong ptr, int len, ulong *retvalue )
{
	io_range_t *ior = _ior;
	ulong ret;

	if( !ior ){
		mem_io_read( ptr, len, retvalue );
		return;
	}

#ifdef PERFORMANCE_INFO
	ior->num_reads++;	/* Performance statistics */
#endif

	if( ior->flags & IO_STOP )
		stop_emulation();

	ret = ior->ops.read_fp( ptr, len, ior->usr );

	if( ior->flags & IO_VERBOSE )
		ior->ops.print_fp( 1 /*read*/, ptr, ret, len, ior->usr );

	if( retvalue )
		*retvalue = ret;
}

static void 
mem_io_read( ulong mphys, int len, ulong *retvalue ) 
{
	io_range_t 	*cur;

	cur = get_range( mphys );
	if( !cur ) {
		*retvalue = unmapped_read( mphys, len, NULL );
		return;
	}
	do_io_read( cur, mphys, len, retvalue );
}


/**************************************************************
*  mem_io_write
*    len = 1,2,4 - byte, half word, word
*    ret: 1=error
**************************************************************/

void 
do_io_write( void *_ior, ulong mphys, ulong data, int len )
{
	io_range_t *ior = _ior;
	if( !ior ) {
		mem_io_write(mphys, data, len);
		return;
	}

#ifdef PERFORMANCE_INFO
	ior->num_writes++;	/* Performance statistics */
#endif

	if( ior->flags & IO_STOP )
		stop_emulation();

	if( ior->flags & IO_VERBOSE )
		ior->ops.print_fp( 0 /* write */, mphys, data, len, ior->usr );

	ior->ops.write_fp( mphys, data, len, ior->usr );
}


static void 
mem_io_write( ulong mphys, ulong data, int len ) 
{
	io_range_t *cur;

	cur = get_range( mphys );
	if( !cur ) {
		unmapped_write( mphys, data, len, NULL );
		return;
	}

	do_io_write( cur, mphys, data, len );
	return;
}


/**************************************************************
*  fallback functions
*
*
**************************************************************/

static ulong 
unmapped_read( ulong mphys, int len, void *usr )
{
	/* XXX: this is starmax stuff. It should be
	 * moved somewhere else. In particular the identification
	 * register in 0xf8000000.
	 */
	if( get_hacks() & hack_starmax ) {
		if( mphys == 0xf8000000 )
			return 0x10020000;
		else if( mphys == 0xf301a000 )
			return 0xff77;
	}

	/* detect unmapped RAM access */
	if( mphys < 0x80000000 ){
		if( (mregs->msr & MSR_DR) || !is_oldworld_boot() ){
			printm("Unmapped 'RAM-read-access', %08lx\n",mphys );
			stop_emulation();
		}
		return 0;
	}
#ifdef IOPORTS_VERBOSE
	if( mphys >= 0xf0000000 )
		printm("IOPORTS: Read from unmapped memory %08lX\n", mphys );
#endif
#ifdef PHYSICAL_READS
	return ioread( mphys, len );
#endif

#if 0
	if( (mphys >>28) == 0xc ){
		/* This probably is an address from the linux kernel 
		 * - an example of register tunneling.
		 */
		printm("suspicious 0xc0000000 address...Breaking\n");
		stop_emulation();
	}
#endif

	/* silently ignore */
	return 0;
}

static void 
unmapped_write( ulong mphys, ulong data, int len, void *usr )
{
	if( mphys < 0x80000000 ){
		if( (mregs->msr & MSR_DR) || !is_oldworld_boot() ) {
			printm("Unmapped RAM-write-access, %08lx\n",mphys );
			/* stop_emulation(); */
		}
		return;
	}
#ifdef IOPORTS_VERBOSE
	if( mphys >= 0xf0000000 )
		printm("IOPORTS: Write to unmapped memory %08lX\n", mphys );
#endif
	/* silently ignore write to unampped I/O */
	return;
}

static void 
print_io_access( int isread, ulong addr, ulong value, 
			     int len, void *usr )
{
	io_range_t *ior = get_range( addr );
	
	printm("%s: %s Address %08lX  Data: ", 
	       ior? ior->name : "** NO RANGE **",
	       isread? "I/O read " : "I/O write", addr );

	printm( (len==1)? "%02X" : (len==2)? "%04X" : "%08X", value );
	printm("\n");
}


/************************************************************************/
/*	Debugger CMDs							*/
/************************************************************************/

/* --- IO show --- */
static int 
cmd_ios( int numargs, char** args){
	io_range_t *cur;
	if( numargs != 1 )
		return 1;

	for( cur=sroot; cur; cur=cur->snext ) {
		printm("%c%c",
		       cur->flags & IO_STOP ? '*' : ' ',
		       cur->flags & IO_VERBOSE ? 'V' : ' ');
		printm("  Start: %08lX  Size: %08lX  ",cur->mphys,cur->size );
		if( cur->name )
			printm("%s\n",cur->name );
		else
			printm("\n");
	}
	return 0;
}

/* --- IO debug break --- */
static int 
cmd_iob( int argv, char **args) {	/* IO debug break */
	io_range_t *r;
	ulong	addr;
	
	if( argv != 2 )
		return 1;

	addr = string_to_ulong(args[1]);
	r =  get_range( addr );
	if( !r ) {
		printm("No IO-range covers the address %08lX\n",addr );
		return 0;
	}
	
	if( r->flags & IO_STOP )
		r->flags &= ~IO_STOP;
	else
		r->flags |= IO_STOP;
	
	printm("%s: %08lX-%08lX ", r->name, r->mphys, r->mphys+r->size-1);
	if( r->flags & IO_STOP )
		printm("will break on access\n");
	else
		printm("break cleared\n");
	return 0;
}


/* --- IO debug print ---*/
static int 
cmd_iov( int argv, char **args) {	/* IO debug break */
	io_range_t *r;
	ulong	addr;
	
	if( argv != 2 )
		return 1;

	addr = string_to_ulong(args[1]);
	r =  get_range( addr );
	if( !r ) {
		printm("No IO-range covers the address %08lX\n",addr );
		return 0;
	}
	
	if( r->flags & IO_VERBOSE )
		r->flags &= ~IO_VERBOSE;
	else
		r->flags |= IO_VERBOSE;
	
	printm("%s: %08lX-%08lX ",r->name, r->mphys, r->mphys+r->size+1);
	if( r->flags & IO_VERBOSE )
		printm("will be verbose on access\n");
	else
		printm("verbose flag cleared\n");
	return 0;
}

/* --- IO-range clear verbose ---*/
static int 
cmd_iocv( int argv, char **args ) {
	io_range_t *r;
	if( argv != 1 )
		return 1;
	for( r=root; r; r=r->next )
		r->flags &= ~IO_VERBOSE;
	printm("Verbose flags cleared\n");
	return 0;
}

/* --- IO-range clear break ---*/
static int 
cmd_iocb( int argv, char **args ) {
	io_range_t *r;
	if( argv != 1 )
		return 1;
	for( r=root; r; r=r->next )
		r->flags &= ~IO_STOP;
	printm("Break flags cleared\n");
	return 0;
}

/* --- IO-range set verbose ---*/
static int 
cmd_iosv( int argv, char **args ) {
	io_range_t *r;
	if( argv != 1 )
		return 1;
	for( r=root; r; r=r->next )
		r->flags |= IO_VERBOSE;
	printm("Verbose flags set\n");
	return 0;
}

/* --- IO-range clear break ---*/
static int 
cmd_iosb( int argv, char **args ) {
	io_range_t *r;
	if( argv != 1 )
		return 1;
	for( r=root; r; r=r->next )
		r->flags |= IO_STOP;
	printm("Break flags set\n");
	return 0;
}


/**************************************************************
*  read_mem / write_mem
*    these functions simplify the access of emulated io-registers
**************************************************************/

ulong 
read_mem( char *ptr, int len )
{
	switch( len ){
	case 1:
		return *(unsigned char*)ptr;
	case 2:
		return *(unsigned short*)ptr;
	case 4:
		return *(ulong*)ptr;
	default:
		printm("Error in read_mem\n");
		break;
	}
	return 0;
}

void 
write_mem( char *ptr, ulong data, int len )
{
	switch( len ){
	case 1:
		*(unsigned char *)ptr = data;
		break;
	case 2:
		*(unsigned short*)ptr = data;
		break;
	case 4:
		*(ulong*)ptr = data;
		break;
	default:
		printm("Error in write_mem\n");
		break;
	}
}
