/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.remoting.transport.multiplex;

import org.jboss.logging.Logger;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.PortUtil;
import org.jboss.remoting.transport.multiplex.Multiplex;
import org.jboss.remoting.transport.multiplex.MultiplexingManager;
import org.jboss.test.remoting.transport.multiplex.utility.SuccessCountingTestCase;

import javax.management.MBeanServer;
import java.lang.reflect.Field;
import java.util.HashMap;


/**
 * 
 * @author <a href="mailto:ron.sigal@jboss.com">Ron Sigal</a>
 * <p>
 * Copyright (c) Jun 25, 2006
 * </p>
 */
public class MultiplexingManagerStaticThreadsTestCase extends SuccessCountingTestCase
{
   private static Logger log = Logger.getLogger(MultiplexingManagerStaticThreadsTestCase.class);
   
   private Connector connector;
   private InvokerLocator serverLocator;
   private String serverLocatorURI;
   private int connectPort;
   
   
   public void setUp() throws Exception
   {
      super.setUp();
      
      try
      {
         connectPort = PortUtil.findFreePort("localhost");
         serverLocatorURI = "multiplex://localhost:" + connectPort;
         serverLocator = new InvokerLocator(serverLocatorURI);
         connector = new Connector(serverLocator.getLocatorURI());
         connector.create();
         connector.addInvocationHandler("test", new SimpleServerInvocationHandler());
         connector.start();
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
      }
   }
   
   
   public void tearDown()
   {
      connector.stop();
   }
   
   
   /**
    * Verifies that the static threads in MultiplexingManager are handled correctly.
    *
    */
   public void testStaticThreadsShutdown()
   {
      log.info("entering: " + getName());
      
      try
      {
         // Make  Client use remote invoker.
         HashMap config = new HashMap();
         config.put(InvokerLocator.FORCE_REMOTE, "true");
         
         // Create Client and call invoke() to set up server invoker.
         Client client = new Client(serverLocator, config);
         client.connect();
         assertEquals("abc", client.invoke("abc"));
         
         // Get staticThreadsRunning field
         Field field = MultiplexingManager.class.getDeclaredField("staticThreadsRunning");
         field.setAccessible(true); 

         // Verify static threads are running
         boolean staticThreadsRunning = field.getBoolean(null);
         assertTrue(staticThreadsRunning);
         
         // Disconnect Client and wait for static threads to shut down.
         client.disconnect();
         Thread.sleep(Multiplex.STATIC_THREADS_MONITOR_PERIOD_DEFAULT * 2 + 1000);
         
         // Verify static threads are not running.
         staticThreadsRunning = field.getBoolean(null);
         assertFalse(staticThreadsRunning);
         
         // Recreate Client and call invoke() to set up server invoker.
         client = new Client(serverLocator, config);
         client.connect();
         assertEquals("xyz", client.invoke("xyz"));
         
         // Verify static threads are running again.
         staticThreadsRunning = field.getBoolean(null);
         assertTrue(staticThreadsRunning);
         
         client.disconnect();
         log.info(getName() + ": PASSES");
      }
      catch (Throwable t)
      {
         log.error(t);
         t.printStackTrace();
         log.error(getName() + ": FAILS");
         fail();
      }
   }
   
   class SimpleServerInvocationHandler implements ServerInvocationHandler
   {
      InvokerCallbackHandler handler;
      
      public void addListener(InvokerCallbackHandler callbackHandler)
      {
      }
      
      public Object invoke(InvocationRequest invocation) throws Throwable
      {
         return invocation.getParameter();
      }
      
      public void removeListener(InvokerCallbackHandler callbackHandler)
      {
      }
      
      public void setInvoker(ServerInvoker invoker)
      {
      }
      
      public void setMBeanServer(MBeanServer server)
      { 
      }
   }
}
