/* WorksheetWindowWrapper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.workspace;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.conjecture.engine.Engine;
import org.grinvin.worksheet.GraphListComponentType;
import org.grinvin.worksheet.WorksheetWindow;
import org.grinvin.worksheet.WorksheetWindowModel;
import org.grinvin.worksheet.WorksheetWindowModel.SubwindowType;
import org.grinvin.io.list.ConjectureHistoryListLoader;
import org.grinvin.io.list.ConjectureHistoryListSaver;
import org.grinvin.io.list.ConjectureListLoader;
import org.grinvin.io.list.ConjectureListSaver;
import org.grinvin.io.EngineLoader;
import org.grinvin.io.EngineSaver;
import org.grinvin.io.list.FilterListLoader;
import org.grinvin.io.list.FilterListSaver;
import org.grinvin.io.list.GraphGeneratorInstanceListLoader;
import org.grinvin.io.list.GraphGeneratorInstanceListSaver;
import org.grinvin.io.list.GraphListLoader;
import org.grinvin.io.list.GraphListSaver;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.list.InvariantListLoader;
import org.grinvin.io.list.InvariantListSaver;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionLoader;
import org.grinvin.io.SectionSaver;
import org.grinvin.list.ConjectureHistoryList;
import org.grinvin.list.ConjectureList;
import org.grinvin.list.FilterList;
import org.grinvin.list.generators.GraphGeneratorInstanceList;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.HasURI;
import org.grinvin.list.invariants.InvariantList;

import org.jdom.DocType;
import org.jdom.Document;
import org.jdom.Element;

/**
 * Wrapper for the conjecturing window.
 */
public class WorksheetWindowWrapper
        extends DefaultWindowWrapper<WorksheetWindow> {
    
    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//Worksheet 1.0//EN";
    
    @Override
    public WorksheetWindow create() {
        return new WorksheetWindow(new WorksheetWindowModel());
    }
    
    public String getElementName() {
        return "worksheetWindow";
    }
        
    /**
     * Load the window, its graph list and its invariant list.
     */
    @Override
    public WorksheetWindow load(Element element, SectionLoader sloader) throws IOException {
        try {
            URI uri = new URI(element.getAttributeValue("uri"));
            String name = uri.getSchemeSpecificPart();
            WorksheetWindow window = load(sloader, name);
            window.getModel().setURI(uri);
            return window;
        } catch (URISyntaxException ex) {
            throw new IOFormatException("invalid uri");
        }
    }
    
    //
    private WorksheetWindow load(SectionLoader sloader, String name) throws IOException {
        try {
            
            InputStream rootinput = sloader.openSection(name);
        
            Document document = LoaderSaverHelper.loadDocument(rootinput);
            if (!PUBLIC_ID.equals(document.getDocType().getPublicID()))
                throw new IOFormatException("Unkown document type: " + document.getDocType().getPublicID() + " expects " + PUBLIC_ID);
            Element element = document.getRootElement();
            
            WorksheetWindowModel worksheetWindowModel = new WorksheetWindowModel();
            
            // load graph list
            loadAllAttributes(worksheetWindowModel.getGraphList(), element, "graphList", SubwindowType.GRAPHLIST, worksheetWindowModel);
            InputStream input = sloader.openSection(worksheetWindowModel.getGraphList().getURI().getSchemeSpecificPart());
            GraphListLoader.load(worksheetWindowModel.getGraphList(), input, sloader);
            worksheetWindowModel.setGraphListComponentSelected(GraphListComponentType.valueOf(element.getChild("graphList").getAttributeValue("component")));
            
            // load invariant list
            loadAllAttributes(worksheetWindowModel.getInvariantList(), element, "invariantList", SubwindowType.INVARIANTLIST, worksheetWindowModel);
            InputStream invariantlist = sloader.openSection(worksheetWindowModel.getInvariantList().getURI().getSchemeSpecificPart());
            InvariantListLoader.load(worksheetWindowModel.getInvariantList(), invariantlist);
            
            // load list of generator instances
            loadAllAttributes(worksheetWindowModel.getGeneratorInstanceList(), element, "graphGeneratorInstanceList", SubwindowType.GENERATORINSTANCELIST, worksheetWindowModel);
            InputStream graphgeneratorinstancelist = sloader.openSection(worksheetWindowModel.getGeneratorInstanceList().getURI().getSchemeSpecificPart());
            GraphGeneratorInstanceListLoader.load(worksheetWindowModel.getGeneratorInstanceList(), graphgeneratorinstancelist);
            
            // load filter
            loadAllAttributes(worksheetWindowModel.getFilterList(), element, "filterList", SubwindowType.FILTERLIST, worksheetWindowModel);
            InputStream filterList = sloader.openSection(worksheetWindowModel.getFilterList().getURI().getSchemeSpecificPart());
            FilterListLoader.load(worksheetWindowModel.getFilterList(), filterList);
            
            // load list of conjectures
            loadAllAttributes(worksheetWindowModel.getConjectureList(), element, "conjectureList", SubwindowType.CONJECTURELIST, worksheetWindowModel);
            InputStream conjectureList = sloader.openSection(worksheetWindowModel.getConjectureList().getURI().getSchemeSpecificPart());
            ConjectureListLoader.load(worksheetWindowModel.getConjectureList(), conjectureList);
            
            // load engine settings
            // TODO: load engine based on id
            loadAllAttributes(worksheetWindowModel.getEngineRunner().getEngine(), element, "enginePanel", SubwindowType.ENGINEPANEL, worksheetWindowModel);
            InputStream engine = sloader.openSection(worksheetWindowModel.getEngineRunner().getEngine().getURI().getSchemeSpecificPart());
            EngineLoader.load(worksheetWindowModel.getEngineRunner().getEngine(), engine, worksheetWindowModel);

            // load history
            loadAllAttributes(worksheetWindowModel.getConjectureHistoryList(), element, "historyList", null, worksheetWindowModel);
            InputStream historyListInput = sloader.openSection(worksheetWindowModel.getConjectureHistoryList().getURI().getSchemeSpecificPart());
            ConjectureHistoryListLoader.load(worksheetWindowModel.getConjectureHistoryList(), historyListInput, sloader);
            
            worksheetWindowModel.setName(element.getAttributeValue("worksheetName"));
            
            // create and initialize the window
            WorksheetWindow window = new WorksheetWindow(worksheetWindowModel);
            
            // initialize properties of this window
            loadWindowProperties(element, window);
            
            return window;
            
        } catch (URISyntaxException ex) {
            throw new IOFormatException("Invalid URI in " + name, ex);
        }
    }

    private Element loadAllAttributes(HasURI hasURI, Element element, String elementName, SubwindowType type, WorksheetWindowModel worksheetWindowModel) throws URISyntaxException {
        Element subwindow = element.getChild(elementName);
        if (type != null)
            loadAttributes(subwindow, type, worksheetWindowModel);
        if (hasURI != null)
            loadURI(hasURI, subwindow);
        return subwindow;
    }

    private void loadAttributes(Element subwindow, SubwindowType type, WorksheetWindowModel worksheetWindowModel) {
        worksheetWindowModel.getSubwindowModel(type).setVisible(Boolean.parseBoolean(subwindow.getAttributeValue("visible")));
        worksheetWindowModel.getSubwindowModel(type).setCollapsed(Boolean.parseBoolean(subwindow.getAttributeValue("collapsed")));
    }
    
    private void loadURI(HasURI hasURI, Element subwindow) throws URISyntaxException {
        hasURI.setURI(new URI(subwindow.getAttributeValue("uri")));
    }
    
    /**
     * Save the window, its graph list, list of invariants, configuration info
     * for the engine and list of expressions.
     */
    @Override
    public void save(WorksheetWindow window, Element parent,
            SectionSaver ssaver) throws IOException {
        
        WorksheetWindowModel worksheetWindowModel = window.getModel();
        
        String name = LoaderSaverHelper.prepareSessionURI(worksheetWindowModel, "worksheets", "gwsheet");
        save(window, ssaver, name);
        
        Element worksheet = new Element("worksheetWindow");
        worksheet.setAttribute("uri", worksheetWindowModel.getURI().toString());
        parent.addContent(worksheet);
    }
    
    private void save(WorksheetWindow window,
            SectionSaver ssaver, String name) throws IOException {
        
        WorksheetWindowModel worksheetWindowModel = window.getModel();
        
        // save properties of window
        Element element = baseElement();
        saveWindowProperties(element, window);
        element.setAttribute("worksheetName", worksheetWindowModel.getName());
        
        // save graph list (including contents)
        GraphList list = worksheetWindowModel.getGraphList();
        GraphListSaver.saveIntoWorkspace(list, ssaver);
        Element subwindow = saveAllAttributes(element, "graphList",SubwindowType.GRAPHLIST, list, worksheetWindowModel);
        subwindow.setAttribute("component", worksheetWindowModel.getGraphListComponentSelected().toString());
        
        // save invariant list
        InvariantList ilist = worksheetWindowModel.getInvariantList();
        InvariantListSaver.saveIntoWorkspace(ilist, ssaver);
        saveAllAttributes(element, "invariantList", SubwindowType.INVARIANTLIST, ilist, worksheetWindowModel);
        
        // save engine configuration
        Engine engine = worksheetWindowModel.getEngineRunner().getEngine();
        EngineSaver.saveIntoWorkspace(engine, ssaver);
        saveAllAttributes(element, "enginePanel", SubwindowType.ENGINEPANEL, engine, worksheetWindowModel);
        
        // save list of generator instances
        GraphGeneratorInstanceList generators = worksheetWindowModel.getGeneratorInstanceList();
        GraphGeneratorInstanceListSaver.saveIntoWorkspace(generators, ssaver);
        saveAllAttributes(element, "graphGeneratorInstanceList",SubwindowType.GENERATORINSTANCELIST, generators, worksheetWindowModel);
        
        // save filters
        FilterList filterList = worksheetWindowModel.getFilterList();
        FilterListSaver.saveIntoWorkspace(filterList, ssaver);
        saveAllAttributes(element, "filterList",SubwindowType.FILTERLIST, filterList, worksheetWindowModel);
        
        // save list of conjectures
        ConjectureList conjectureList = worksheetWindowModel.getConjectureList();
        ConjectureListSaver.saveIntoWorkspace(conjectureList, ssaver);
        saveAllAttributes(element, "conjectureList",SubwindowType.CONJECTURELIST, conjectureList, worksheetWindowModel);
        
        // save history
        ConjectureHistoryList historyList = worksheetWindowModel.getConjectureHistoryList();
        ConjectureHistoryListSaver.saveIntoWorkspace(historyList, ssaver);
        saveAllAttributes(element, "historyList", null, historyList, worksheetWindowModel);
        
        LoaderSaverHelper.outputXML(element,
                new DocType("worksheetWindow", PUBLIC_ID, "http://downloads.grinvin.org/dtds/io/worksheet-1.0.dtd"),
                ssaver.createSection(name));
        
    }

    private Element saveAllAttributes(Element element, String elementName, SubwindowType type, HasURI hasURI, WorksheetWindowModel worksheetWindowModel) {
        Element subwindow = new Element(elementName);
        if (hasURI != null)
            saveURI(hasURI, subwindow);
        if (type != null)
            saveAttributes(subwindow, type, worksheetWindowModel);
        element.addContent(subwindow);
        return subwindow;
    }

    private void saveAttributes(Element subwindow, SubwindowType type, WorksheetWindowModel worksheetWindowModel) {
        subwindow.setAttribute("visible", Boolean.toString(worksheetWindowModel.getSubwindowModel(type).isVisible()));
        subwindow.setAttribute("collapsed", Boolean.toString(worksheetWindowModel.getSubwindowModel(type).isCollapsed()));
    }
    
    private void saveURI(HasURI hasURI, Element subwindow) {
        subwindow.setAttribute("uri", hasURI.getURI().toString());
    }
    
}
