/* GraphListElement.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.list.graphs;

import java.net.URI;
import java.util.LinkedList;
import java.util.List;
import java.util.ResourceBundle;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.print.attribute.UnmodifiableSetException;
import javax.swing.SwingUtilities;

import org.grinvin.graphs.GraphBundle;
import org.grinvin.graphs.GraphBundleListener;
import org.grinvin.graphs.GraphURI;
import org.grinvin.invariants.AbstractPrioritizedRunnable;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.PrioritizedRunnableExecutor;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.invariants.values.UnavailableValue;
import org.grinvin.list.HasName;
import org.grinvin.list.HasURI;

/**
 * Represents the elements of a GraphList.<p>
 * A graph list element combines a graph bundle and a uniform resource 
 * identifier (URI).<p>
 * Graph list elements cannot be constructed directly but must be created
 * by the graph list element manager to ensure that no two graph list elements
 * exist with the same URI.
 * @see GraphListElementManager
 */
public class GraphListElement implements GraphBundleListener, HasName, HasURI, GraphNode {
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.list.resources";
    
    //
    private URI uri;
    
    //
    private GraphBundle bundle;
    
    //
    private List<GraphListElementListener> listeners;
    
    //
    private List<Invariant> computingInvariants;
    
    //
    GraphListElement(URI uri, GraphBundle bundle) {
        this.uri = uri;
        this.bundle = bundle;
        this.listeners = new CopyOnWriteArrayList<GraphListElementListener>();
        this.computingInvariants = new LinkedList<Invariant>();
        if(bundle != null)
            bundle.addGraphBundleListener(this);
    }
    
    //
    GraphListElement(GraphBundle bundle) {
        this(null, bundle);
    }
    
    public String toString() {
        return getName();
    }
    
    /**
     * Get the uri.
     * @return the uri, or null
     */
    public URI getURI() {
        return uri;
    }
    
    /**
     * Set the URI of this GraphListElement. Use this to set the URI when the
     * bundle has been saved.
     * @param uri the URI
     */
    public void setURI(URI uri) {
        this.uri = uri;
        fireURIChanged();
    }
    
    /**
     * Returns the name of this graph, i.e., the internationalized
     * property with key <t>graph.name</t>.
     */
    public String getName() {
        if (bundle != null)
            return bundle.getName();
        else
            return ResourceBundle.getBundle(BUNDLE_NAME).getString("GraphListElement.unavailable");
    }
    
    /**
     * Change the name of this graph.
     */
    public void setName(String name) {
        if(isNameEditable()) {
            bundle.getProperties().setProperty("graph.name", name);
            fireNameChanged();
        } else {
            throw new UnmodifiableSetException("Graph cannot be renamed");
        }
    }
    
    /**
     * Is the name of this graph user editable?
     */
    public boolean isNameEditable() {
        return gotGraph() && GraphURI.isSession(uri);
    }
    
    /**
     * Return the current GraphBundle, or null if the bundle does not exist.
     * @return The current GraphBundle
     */
    public GraphBundle getBundle() {
        return bundle;
    }
    
    /**
     * Check if the bundle contains a graph.
     * @return true if the graph exists, false if not
     */
    public boolean gotGraph() {
        return (bundle != null) && (bundle.getGraph() != null);
    }
    
    /**
     * Get the requested property. Waits until the bundle is loaded and the property can be returned
     * @param property the requested property
     */
    public Object getProperty(String property) {
        if (bundle != null)
            return bundle.getProperties().getProperty(property);
        else
            return null;
    }
    
    /**
     * Get the requested invariant.
     * @param invariant the requested invariant
     */
    public InvariantValue getInvariant(Invariant invariant) {
        if (gotGraph()) {
            if (computingInvariants.contains(invariant)) {
                while (bundle.getCachedInvariantValue(invariant) == null) {
                    try {
                        Thread.sleep(200);
                    } catch (InterruptedException ex) {
                        //ignore
                    }
                }
            }
            InvariantValue result;
            computingInvariants.add(invariant);
            result = bundle.getInvariantValue(invariant);
            computingInvariants.remove(invariant);
            return result;
        } else {
            return null;
        }
    }
    
    /**
     * Get the requested invariant. Loads the bundle and computes the invariant in the background if not available
     * @param invariant the requested invariant
     */
    public InvariantValue getInvariantLater(Invariant invariant) {
        InvariantValue result = null;
        if (gotGraph()) {
            InvariantValue cachedValue = bundle.getCachedInvariantValue(invariant);
            if (cachedValue == null) {
                PrioritizedRunnableExecutor.getInstance().execute(new InvariantComputingRunnable(invariant, true));
            } else {
                result = cachedValue;
            }
        }
        return result;
    }
    
    /**
     * Get the requested invariant. Loads the bundle and computes the invariant asap if not available
     * @param invariant the requested invariant
     */
    public InvariantValue getInvariantNow(Invariant invariant) {
        // TODO: almost the same as above
        InvariantValue result = null;
        if (bundle == null) {
            result = UnavailableValue.getInstance();
        } else if (gotGraph()) {
            InvariantValue cachedValue = bundle.getCachedInvariantValue(invariant);
            if (cachedValue == null)  {
                PrioritizedRunnableExecutor.getInstance().execute(new InvariantComputingRunnable(invariant, false));
            } else {
                result = cachedValue;
            } 
        }
        return result;
    }
    
    //
    public void addGraphListElementListener(GraphListElementListener listener) {
        listeners.add(listener);
    }
    
    //
    public void removeGraphListElementListener(GraphListElementListener listener) {
        listeners.remove(listener);
    }
    
    public void fireURIChanged() {
        for (GraphListElementListener listener : listeners)
            listener.graphListElementURIChanged(this);
    }
    
    public void fireNameChanged() {
        for (GraphListElementListener listener : listeners)
            listener.graphListElementNameChanged(this);
    }
    
    public void fireInvariantComputed(Invariant invariant) {
        for (GraphListElementListener listener : listeners)
            listener.graphListElementInvariantComputed(this, invariant);
    }
    
    public void fireGraphChanged() {
        for (GraphListElementListener listener : listeners)
            listener.graphListElementGraphChanged(this);
    }
    
    //implements GraphBundleListener
    @SuppressWarnings("PMD.CompareWithEquals")
    public void graphBundleChanged(GraphBundle graphbundle) {
        assert graphbundle == bundle : "Unexpected graph bundle change";
        fireGraphChanged();
    }

    public String getDescription() {
        if (bundle != null)
            return bundle.getDescription();
        else
            return ResourceBundle.getBundle(BUNDLE_NAME).getString("GraphListElement.unavailable") + ": " + uri;
    }

    //
    public List<GraphNode> getChildren() {
        return null;
    }

    //
    private class InvariantComputingRunnable extends AbstractPrioritizedRunnable {
        
        //
        private final Invariant invariant;
        
        //
        private final int counter;
        
        //
        public InvariantComputingRunnable(Invariant invariant, boolean later) {
            this.invariant = invariant;
            if (later)
                this.counter = 0;
            else
                this.counter = PrioritizedRunnableExecutor.getInstance().getCount();
        }
        
        public void run() {
            InvariantValue result;
            if (computingInvariants.contains(invariant)) {
                // do not compute the same invariant twice in parallel
                return;
            }
            computingInvariants.add(invariant);
            result = bundle.getInvariantValue(invariant);
            computingInvariants.remove(invariant);
            if (result != null)
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        fireInvariantComputed(invariant);
                    }
                });
        }
        
        public int getPriority() {
            return counter;
        }
        
    }
    
}
