/* GraphEditor.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */
package org.grinvin.gred;

import be.ugent.caagt.swirl.SwirlUtilities;
import be.ugent.caagt.swirl.actions.HideWindow;
import be.ugent.caagt.swirl.menus.MenuBuilder;
import be.ugent.caagt.swirl.mouse.MouseTool;
import be.ugent.caagt.swirl.undoredo.RedoAction;
import be.ugent.caagt.swirl.undoredo.UndoAction;
import be.ugent.caagt.swirl.undoredo.UndoManager;

import java.awt.BorderLayout;
import java.awt.Color;
import java.net.URI;
import java.util.ResourceBundle;
import javax.swing.AbstractButton;
import javax.swing.ActionMap;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;

import org.grinvin.graphs.Annotation;
import org.grinvin.graphs.AnnotationView;
import org.grinvin.graphs.DefaultAnnotationModel;
import org.grinvin.graphs.DefaultEmbeddingModel;
import org.grinvin.graphs.DefaultGraphModel;
import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.EmbeddingView;
import org.grinvin.graphs.Graph;
import org.grinvin.graphs.GraphModel;
import org.grinvin.graphs.MutableAnnotationModel;
import org.grinvin.graphs.MutableEmbeddingModel;
import org.grinvin.gui.components.GraphCellListComponent;
import org.grinvin.gred.guides.GridLines;
import org.grinvin.gred.guides.GuidesButton;
import org.grinvin.gred.mouse.AddEdges;
import org.grinvin.gred.mouse.AddVertices;
import org.grinvin.gred.mouse.MoveSelection;
import org.grinvin.gred.mouse.RollOverOverlay;
import org.grinvin.gred.mouse.SplitEdges;
import org.grinvin.gred.mouse.SelectElements;
import org.grinvin.gred.mouse.SelectRectangle;
import org.grinvin.help.HelpManager;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListModel;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;
import org.grinvin.graphs.render.AnnotatedRenderer;
import org.grinvin.graphs.render.Renderer;
import org.grinvin.gred.actions.AutoSnapListener;
import org.grinvin.gred.actions.ClearSelection;
import org.grinvin.gred.actions.ShowAdvancedToolbarListener;
import org.grinvin.gred.actions.Clear;
import org.grinvin.gred.actions.Closure;
import org.grinvin.gred.actions.Complement;
import org.grinvin.gred.actions.ConnectSelection;
import org.grinvin.gred.actions.DeleteSelectedGraphs;
import org.grinvin.gred.actions.DeleteSelection;
import org.grinvin.gred.actions.EdgeInducedGraph;
import org.grinvin.gred.actions.LineGraph;
import org.grinvin.gred.actions.MergeSelection;
import org.grinvin.gred.actions.SelectAll;
import org.grinvin.gred.actions.SnapAll;
import org.grinvin.gred.actions.SnapSelection;
import org.grinvin.gred.actions.SnapshotAction;
import org.grinvin.gred.actions.SplitSelection;
import org.grinvin.gred.actions.SpringEmbedder;
import org.grinvin.gred.actions.VertexInducedGraph;
import org.grinvin.help.actions.DisplayHelpAfterTracking;
import org.grinvin.help.actions.LaunchHelp;
import org.grinvin.list.HasURI;

/**
 * Graph editor window.
 */
public class GraphEditor extends JFrame implements HasURI {

    //
    private static final String BUNDLE_NAME = "org.grinvin.gred.editor";
    //
    private MutableGraphPanel panel;
    //
    private GraphCellListComponent cellList;
    //
    JToolBar advancedToolbar;
    // Toggle button 'show advanced toolbar'
    AbstractButton satButton;

    // Returns the panel for this editor, mainly for testing purposes
    MutableGraphPanel getPanel() {
        return panel;
    }

    /**
     * Default constructor.
     */
    public GraphEditor(GraphListModel list) {
        this(new DefaultGraphModel(), list);
    }

    private GraphEditor(GraphModel graph, GraphListModel list) {
        this(new DefaultEmbeddingModel(graph, 2), new DefaultAnnotationModel(graph), new GraphContext(), list);
    }

    /**
     * Create an editor that uses the given embedding model and context.
     * This constructor is mainly useful when models and contexts should be
     * shared between editors. In most cases users should use the
     * no argument constructor.
     */
    public GraphEditor(MutableEmbeddingModel model, MutableAnnotationModel annotation, GraphContext context, GraphListModel list) {
        super(ResourceBundle.getBundle(BUNDLE_NAME).getString("editor.title"));

        setDefaultCloseOperation(HIDE_ON_CLOSE);

        Renderer renderer = new AnnotatedRenderer();

        this.panel = new MutableGraphPanel(model, annotation, renderer, context, 140);
        panel.setGuides(new GridLines(5, 5));

        RollOverOverlay roo = new RollOverOverlay(panel);

        // the order of these handlers is important
        MouseTool creationTool = new MouseTool();
        creationTool.addHandler(AddEdges.getHandler());
        creationTool.addHandler(SplitEdges.getHandler());
        creationTool.addHandler(AddVertices.getHandler());

        MouseTool selectionTool = new MouseTool();
        selectionTool.addHandler(MoveSelection.getHandler());
        selectionTool.addHandler(SelectElements.getHandler());
        selectionTool.addHandler(SelectRectangle.getHandler());
        roo.setTool(creationTool);

        int graphIconSize = GrinvinPreferences.getInstance().getIntPreference(Preference.GRAPHICON_SIZE);
        this.cellList = new GraphCellListComponent(list, graphIconSize, graphIconSize, 1);

        // background color of the cell list
        Color bg = panel.getBackground();
        int red = Math.min(bg.getRed() + 15, 255);
        int green = Math.min(bg.getGreen() + 15, 255);
        int blue = Math.min(bg.getBlue() + 15, 255);
        cellList.setBackground(new Color(red, green, blue));
        cellList.setPaintWatermark(false);

        MenuBuilder menuBuilder = new MenuBuilder();
        menuBuilder.load("/org/grinvin/gred/editor-menus.xml",
                "org.grinvin.gred.editor-menus");
        ActionMap actionMap = menuBuilder.getActionMap();
        actionMap.setParent(panel.getActionMap());
        panel.setActionMap(actionMap);

        // initialize actions
        UndoManager undoManager = panel.getUndoManager();
        
        actionMap.put("undo", new UndoAction (undoManager));

        actionMap.put("redo", new RedoAction (undoManager));

        actionMap.put("ClearSelection", new ClearSelection(panel));

        actionMap.put("SelectAll", new SelectAll(panel));

        actionMap.put("DeleteSelection", new DeleteSelection(panel));

        actionMap.put("MergeSelection", new MergeSelection(panel));

        actionMap.put("SplitSelection", new SplitSelection(panel));

        actionMap.put("ConnectSelection", new ConnectSelection(panel));

        actionMap.put("VertexInducedGraph", new VertexInducedGraph(panel));

        actionMap.put("EdgeInducedGraph", new EdgeInducedGraph(panel));

        actionMap.put("Clear", new Clear(panel));

        actionMap.put("SnapSelection", new SnapSelection(panel));

        actionMap.put("SnapAll", new SnapAll(panel));

        actionMap.put("Snapshot", new SnapshotAction(panel, list));

        actionMap.put("Complement", new Complement(panel));

        actionMap.put("Closure", new Closure(panel));

        actionMap.put("LineGraph", new LineGraph(panel));

        actionMap.put("DeleteSelectedGraphs", new DeleteSelectedGraphs(
                list, cellList.getSelectionModel()));

        actionMap.put("file.close", new HideWindow(this));

        // TODO: these actions also exist in GrinvinWindow

        actionMap.put("LaunchHelp", new LaunchHelp());

        actionMap.put("DisplayHelpAfterTracking", new DisplayHelpAfterTracking());

        SpringEmbedder springEmbedder = new SpringEmbedder(panel);
        actionMap.put("SpringEmbedder", springEmbedder);

        // Register mouse tools group
        menuBuilder.registerGroup("tools",
                roo.getMouseToolModel(),
                "tool.selection", selectionTool,
                "tool.creation", creationTool);

        // Register toggles
        menuBuilder.registerToggle("AutoSnap", new AutoSnapListener(panel));
        menuBuilder.registerToggle("ShowAdvancedToolbar", new ShowAdvancedToolbarListener(this));


        add(panel);

        JMenuBar menuBar = menuBuilder.createJMenuBar("main.menu");
        setJMenuBar(menuBar);
        this.satButton = (AbstractButton) SwirlUtilities.getDescendantWithName(menuBar, "ShowAdvancedToolbar");
        JToolBar toolbar = menuBuilder.createJToolBar("toolbar");
        springEmbedder.attachRepeaterTo((JButton) SwirlUtilities.getChildWithName(toolbar, "SpringEmbedder"));

        // TODO: use SWIRL for this
        toolbar.add(new GuidesButton(panel), 9);
        add(toolbar, BorderLayout.NORTH);

        advancedToolbar = menuBuilder.createJToolBar("toolbar.advanced");
        advancedToolbar.setOrientation(JToolBar.VERTICAL);
        advancedToolbar.setVisible(false); //false by default
        add(advancedToolbar, BorderLayout.WEST);

        add(new JScrollPane(cellList), BorderLayout.EAST);

        pack();

        // help

        // TODO: incorporate hekp id's into Swirl MenuBuilder
        HelpManager.setHelpIDString(this, "org.grinvin.window.editor");
        HelpManager.setHelpIDString(toolbar, "org.grinvin.window.editor.toolbar");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "Snapshot"), "org.grinvin.window.editor.snapshot");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "tool.selection"), "org.grinvin.window.editor.selection");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "tool.creation"), "org.grinvin.window.editor.creation");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "undo"), "org.grinvin.window.editor.undo");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "redo"), "org.grinvin.window.editor.redo");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "AutoSnap"), "org.grinvin.window.editor.autosnap");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "SnapSelection"), "org.grinvin.window.editor.snapselection");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "SnapAll"), "org.grinvin.window.editor.snapall");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "Clear"), "org.grinvin.window.editor.clear");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(toolbar, "SpringEmbedder"), "org.grinvin.window.editor.springembedder");
        HelpManager.setHelpIDString(toolbar.getComponentAtIndex(9), "org.grinvin.window.editor.guides");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "file.menu"), "org.grinvin.window.editor.menu.file");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "edit.menu"), "org.grinvin.window.editor.menu.edit");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "selection.menu"), "org.grinvin.window.editor.menu.selection");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "guides.menu"), "org.grinvin.window.editor.menu.guides");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "advanced.menu"), "org.grinvin.window.editor.menu.advanced");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(menuBar, "menu.help"), "org.grinvin.window.editor.menu.help");
        HelpManager.setHelpIDString(advancedToolbar, "org.grinvin.window.editor.advancedtoolbar");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "MergeSelection"), "org.grinvin.window.editor.mergeselection");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "SplitSelection"), "org.grinvin.window.editor.splitselection");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "ConnectSelection"), "org.grinvin.window.editor.connectselection");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "VertexInducedGraph"), "org.grinvin.window.editor.vertexinducedgraph");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "EdgeInducedGraph"), "org.grinvin.window.editor.edgeinducedgraph");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "Complement"), "org.grinvin.window.editor.complement");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "Closure"), "org.grinvin.window.editor.closure");
        HelpManager.setHelpIDString(SwirlUtilities.getChildWithName(advancedToolbar, "LineGraph"), "org.grinvin.window.editor.linegraph");
        HelpManager.enableHelpKey(this.getRootPane(), "org.grinvin.window.editor");
    }

    /**
     * Load (a copy of) the given embedding into the editor.
     */
    public void setGraph(EmbeddingView newEmbedding, AnnotationView newAnnotation) {
        panel.loadGraph(newEmbedding, newAnnotation);
    }

    /**
     * Get the embedding for the current graph in the editor.
     */
    public Embedding getEmbedding() {
        return panel.getEmbedding();
    }

    /**
     * Get the annotation for the current graph in the editor.
     */
    public Annotation getAnnotation() {
        return panel.getAnnotation();
    }

    /**
     * Get the current graph in the editor.
     */
    public Graph getGraph() {
        return panel.getGraph();
    }

    public GraphList getGraphList() {
        return cellList.getGraphList();
    }

    /**
     * Set the 'dirty' flag of the editor.
     */
    public void setDirty(boolean dirty) {
        panel.setDirty(dirty);
    }

    /**
     * Check whether the editor is 'dirty'.
     */
    public boolean isDirty() {
        return panel.isDirty();
    }
    //
    private URI uri;

    public URI getURI() {
        return uri;
    }

    public void setURI(URI uri) {
        this.uri = uri;
    }

    // TODO: separate into a different class?
    public void setAdvancedToolbar(boolean show) {
        advancedToolbar.setVisible(show);
        this.satButton.setSelected(show);
    }

    public void showAdvancedToolbar(boolean show) {
        advancedToolbar.setVisible(show);
    }

    public boolean showAdvancedToolbar() {
        return advancedToolbar.isVisible();
    }
}
