/* FactoryManager.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories; // NOPMD

import be.ugent.caagt.swirl.dialogs.SimpleDialog;

import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;

import org.grinvin.factories.graphs.AbstractGraphFactory;
import org.grinvin.factories.graphs.GraphFactory;
import org.grinvin.generators.graphs.AbstractGraphGenerator;
import org.grinvin.generators.graphs.GraphGenerator;
import org.grinvin.gui.components.render.AbstractGrinvinCellListCellRenderer;
import org.grinvin.gui.components.render.GraphFactoryCellRenderer;
import org.grinvin.gui.components.render.GraphGeneratorCellRenderer;
import org.grinvin.params.ParameterList;
import org.grinvin.params.ParameterPanel;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.preferences.GrinvinPreferences.Preference;
import org.grinvin.util.SingletonBroker;

/**
 * Helper class to handle configuration of {@link Factory} objects.
 */
public final class FactoryManager {
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.factories.resources";
    
    //
    private static final SingletonBroker<GraphFactory> GRAPH_FACTORY_BROKER
            = new SingletonBroker<GraphFactory>("org/grinvin/factories/graphs/resources/graphfactories.xml");
    
    
    //
    private static final SingletonBroker<GraphGenerator> GENERATOR_BROKER
            = new SingletonBroker<GraphGenerator>("org/grinvin/generators/graphs/resources/generators.xml");
    
    //
    private static final Map<Class<? extends Factory>, AbstractGrinvinCellListCellRenderer> renderers
            = new HashMap<Class<? extends Factory>, AbstractGrinvinCellListCellRenderer>();
    
    static {
        renderers.put(AbstractGraphFactory.class, GraphFactoryCellRenderer.getInstance());
        renderers.put(AbstractGraphGenerator.class, GraphGeneratorCellRenderer.getInstance());
    }

    /**
     * Return the shared graph factory broker.
     */
    public static SingletonBroker<GraphFactory> getGraphFactoryBroker() {
        return GRAPH_FACTORY_BROKER;
    }
    
    /**
     * Return the shared generator broker.
     */
    public static SingletonBroker<GraphGenerator> getGeneratorBroker() {
        return GENERATOR_BROKER;
    }
    
    //
    private static AbstractGrinvinCellListCellRenderer getRenderer(Class<?> clazz) {
        if (clazz == null) {
            return null;
        } else {
            AbstractGrinvinCellListCellRenderer renderer = renderers.get(clazz);
            if (renderer != null) {
                return renderer;
            } else {
                return getRenderer(clazz.getSuperclass());
            }
        }
    }
    
    //
    public static boolean configureFactory(Factory factory) throws FactoryParameterException {
        // open a factory parameter window if at least one parameter
        // is present
        ParameterList list = factory.getParameters();
        int paramCount = list.getCount();
        if (paramCount > 0) {
            ParameterPanel panel = new ParameterPanel(list);
            SimpleDialog dialog = panel.createDialog(null,
                    ResourceBundle.getBundle(BUNDLE_NAME).getString("Parameters.title"),
                    true);
            AbstractGrinvinCellListCellRenderer renderer = getRenderer(factory.getClass());
            if (renderer != null) {
                int size = GrinvinPreferences.getInstance().getIntPreference(Preference.GRAPHICON_SIZE) - 10;
                dialog.setIcon(renderer.createIcon(factory, size));
            }
            if (dialog.launchAtPointerPosition() == SimpleDialog.APPROVE_OPTION) {
                Object[] values = new Object[paramCount];
                panel.getValues(values);
                factory.setParameterValues(values);
            } else {
                return false;
            }
        }
        return true;
    }
    
}
