/* Linux Video Studio - a program to capture video using MJPEG-codec boards
 * Copyright (C) 2000-2001 Ronald Bultje
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtk.h>
#include <glib.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <signal.h>
#include "pipes.h"

extern int verbose;

int pipe_out[NUM], pipe_in[NUM];
int pid[NUM];
int reader[NUM];
int active[NUM];

/* external callbacks, I'll make void-variables later on */
void continue_encoding(void);
void lavencode_callback(int number, char *input);
void lavplay_trimming_callback(char *msg);
void scene_detection_input_cb(char *input);
void scene_detection_finished(void);
void process_lavplay_edit_input(char *msg);
void quit_trimming(GtkWidget *widget, gpointer data);
void quit_lavplay_edit(void);
void continue_encoding(void);
void scene_detection_finished(void);
void dispatch_input(char *buff);
void lavrec_quit(void);
void process_lavplay_input(char *msg);
void lavplay_stopped(void);
void lavplay_edit_stopped(void);

/* some static private functions */
static void callback_pipes(gpointer data, gint source,GdkInputCondition condition);

int pipe_is_active(int number) {
   return active[number];
}

void write_pipe(int number, char *message) {
   if (active[number]) {
      write(pipe_out[number], message, strlen(message));
   }
   if (verbose) printf("  TO %10.10s: %s", app_name(number), message);
}

void close_pipe(int number) {
   if (active[number]) {
      kill(pid[number],SIGINT);
      waitpid (pid[number], NULL, 0);
      active[number] = 0;
   }
}

char *app_name(int number) {
   char *app;

   switch (number) {
      case LAV2YUV:
      case LAV2YUV_S:
        app = "lav2yuv";
        break;
      case LAVPLAY:
      case LAVPLAY_E:
      case LAVPLAY_T:
        app = "lavplay";
        break;
      case LAVREC:
        app = "lavrec";
        break;
      case MPLEX:
        app = "mplex";
        break;
      case MPEG2ENC:
        app = "mpeg2enc";
        break;
      case MP2ENC:
        app = "mp2enc";
        break;
      case LAV2WAV:
        app = "lav2wav";
        break;
      case YUVSCALER:
        app = "yuvscaler";
        break;
      default:
        app = "unknown";
        break;
   }

   return app;
}

void callback_pipes(gpointer data, gint source,
   GdkInputCondition condition)
{
   char input[4096];
   int n, i, number;
   char *app;

   number = (int)data;

   app = app_name(number);

   n = read(source, input, 4095);
   if (n==0)
   {
      extern int use_yuvscaler_pipe;

      /* program finished */
      if (verbose) printf("%s finished\n", app);

      /* close pipes/app */
      close(pipe_in[number]);
      if (number != MP2ENC && number != MPEG2ENC && number != YUVSCALER) {
         close(pipe_out[number]);
      }
      close_pipe(number);

      if (number == LAV2WAV) {
         close(pipe_out[MP2ENC]);
      }
      if (number == LAV2YUV) {
         if (use_yuvscaler_pipe)
            close(pipe_out[YUVSCALER]);
         else
            close(pipe_out[MPEG2ENC]);
      }
      if (number == YUVSCALER) {
         close(pipe_out[MPEG2ENC]);
      }

      /* trigger callback function for each specific app */
      if (number == MPEG2ENC || number == MP2ENC || number == MPLEX) {
         continue_encoding();
      }
      else if (number == LAV2YUV_S) {
         scene_detection_finished();
      }
      else if (number == LAVPLAY_E) {
         lavplay_edit_stopped();
      }
      else if (number == LAVPLAY_T) {
         quit_trimming(NULL,NULL);
      }
      else if (number == LAVREC) {
         lavrec_quit();
      }
	else if (number == LAVPLAY) {
         lavplay_stopped();
      }

      /* officially, we should detach the gdk_input here */
      gdk_input_remove(reader[number]);
   }
   else
   {
      int x = 0;
      char temp[256], endsign;

      for(i=0;i<n;i++)
      {
         if(input[i]=='\n' || input[i]=='\r' || i==n-1)
         {
            strncpy(temp, input+x, i-x);
            if (i-x<255) {
               if (i==n-1 && input[i]!='\r' && input[i]!='\n') {
                  temp[i-x] = input[i];
                  if (i-x<254)
                     temp[i-x+1] = '\0';
               }
               else
                  temp[i-x] = '\0';
            }
            endsign = '\n';
            if (input[i] == '\n' || input[i] == '\r')
               endsign = input[i];
            if (input[x] == '@')
               endsign = '\r';
            if (number == LAV2YUV_S && strncmp(temp, "--DEBUG: frame", 14)==0)
               endsign = '\r';
            if (number == MPEG2ENC && strncmp(temp, "   INFO: Frame", 14)==0)
               endsign = '\r';

            if(!(number == LAVPLAY && strncmp(temp, "--DEBUG: frame=", 15)==0)) {
               if (verbose) {
                  fprintf(stdout, "FROM %10.10s: %s%c", app, temp, endsign);
                  fflush(stdout);
               }
            }

            switch(number) {
               case MPEG2ENC:
               case LAV2YUV:
               case LAV2WAV:
               case MP2ENC:
               case MPLEX:
               case YUVSCALER:
                  lavencode_callback(number, temp);
                  break;
               case LAV2YUV_S:
                  scene_detection_input_cb(temp);
                  break;
               case LAVPLAY_E:
                  process_lavplay_edit_input(temp);
                  break;
               case LAVPLAY:
                  process_lavplay_input(temp);
                  break;
               case LAVPLAY_T:
                  lavplay_trimming_callback(temp);
                  break;
               case LAVREC:
                  dispatch_input(temp);
                  break;
            }

            x = i+1;
         }
      }
   }
}

void start_pipe_command(char *command[], int number)
{
   if (!active[number])
   {
      int ipipe[2], opipe[2];
      int n;

      if(pipe(ipipe) || pipe(opipe))
      {
         perror("pipe() failed");
         exit(1);
      }

      pid[number] = fork();
      if(pid[number]<0)
      {
         perror("fork() failed");
         exit(1);
      }

      active[number] = 1;

      if (pid[number]) /* This is the parent process (i.e. LVS) */
      {
         /* parent */
         pipe_in[number] = opipe[0];
         close(opipe[1]);
         fcntl (pipe_in[number], F_SETFL, O_NONBLOCK);
         reader[number] = gdk_input_add (pipe_in[number],
            GDK_INPUT_READ, callback_pipes, (gpointer)number);

         pipe_out[number] = ipipe[1]; /* don't O_NONBLOCK it! */
         close(ipipe[0]);
      }
      else /* This is the child process (i.e. lav2wav/mp2enc) */
      {
         extern int use_yuvscaler_pipe;

         /* child */
         close(ipipe[1]);
         close(opipe[0]);

         n = dup2(ipipe[0],0);
         if(n!=0) exit(1);
         close(ipipe[0]);

         if (number == LAV2WAV) {
            n = dup2(pipe_out[MP2ENC],1); /* writes lav2wav directly to mp2enc */
         }
         else if (number == LAV2YUV) {
            if (use_yuvscaler_pipe)
               n = dup2(pipe_out[YUVSCALER],1); /* writes lav2wav directly to yuvscaler */
            else
               n = dup2(pipe_out[MPEG2ENC],1); /* writes lav2wav directly to mpeg2enc */
         }
         else if (number == YUVSCALER) {
            n = dup2(pipe_out[MPEG2ENC],1); /* writes yuvscaler directly to mpeg2enc */
         }
         else {
            n = dup2(opipe[1],1);
         }
         if(n!=1) exit(1);

         n = dup2(opipe[1],2);
         if(n!=2) exit(1);
         close(opipe[1]);

         execvp(command[0], command);
         exit(1);
      }
   }
   else
   {
      printf("**ERROR: %s is already active\n", app_name(number));
   }
}
