/* GSnes9x -- romprefs.c: ROM Preferences dialog related functions.
 * Copyright (C) 1999 Canek Pelez Valds <canek@abulafia.fciencias.unam.mx>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsnes9x.h"

void
rom_preferences_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	ROMPrefsData *rpd = rom_prefs_data_new ();
	GtkWidget    *prefs_dialog;
	GtkWidget    *notebook;
	GtkWidget    *cont;
	GList        *list;
	gchar        *title;
	gchar        *rom_title;

	F_NAME();
	
	if (gsnes_data->current_rom == -1) {
		g_free (rpd);
		return;
	}

	gsnes_data->dialog_open = TRUE;
	
	rpd->data = gsnes_data;
	
	gtk_clist_get_text (GTK_CLIST (gsnes_data->clist),
			    gsnes_data->current_rom, 0, &rom_title);

	list = g_list_first (gsnes_data->roms_list);

	while (strcmp (rom_title,
		       ((ROMInfo*)(list->data))->string_prefs[NAME]))
		list = g_list_next (list);

	rpd->rom_info = (ROMInfo*)(list->data);
	
	title = g_strconcat (_("Preferences for "), rom_title, NULL);
	
	prefs_dialog = gnome_dialog_new (title,
					GNOME_STOCK_BUTTON_OK,
					GNOME_STOCK_BUTTON_APPLY,
					GNOME_STOCK_BUTTON_CLOSE,
					GNOME_STOCK_BUTTON_HELP, 
					NULL);

	g_free (title);

	gtk_window_set_modal (GTK_WINDOW (prefs_dialog), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (prefs_dialog),
				      GTK_WINDOW (gsnes_data->app));
	
	gtk_object_set_data (GTK_OBJECT (prefs_dialog), "rom-prefs",
			     (gpointer) rpd);
	
	rpd->rom_prefs_dialog = prefs_dialog;

	notebook = gtk_notebook_new ();
	
	cont = rom_prefs_info_ops (rpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("ROM Info")));
	gtk_widget_show (cont);
	
	cont = rom_prefs_graphics_ops (rpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Graphics")));
	gtk_widget_show (cont);

	cont = rom_prefs_sound_ops (rpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Sound")));
	gtk_widget_show (cont);

	cont = rom_prefs_cheat_ops (rpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Cheat Codes")));
	gtk_widget_show (cont);

	cont = rom_prefs_speed_ops (rpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Speed up/down")));
	gtk_widget_show (cont);

	cont = rom_prefs_format_ops (rpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("ROM image format")));
	gtk_widget_show (cont);

	cont = rom_prefs_joystick_ops (rpd);
	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), cont,
				  gtk_label_new (_("Joystick")));
	gtk_widget_show (cont);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (prefs_dialog)->vbox),
			    notebook, FALSE, FALSE, 10);
	gtk_widget_show (notebook);
	
	rom_prefs_dialog_set_status (rpd);
	rom_prefs_connect_changes   (rpd);
	rom_prefs_set_tooltips      (rpd);

	gtk_signal_connect (GTK_OBJECT (prefs_dialog), "close",
			    GTK_SIGNAL_FUNC (rom_prefs_delete_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT (prefs_dialog), "destroy",
			    GTK_SIGNAL_FUNC (rom_prefs_free_cb),
			    (gpointer) rpd);

	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog),
				     0, GTK_SIGNAL_FUNC (rom_prefs_apply_cb),
				     (gpointer) rpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog),
				     0, GTK_SIGNAL_FUNC (rom_prefs_close_cb),
				     (gpointer) rpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog),
				     1, GTK_SIGNAL_FUNC (rom_prefs_apply_cb),
				     (gpointer) rpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog),
				     2, GTK_SIGNAL_FUNC (rom_prefs_close_cb),
				     (gpointer) rpd);
	gnome_dialog_button_connect (GNOME_DIALOG (prefs_dialog),
				     3, GTK_SIGNAL_FUNC (rom_prefs_help_cb),
				     (gpointer) notebook);
	
	gnome_dialog_set_sensitive (GNOME_DIALOG (prefs_dialog), 0, FALSE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (prefs_dialog), 1, FALSE);

	gtk_widget_show (prefs_dialog);
}

ROMInfo*
rom_info_new ()
{
	gint i, j;
	ROMInfo *rom_info = g_new (ROMInfo, 1);

	F_NAME();
	
	for (i = 0; i < ROM_STR_PREFS; i++)
		rom_info->string_prefs[i] = NULL;
	
	for (i = 0; i < MIX_BOOL_PREFS+ROM_BOOL_PREFS; i++)
		rom_info->bool_prefs[i] = 0;
		
	for (i = 0; i < MIX_NUM_PREFS; i++)
		rom_info->number_prefs[i] = 0;
		
	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			rom_info->buttons[i][j] = j;
	
	for (i = 0; i < MAX_CODES; i++) {
		rom_info->gamegenie_code[i]    = NULL;
		rom_info->actionreplay_code[i] = NULL;
		rom_info->goldfinger_code[i]   = NULL;
	}

	return rom_info;
}

ROMPrefsData*
rom_prefs_data_new ()
{
	gint i, j;
	ROMPrefsData *rpd = g_new (ROMPrefsData, 1);	

	F_NAME();
	
	rpd->changed           = FALSE;
	rpd->nodata            = FALSE;

	for (i = 0; i < MIX_BOOL_PREFS+ROM_BOOL_PREFS; i++)
		rpd->check[i] = NULL;

	rpd->mono = NULL;
	
	for (i = 0; i < MIX_NUM_PREFS; i++)
		rpd->entry[i] = NULL;
	
	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			rpd->buttons[i][j] = NULL;

	for (i = 0; i < CODES; i++)
		rpd->code_change[i] = NULL;

	for (i = 0; i < MIX_NUM_PREFS; i++)
		rpd->spin[i] = NULL;

	rpd->data     = NULL;
	rpd->rom_info = NULL;
	
	return rpd;
}

GtkWidget*
rom_prefs_info_ops (ROMPrefsData *rpd)
{
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *text;
	GtkWidget *separator;
	GtkWidget *button;


	F_NAME();
	
	table = gtk_table_new (4, 4, FALSE);

	label = gtk_label_new (_("ROM file:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);

	text = gnome_file_entry_new (NULL, _("Select ROM file"));
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	rpd->entry[ROM_FILE] =  gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(text));
	
	label = gtk_label_new (_("Name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);

	text = gtk_entry_new_with_max_length (CONF_STR_LEN);
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	rpd->entry[NAME] = text;

	label = gtk_label_new (_("Company:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);

	text = gtk_entry_new_with_max_length (CONF_STR_LEN);
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	rpd->entry[COMPANY] = text;

	label = gtk_label_new (_("Image File:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);

	text = gnome_file_entry_new (NULL, _("Select Image file"));
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 3, 4,
			  GTK_SHRINK, GTK_SHRINK, 5, 2);
	gtk_widget_show (text);
	rpd->entry[IMAGE_FILE] =  gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(text));

	separator = gtk_vseparator_new ();
	gtk_table_attach (GTK_TABLE (table), separator, 2, 3, 0, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (separator);

	button = gtk_button_new_with_label (_("Load Defaults"));
	gtk_table_attach (GTK_TABLE (table), button, 3, 4, 1, 2,
			  GTK_SHRINK, GTK_SHRINK, 5, 2);
	gtk_widget_show (button);
	rpd->load_defaults = button;
	gtk_signal_connect (GTK_OBJECT (rpd->load_defaults), "clicked",
			    GTK_SIGNAL_FUNC (rom_load_defaults_cb),
			    (gpointer)(rpd));
	
	return table;
}

GtkWidget*
rom_prefs_graphics_ops (ROMPrefsData *rpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *frame;
	GtkWidget *option;
	GtkWidget *menu;
	GtkWidget *item;
	

	F_NAME();
	
	table = gtk_table_new (4, 3, TRUE);
	
	check = gtk_check_button_new_with_label (_("Transparency effects"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[TRANSPARENCY] = check;
	
	check = gtk_check_button_new_with_label
		(_("16-bit internal screen rendering"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[SIXTEEN] = check;
	
	check = gtk_check_button_new_with_label (_("Hi-res"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[HIRES] = check;

	check = gtk_check_button_new_with_label (_("No mode switch"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOMODESWITCH] = check;

	check = gtk_check_button_new_with_label (_("Full screen"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[FULLSCREEN] = check;

	check = gtk_check_button_new_with_label (_("Scale"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[SCALE] = check;
	
	check = gtk_check_button_new_with_label (_("Display frame rate"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[DISPLAYFRAMERATE] = check;

	frame = gtk_frame_new (_("Image Processing"));

	option = gtk_option_menu_new ();
	
	menu = gtk_menu_new ();

	item = gtk_menu_item_new_with_label (_("None"));
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_widget_show (item);
	
	item = gtk_menu_item_new_with_label (_("TV Mode"));
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_widget_show (item);	
	
	item = gtk_menu_item_new_with_label (_("Super 2xSaI"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	item = gtk_menu_item_new_with_label (_("Super Eagle"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	item = gtk_menu_item_new_with_label (_("2xSaI"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	item = gtk_menu_item_new_with_label (_("Bi-linear filter"));
	gtk_menu_append (GTK_MENU(menu), item);
	gtk_widget_show (item);

	gtk_widget_show (menu);
	
	gtk_option_menu_set_menu (GTK_OPTION_MENU (option), menu);
	gtk_widget_show (option);
	rpd->spin[FILTER_NUMBER] = option;

	gtk_container_add (GTK_CONTAINER (frame), option);
	gtk_table_attach (GTK_TABLE (table), frame, 2, 3, 0, 2,
			  GTK_FILL, GTK_FILL, 5, 4);
	gtk_widget_show (frame);

	return table;
}

GtkWidget*
rom_prefs_sound_ops (ROMPrefsData *rpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *radio;
	GtkWidget *spin;
	GtkObject *adj;	

	F_NAME();
	
	table = gtk_table_new (4, 4, TRUE);

	check = gtk_check_button_new_with_label (_("No sound"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOSOUND] = check;
	
	check = gtk_check_button_new_with_label (_("Sound skip"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[SOUNDSKIP] = check;
	
	adj = gtk_adjustment_new (0, 0, 3, 1, 1, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 2, 3, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	rpd->spin[SOUNDSKIP_NUMBER] = spin;

	radio = gtk_radio_button_new_with_label (NULL, _("Stereo"));
	gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (radio);
	rpd->check[STEREO] = radio;

	radio = gtk_radio_button_new_with_label (
		gtk_radio_button_group (GTK_RADIO_BUTTON (radio)),
		_("Mono"));
	gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (radio);
	rpd->mono = radio;
	
	check = gtk_check_button_new_with_label (_("Sound quality"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[SOUNDQUALITY] = check;

	adj = gtk_adjustment_new (4, 0, 7, 1, 1, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 2, 3, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	rpd->spin[SOUNDQUALITY_NUMBER] = spin;

	check = gtk_check_button_new_with_label (_("Volume envelope"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[ENVX] = check;

	check = gtk_check_button_new_with_label (_("Sound sync"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[SOUNDSYNC] = check;

	check = gtk_check_button_new_with_label (_("Alternative decoding"));
	gtk_table_attach (GTK_TABLE (table), check, 2, 3, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[ALTSAMPLEDECODE] = check;

	check = gtk_check_button_new_with_label (_("Interpolated sound"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[INTERPOLATEDSOUND] = check;

	check = gtk_check_button_new_with_label (_("Buffer size"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[BUFFERSIZE] = check;

	adj = gtk_adjustment_new (128, 128, 4096, 1, 8, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 2, 3, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	rpd->spin[BUFFERSIZE_NUMBER] = spin;

	check = gtk_check_button_new_with_label (_("No sample caching"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOSAMPLECACHING] = check;

	check = gtk_check_button_new_with_label (_("No echo"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOECHO] = check;
		
	check = gtk_check_button_new_with_label (_("No master volume"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOMASTERVOLUME] = check;

	gtk_signal_connect (GTK_OBJECT (rpd->check[NOSOUND]),
			    "toggled",
			    GTK_SIGNAL_FUNC (rom_prefs_nosound_cb),
			    (gpointer) rpd);
	
	gtk_signal_connect (GTK_OBJECT (rpd->check[SOUNDSKIP]),
			    "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer)
			    (rpd->spin[SOUNDSKIP_NUMBER]));

	gtk_signal_connect (GTK_OBJECT (rpd->check[SOUNDQUALITY]),
			    "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer)
			    (rpd->spin[SOUNDQUALITY_NUMBER]));
	
	gtk_signal_connect (GTK_OBJECT (rpd->check[BUFFERSIZE]),
			    "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer)
			    (rpd->spin[BUFFERSIZE_NUMBER]));
	
	return table;
}

GtkWidget*
rom_prefs_cheat_ops (ROMPrefsData *rpd)
{
	gint i;
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *label;
	GtkWidget *button;

	F_NAME();
	
	table = gtk_table_new (6, 11, FALSE);
	
	check = gtk_check_button_new_with_label (_("Game Genie Code(s)"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[GAMEGENIE] = check;
	
	check = gtk_check_button_new_with_label (_("Action Replay Code(s)"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[ACTIONREPLAY] = check;

	check = gtk_check_button_new_with_label (_("Gold Finger Code(s)"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 4, 5,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[GOLDFINGER] = check;

	for (i = 0; i < MAX_CODES; i++) {
		if (rpd->rom_info->gamegenie_code[i])
			label = gtk_label_new
				(rpd->rom_info->gamegenie_code[i]);
		else
			label = gtk_label_new ("-");

		gtk_table_attach (GTK_TABLE (table), label, i+1, i+2, 0, 1,
				  GTK_SHRINK,GTK_SHRINK, 5, 2);
		gtk_widget_set_usize (label, 50, 23);
		gtk_widget_show (label);
		rpd->code[0][i] = label;
		
		if (rpd->rom_info->actionreplay_code[i])
			label = gtk_label_new
				(rpd->rom_info->actionreplay_code[i]);
		else
			label = gtk_label_new ("-");
		
		gtk_table_attach (GTK_TABLE (table), label, i+1, i+2, 2, 3,
				  GTK_SHRINK, GTK_SHRINK, 5, 2);
		gtk_widget_set_usize (label, 50, 23);
		gtk_widget_show (label);
		rpd->code[1][i] = label;
		
		if (rpd->rom_info->goldfinger_code[i])
			label = gtk_label_new
				(rpd->rom_info->goldfinger_code[i]);
		else
			label = gtk_label_new ("-");
		gtk_table_attach (GTK_TABLE (table), label, i+1, i+2, 4, 5,
				  GTK_SHRINK, GTK_SHRINK, 5, 2);
		gtk_widget_set_usize (label, 50, 23);
		gtk_widget_show (label);
		rpd->code[2][i] = label;
	}

	for (i = 0; i < CODES; i++) {
		button = gtk_button_new_with_label (_("Change"));
		gtk_table_attach (GTK_TABLE (table), button, 1, 5,
				  (i*2)+1, (i*2)+2, GTK_FILL, GTK_FILL, 5, 2);
		gtk_widget_show (button);
		rpd->code_change[i] = button;
		
		gtk_signal_connect (GTK_OBJECT
				    (rpd->check[GAMEGENIE+i]),
				    "toggled",
				    GTK_SIGNAL_FUNC (toggle_to_active_cb),
				    button);

		button = gtk_button_new_with_label (_("Clean"));
		gtk_table_attach (GTK_TABLE (table), button, 7, 11,
				  (i*2)+1, (i*2)+2, GTK_FILL, GTK_FILL, 5, 2);
		gtk_widget_show (button);
		rpd->code_clean[i] = button;

		gtk_signal_connect (GTK_OBJECT
				    (rpd->check[GAMEGENIE+i]),
				    "toggled",
				    GTK_SIGNAL_FUNC (toggle_to_active_cb),
				    button);
	}
	
	gtk_signal_connect (GTK_OBJECT (rpd->code_change[GG_CODE]),
			    "clicked", GTK_SIGNAL_FUNC (gg_codes_change_cb),
			    (gpointer) rpd);
	gtk_signal_connect (GTK_OBJECT (rpd->code_clean[GG_CODE]),
			    "clicked", GTK_SIGNAL_FUNC (gg_codes_clean_cb),
			    (gpointer) rpd);
	gtk_signal_connect (GTK_OBJECT (rpd->code_change[AR_CODE]),
			    "clicked", GTK_SIGNAL_FUNC (ar_codes_change_cb),
			    (gpointer) rpd);
	gtk_signal_connect (GTK_OBJECT (rpd->code_change[AR_CODE]),
			    "clicked", GTK_SIGNAL_FUNC (ar_codes_clean_cb),
			    (gpointer) rpd);
	gtk_signal_connect (GTK_OBJECT (rpd->code_change[GF_CODE]),
			    "clicked", GTK_SIGNAL_FUNC (gf_codes_change_cb),
			    (gpointer) rpd);
	gtk_signal_connect (GTK_OBJECT (rpd->code_change[GF_CODE]),
			    "clicked", GTK_SIGNAL_FUNC (gf_codes_clean_cb),
			    (gpointer) rpd);
	
	return table;
}

GtkWidget*
rom_prefs_speed_ops (ROMPrefsData *rpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *spin;
	GtkObject *adj;	

	F_NAME();
	
	table = gtk_table_new (3, 2, TRUE);

	check = gtk_check_button_new_with_label (_("Frame skip"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[FRAMESKIP] = check;

	adj = gtk_adjustment_new (1, 1, G_MAXINT, 1, 10, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	rpd->spin[FRAMESKIP_NUMBER] = spin;


	check = gtk_check_button_new_with_label (_("Frame time"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[FRAMETIME] = check;

	adj = gtk_adjustment_new (1, 1, 100, 0.1, 1, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 1);
	gtk_table_attach (GTK_TABLE (table), spin, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	rpd->spin[FRAMETIME_NUMBER] = spin;

	check = gtk_check_button_new_with_label (_("Cycles"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[CYCLES] = check;

	adj = gtk_adjustment_new (100, 0, 200, 1, 10, 0);
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
				    0.5, 0);
	gtk_table_attach (GTK_TABLE (table), spin, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (spin);
	rpd->spin[CYCLES_NUMBER] = spin;

	gtk_signal_connect (GTK_OBJECT (rpd->check[FRAMESKIP]),
			    "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) rpd->spin[FRAMESKIP_NUMBER]);

	gtk_signal_connect (GTK_OBJECT (rpd->check[FRAMETIME]),
			    "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) rpd->spin[FRAMETIME_NUMBER]);

	gtk_signal_connect (GTK_OBJECT (rpd->check[CYCLES]),
			    "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) rpd->spin[CYCLES_NUMBER]);
	
	return table;
}

GtkWidget*
rom_prefs_format_ops (ROMPrefsData *rpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *text;

	F_NAME();
	
	table = gtk_table_new (4, 4, FALSE);

	check = gtk_check_button_new_with_label (_("Interleaved"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[INTERLEAVED] = check;

	check = gtk_check_button_new_with_label (_("Interleaved 2"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[INTERLEAVED2] = check;

	check = gtk_check_button_new_with_label (_("Hi-ROM"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[HIROM] = check;

	check = gtk_check_button_new_with_label (_("Lo-ROM"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[LOROM] = check;
	
	check = gtk_check_button_new_with_label (_("Header"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[HEADER] = check;

	check = gtk_check_button_new_with_label (_("No header"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOHEADER] = check;

	check = gtk_check_button_new_with_label (_("Pal"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[PAL] = check;

	check = gtk_check_button_new_with_label (_("NTSC"));
	gtk_table_attach (GTK_TABLE (table), check, 1, 2, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NTSC] = check;

	check = gtk_check_button_new_with_label (_("Layering"));
	gtk_table_attach (GTK_TABLE (table), check, 2, 3, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[LAYERING] = check;

	check = gtk_check_button_new_with_label (_("No H-DMA"));
	gtk_table_attach (GTK_TABLE (table), check, 2, 3, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOHDMA] = check;

	check = gtk_check_button_new_with_label (_("No speed hacks"));
	gtk_table_attach (GTK_TABLE (table), check, 2, 3, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOSPEEDHACKS] = check;

	check = gtk_check_button_new_with_label (_("No windows"));
	gtk_table_attach (GTK_TABLE (table), check, 2, 3, 3, 4,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOWINDOWS] = check;

	check = gtk_check_button_new_with_label (_("Load snapshot file"));
	gtk_table_attach (GTK_TABLE (table), check, 3, 4, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[LOADSNAPSHOT] = check;

	text = gnome_file_entry_new(NULL, _("Select ROM file"));
	rpd->browse_snapfile = text;
	gnome_file_entry_set_default_path (GNOME_FILE_ENTRY(text), 
			get_default_snap_dir());
	gtk_table_attach (GTK_TABLE (table), text, 3, 4, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	rpd->entry[SNAPSHOT_FILE] =  gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(text));
	
	gtk_signal_connect (GTK_OBJECT (rpd->check[LOADSNAPSHOT]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) rpd->entry[SNAPSHOT_FILE]);

	gtk_signal_connect (GTK_OBJECT (rpd->check[LOADSNAPSHOT]), "toggled",
			    GTK_SIGNAL_FUNC (toggle_to_active_cb),
			    (gpointer) rpd->browse_snapfile);
	
	return table;
}

GtkWidget*
rom_prefs_joystick_ops (ROMPrefsData *rpd)
{
	GtkWidget *table;
	GtkWidget *check;
	GtkWidget *spin;
	GtkWidget *button;
	GtkObject *adj;
	gchar     *label;
	gint       i, j;
	const gchar *button_names[] = {"A", "B", "X", "Y", "TL", "TR", 
		"Start", "Select"};


	F_NAME();
	
	table = gtk_table_new (5, 11, FALSE);

	for (j = 0; j < NUM_OF_BUTTONS; j++) {
		button = gtk_button_new_with_label (_(button_names[j]));
		gtk_table_attach (GTK_TABLE (table), button, j+2, j+3, 0, 1,
				  GTK_FILL, GTK_FILL, 5, 2);
		gtk_widget_show (button);
	}

	check = gtk_check_button_new_with_label (_("No joystick"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[NOJOY] = check;

	check = gtk_check_button_new_with_label (_("Swap joysticks"));
	gtk_table_attach (GTK_TABLE (table), check, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (check);
	rpd->check[SWAP] = check;

	label = g_strdup (_("Joymap x"));
	
	for (i = 0; i < NUM_OF_JOYS; i++) {
		label[strlen(_("Joymap x"))-1] = (gchar)('0'+i+1);
		check = gtk_check_button_new_with_label (label);
		gtk_table_attach (GTK_TABLE (table), check, 1, 2, i+1, i+2,
				  GTK_FILL, GTK_FILL, 5, 2);
		gtk_widget_show (check);
		rpd->check[JOYMAP0+i] = check;
		for (j = 0; j < NUM_OF_BUTTONS; j++) {
			adj = gtk_adjustment_new (rpd->rom_info->buttons[i][j],
						  0, 9, 1, 1, 0);
			spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj),
						    0.5, 0);
			gtk_table_attach (GTK_TABLE (table), spin,
					  j+2, j+3, i+1, i+2,
					  GTK_FILL, GTK_FILL, 5, 2);
			gtk_widget_set_usize (spin, 35, 23);
			gtk_widget_show (spin);
			rpd->buttons[i][j] = spin;
		}
	}

	gtk_signal_connect (GTK_OBJECT (rpd->check[NOJOY]), "toggled",
			    GTK_SIGNAL_FUNC (rom_prefs_nojoy_cb),
			    (gpointer) rpd);
	
	for (i = 0; i < NUM_OF_JOYS; i++)
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			gtk_signal_connect (GTK_OBJECT (rpd->check[JOYMAP0+i]),
					    "toggled",
					    GTK_SIGNAL_FUNC
					    (toggle_to_active_cb),
					    (gpointer) rpd->buttons[i][j]);
	
	g_free (label);
	
	return table;
}

void
rom_prefs_nosound_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	gint i;	

	F_NAME();
	
	if (GTK_TOGGLE_BUTTON (widget)->active) {
		gtk_widget_set_sensitive
			(rpd->spin[SOUNDSKIP_NUMBER],
			 GTK_TOGGLE_BUTTON
			 (rpd->check[SOUNDSKIP])->active);
		gtk_widget_set_sensitive (rpd->mono, FALSE);

		for (i = STEREO; i <= NOMASTERVOLUME; i++)
			gtk_widget_set_sensitive (rpd->check[i], FALSE);

		gtk_widget_set_sensitive
			(rpd->check[SOUNDSKIP], TRUE);
		
		gtk_widget_set_sensitive (rpd->
					  spin[SOUNDQUALITY_NUMBER],
					  FALSE);
		gtk_widget_set_sensitive (rpd->
					  spin[BUFFERSIZE_NUMBER],
					  FALSE);
	} else {
		gtk_widget_set_sensitive
			(rpd->spin[SOUNDSKIP_NUMBER], FALSE);
		gtk_widget_set_sensitive (rpd->mono, TRUE);

		for (i = STEREO; i <= NOMASTERVOLUME; i++)
			gtk_widget_set_sensitive (rpd->check[i], TRUE);

		gtk_widget_set_sensitive
			(rpd->check[SOUNDSKIP], FALSE);
		
		gtk_widget_set_sensitive
			(rpd->spin[SOUNDQUALITY_NUMBER],
			 GTK_TOGGLE_BUTTON
			 (rpd->check[SOUNDQUALITY])->active);
		gtk_widget_set_sensitive
			(rpd->spin[BUFFERSIZE_NUMBER],
			 GTK_TOGGLE_BUTTON
			 (rpd->check[BUFFERSIZE])->active);
	}	
}

void
rom_prefs_nojoy_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	gint i, j;

	F_NAME();
	
	if (GTK_TOGGLE_BUTTON (widget)->active) {
		gtk_widget_set_sensitive (rpd->check[SWAP], FALSE);
		for (i = 0; i < NUM_OF_JOYS; i++) {
			gtk_widget_set_sensitive (rpd->check[JOYMAP0+i],
						  FALSE);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive (rpd->buttons[i][j],
							  FALSE);
		}
	} else {
		gtk_widget_set_sensitive (rpd->check[SWAP], TRUE);
		for (i = 0; i < NUM_OF_JOYS; i++) {
			gtk_widget_set_sensitive (rpd->check[JOYMAP0+i], TRUE);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive
					(rpd->buttons[i][j],
					 GTK_TOGGLE_BUTTON
					 (rpd->check[JOYMAP0+i])->active);
		}
	}	
}

void
rom_prefs_dialog_changed_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	FILE  *file;
	GList *list;
	gchar *rom_info_file, *name;
	gint   i, l;

	F_NAME();
	
	gnome_dialog_set_sensitive (GNOME_DIALOG (rpd->rom_prefs_dialog),
				    0, TRUE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (rpd->rom_prefs_dialog),
				    1, TRUE);
	
	rpd->nodata = FALSE;
	
	if (rpd->changed)
		return;
	
	rom_info_file = g_strconcat (CONF_DIR, "/",
				     rpd->rom_info->string_prefs[NAME], NULL);
		
	remove (gnome_util_home_file (rom_info_file));
	
	g_free (rom_info_file);

	list = g_list_first  (rpd->data->roms_list);
	l    = g_list_length (list);

	if (l > 1) {
		file = fopen (gnome_util_home_file (ROMS_FILE), "w");
		
		for (i = 0; i < l; i++) {
			name = ((ROMInfo*)
				(list->data))->string_prefs[NAME];
			if (strcmp (name,
				    rpd->rom_info->string_prefs[NAME]))
				fprintf (file, "%s\n", name);
			list = g_list_next (list);
		}

		fclose (file);
	} else
		remove (gnome_util_home_file (ROMS_FILE));

	rpd->changed = TRUE;
}

void
rom_prefs_connect_changes (ROMPrefsData *rpd)
{
	gint i, j;

	F_NAME();

	for (i = 0; i < MIX_BOOL_PREFS+ROM_BOOL_PREFS; i++)
		gtk_signal_connect
			(GTK_OBJECT (rpd->check[i]), "toggled",
			 GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
			 (gpointer) rpd);
	
	for (i = 0; i < MIX_NUM_PREFS+ROM_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER)
			gtk_signal_connect
			(GTK_OBJECT (gtk_option_menu_get_menu 
				     (GTK_OPTION_MENU (rpd->spin[FILTER_NUMBER]))),
			 "selection-done",
		 	GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
			(gpointer) rpd);
		else
		gtk_signal_connect
			(GTK_OBJECT (rpd->spin[i]), "changed",
			 GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
			 (gpointer) rpd);
		}
		
	for (i = 0; i < ROM_STR_PREFS; i++)
		gtk_signal_connect
			(GTK_OBJECT (rpd->entry[i]), "changed",
			 GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
			 (gpointer) rpd);

	for (i = 0; i < NUM_OF_JOYS; i++) {
		for (j = 0; j < NUM_OF_BUTTONS; j++)
			gtk_signal_connect
				(GTK_OBJECT (rpd->buttons[i][j]), "changed",
				 GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
				 (gpointer) rpd);
	}
	
	for (i = 0; i < CODES; i++) {
		gtk_signal_connect
			(GTK_OBJECT (rpd->code_clean[i]), "clicked",
			 GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
			 (gpointer) rpd);
	}

	gtk_signal_connect (GTK_OBJECT (rpd->entry[SNAPSHOT_FILE]), "changed",
			    GTK_SIGNAL_FUNC (rom_prefs_dialog_changed_cb),
			    (gpointer) rpd);
}

void
rom_prefs_set_tooltips (ROMPrefsData *rpd)
{
	gint         i;
	GtkTooltips *tool;

	F_NAME();
	
	tool = gtk_tooltips_new ();
	
	gtk_tooltips_set_tip (tool, rpd->check[TRANSPARENCY],
			      _("Enable transparency effects, also enables "
				"16-bit screen mode selection."
				"Transparency effects are automatically "
				"enabled if the depth of your X  "
				"server is 15 or greater."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[SIXTEEN],
			      _("Enable 16-bit internal screen rendering, "
				"allows palette changes but no "
				"transparency effects."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[HIRES],
			      _("Enable support for SNES hi-res. and "
				"interlace modes. USE ONLY IF GAME "
				"REQUIRES IT (FEW DO) BECAUSE IT REALLY "
				"SLOWS DOWN THE EMULATOR."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOMODESWITCH],
			      _("The Linux X Windows System port can change "
				"the screen resolution when "
				"switching to full-screen mode so the SNES "
				"display fills the whole screen. "
				"Specify this option to stop it if causes "
				"you problems."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[FULLSCREEN],
			      _("Start the emulator in full screen mode, "
				"rather than a window on the X11 "
				"port if supported by your X server, a "
				"suitable screen mode is defined in "
				"your XF86Config file and the snes9x binary "
				"has root permissions (see above)."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[SCALE],
			      _("Stretch the SNES display to fit the whole "
				"of the computer display. "
				"Linux X Window System full-screen mode or "
				"SVGA port only. Use only if you "
				"have a very fast machine."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[DISPLAYFRAMERATE],
			      _("Display a frame rate counter superimposed "
				"over the bottom, left-hand corner "
				"of the SNES display. The value before the "
				"slash (/) indicates how many "
				"frames per second are being rendered "
				"compared to a real SNES, the value of "
				"which is displayed after the slash."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOSOUND],
			      _("Disable sound CPU emulation and sound "
				"output, useful for the few ROMs "
				"where sound emulation causes them to lock "
				"up due to timing errors."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[SOUNDSKIP],
			      _("Sound skip. ONLY USED IF SOUND IS DISABLED."),
			      NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[STEREO],
			      _("Enable stereo sound output."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->mono,
			      _("Enable mono sound output. Faster, but "
				"doesn\'t sound as nice."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[SOUNDQUALITY],
			      _("Sound playback rate/quality: "
				"0 - disable sound, 1 - 8192, 2 - 11025, "
				"3 - 16500, 4 - 22050 (default), "
				"5 - 29300, 6 - 36600, 7 - 44000."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[ENVX],
			      _("Enable volume envelope height reading by "
				"the sound CPU. Can cure sound "
				"repeat problems with some games (e.g. "
				"Mortal Kombat series), while causing "
				"others to lock if enabled (eg. Bomberman "
				"series). Use with the \'Sound sync\' "
				"option to fix the problems this option can "
				"otherwise create."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[SOUNDSYNC],
			      _("Turns on sound-sync code where sound data "
				"is generated in sync with the "
				"emulation of the SNES sound CPU "
				"instructions. Slighty slower emulation "
				"speed but helps reduce note and sound "
				"effects timing problems and one or two  "
				"games actually require this mode, notably "
				"Williams Arcade Classics. "
				"Also turns on interpolated sound and "
				"envelope height reading. "
				"Pressing the \']\' key during gameplay "
				"toggles sound-sync only on and off."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[ALTSAMPLEDECODE],
			      _("Switch to using Anti Resonance's sample "
				"decoding method. Some games, notably "
				"Squaresoft RPG games, make use of a SNES "
				"hardare sample decoding bug/feature to "
				"generate wind and swish-type sound effects; "
				"selecting this option enables code that "
				"tries to simulate the feature. The "
				"disadvantage is that it can decode some "
				"\'normal\' samples incorrectly."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[INTERPOLATEDSOUND],
			      _("Turns on interpolated sound. Any samples "
				"played at a lower frequency than "
				"the current playback rate (\'Sound quality\' "
				"option) have additional sample points "
				"computed and played where previously the "
				"same sample data was played "
				"repeatedly. Lower frequency sounds have "
				"improved bass and sound much less "
				"harsh - I suspect the real SNES does this. "
				"Pressing \'[\' during gameplay toggles the "
				"feature on and off."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[BUFFERSIZE],
			      _("Sound playback buffer size in bytes "
				"128-4096."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOSAMPLECACHING],
			      _("Disable decompressed sound sample caching. "
				"Decompressing samples takes time, "
				"slowing down the emulator. Normally the "
				"decompressed samples are saved "
				"just in case they need to be played again, "
				"but the way samples are stored "
				"and played on the SNES, it can result in a "
				"click sound or distortion when "
				"caching samples with loops in them."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOECHO],
			      _("Turn off sound echo and FIR filter effects. "
				"Processing these effects can "
				"really slow down a non-MMX Pentium machine "
				"due to the number of calculations "
				"required to implement these features."),
			      NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOMASTERVOLUME],
			      _("Disable emulation of the sound DSP master "
				"volume control. Some ROMs set "
				"the volume level very low requiring you to "
				"turn up the volume level of "
				"your speakers introducing more background "
				"noise. Use this option to "
				"always have the master volume set on full "
				"and to by-pass a bug which "
				"prevents the music and sound effects being "
				"heard on Turrican."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[GAMEGENIE],
			      _("Supply Game Genie code(s) for the current "
				"ROM. Up to 10 codes can be in "
				"affect at once. Game Genie codes for many "
				"SNES games are available from: "
				"http://game-genie.nvc.cc.ca.us "), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[ACTIONREPLAY],
			      _("Supply Pro-Action Reply code(s) for the "
				"current ROM. Up to 10 codes can be in "
				"affect at once. At the moment, codes which "
				"alter RAM do not work."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[GOLDFINGER],
			      _("Supply Gold Finger code(s) for the current "
				"ROM. Up to 10 codes can be active "
				"at once."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[FRAMESKIP],
			      _("Set this value to deliberately fix the "
				"frame skip rate and disable auto-speed "
				"regulation. Use a larger value faster "
				"emulation but more jerky movement and a "
				"smaller value for smooth but slower screen "
				"updates. Use \'+\' and \'-\' keys to modify "
				"the value during a game. Ideal for some "
				"Super FX games that confuse the auto-adjust "
				"code or for games that deliberately flash "
				"the screen every alternate frame."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[FRAMETIME],
			      _("(default: 16.6ms NTSC games and 20ms for "
				"PAL games) If auto-adjust frame skip option "
				"is in effect, then the emulator will try "
				"to maintain a constant game and music speed "
				"locked to this value by skipping "
				"the rendering of some frames or waiting "
				"until the required time is reached. "
				"Increase the value to slow down games, "
				"decrease it to speed up games. "
				"During a game the value can be adjusted "
				"in millisecond steps by pressing "
				"Shift \'-\' or Shift \'+\'."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[CYCLES],
			      _("Percentage of CPU cycles to execute per "
				"scan line, decrease value to "
				"increase emulation frame rate. Most "
				"ROMs work with a value of 85 or above."),
			      NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[INTERLEAVED],
			      _("Force interleaved ROM image format."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[INTERLEAVED2],
			      _("Force alternate interleaved format (i.e. "
			      "most Super FX games)."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[HIROM],
			      _("Force Hi-ROM memory map for ROMs where the "
				"Hi-ROM header test fails."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[LOROM],
			      _("Force Lo-ROM memory map for ROMs where the "
				"Hi-ROM header test fails."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[HEADER],
			      _("Force the detection of a ROM image header. "
				"Some ROM images have been  "
				"hand-edited to remove unused space from the "
				"end of the file; if the "
				"resultant image size is not a multiple of "
				"32k then Snes9X can\'t  "
				"auto-detect the presense of a 512 byte ROM "
				"image header."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOHEADER],
			      _("Force Snes9X into thinking no ROM image "
				"header is present. See \'Header\' "
				"above."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[PAL],
			      _("Fool ROM into thinking this is a PAL SNES "
				"system and adjust frame time "
				"to 20ms (50 frames per second) "), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NTSC],
			      _("Fool ROM into thinking this is a NTSC SNES "
				"system and adjust frame time "
				"to 16.7ms (60 frames per second) "), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[LAYERING],
			      _("Swap background layer priorities from "
				"background involved in sub-screen "
				"addition/subtraction. Can improve some "
				"games play-ability - no need to "
				"constantly toggle background layers on and "
				"off to read text/see maps, etc. "
				"Toggle feature on and off during game by "
				"pressing \'8\'. "
				"Not used if transparency effects are "
				"enabled."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[LOADSNAPSHOT],
			      _("Load snapshot file and restart game from "
				"saved position."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOHDMA],
			      _("Turn off the H-DMA emulation. Pressing "
				"\'0\' during a game toggles H-DMA on "
				"and off."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOSPEEDHACKS],
			      _("Turn off a couple of speed hacks. The hacks "
				"boost the speed of many ROMs "
				"but cause problems a few ROMs."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOWINDOWS],
			      _("Disable graphics windows emulation. Use "
				"\'backspace\' key during a game to "
				"toggle the emulation on and off."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[SWAP],
			      _("Swap emulated joy-pad 1 and 2 around, "
				"pressing \'6\' during a game does the same "
				"thing."), NULL);
	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->check[NOJOY],
			      _("Turn off joystick, SideWinder and GrIP "
				"detection (joystick polling on the "
				"PC slows the emulator down)."), NULL);
	
	for (i = 0; i < NUM_OF_JOYS; i++) {
		tool = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tool, rpd->check[JOYMAP0+i],
				      _("Specify the SNES joypad buttons to "
					"Linux joystick driver buttons "
					"mapping for each of the four "
					"supported joypads. Specify the Linux "
					"joystick button number for the "
					"corresponding SNES button in the "
					"following order: A, B, X, Y, TL, TR, "
					"Start and Select The default map is: "
					"1 0 4 3 6 7 8 9 which is suitable "
					"for Sidewinder gamepads."), NULL);
	}

	tool = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tool, rpd->load_defaults,
			      _("Reset this ROM to the defaults specified in "
				"Global Preferences."), NULL);
	
}

void
rom_prefs_dialog_set_status (ROMPrefsData *rpd)
{
	gint i, j;
	gint numberops[] = {
		SOUNDSKIP,
		SOUNDQUALITY,
		BUFFERSIZE,
		FRAMESKIP,
		FRAMETIME,
		CYCLES
	};

	F_NAME();
	
	gtk_entry_set_text (GTK_ENTRY (rpd->entry[ROM_FILE]),
			    rpd->rom_info->string_prefs[ROM_FILE]);
	gtk_entry_set_text (GTK_ENTRY (rpd->entry[NAME]),
			    rpd->rom_info->string_prefs[NAME]);
	gtk_entry_set_text (GTK_ENTRY (rpd->entry[COMPANY]),
			    rpd->rom_info->string_prefs[COMPANY]);
	gtk_entry_set_text (GTK_ENTRY (rpd->entry[IMAGE_FILE]),
			    rpd->rom_info->string_prefs[IMAGE_FILE]);

	for (i = 0; i < MIX_BOOL_PREFS+ROM_BOOL_PREFS; i++)
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (rpd->check[i]),
			 rpd->rom_info->bool_prefs[i]);

	for (i = 0; i < MIX_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER) {
			 gtk_option_menu_set_history (
					 GTK_OPTION_MENU (rpd->spin[i]),
					 rpd->rom_info->number_prefs[i]);
		}
		else {
			if (rpd->rom_info->bool_prefs[(numberops[i])])
				gtk_spin_button_set_value
					(GTK_SPIN_BUTTON (rpd->spin[i]),
				 	rpd->rom_info->number_prefs[i]);
			else
				gtk_widget_set_sensitive
					(rpd->spin[i], FALSE);
		}
	}	

	if (rpd->rom_info->bool_prefs[STEREO])
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (rpd->check[STEREO]), TRUE);
	else
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (rpd->mono), TRUE);
	
	if (rpd->rom_info->bool_prefs[NOSOUND])
		for (i = STEREO; i < NOMASTERVOLUME; i++)
			gtk_widget_set_sensitive (rpd->check[i], FALSE);
	else {
		gtk_widget_set_sensitive
			(rpd->check[SOUNDSKIP], FALSE);
		gtk_widget_set_sensitive
			(rpd->spin[SOUNDSKIP_NUMBER], FALSE);
	}

	for (i = 0; i < CODES; i++) {
		if (!rpd->rom_info->bool_prefs[GAMEGENIE+i]) {
			gtk_widget_set_sensitive
				(rpd->code_change[i], FALSE);
			gtk_widget_set_sensitive
				(rpd->code_clean[i], FALSE);
		}
	}

	if (rpd->rom_info->bool_prefs[NOJOY]) {
		gtk_widget_set_sensitive (rpd->check[SWAP], FALSE);
		for (i = 0; i < NUM_OF_JOYS; i++) {
			gtk_widget_set_sensitive (rpd->check[JOYMAP0+i],
						  FALSE);
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive (rpd->buttons[i][j],
							  FALSE);
		}
	}

	for (i = 0; i < NUM_OF_JOYS; i++)
		if (!rpd->rom_info->bool_prefs[JOYMAP0+i])
			for (j = 0; j < NUM_OF_BUTTONS; j++)
				gtk_widget_set_sensitive (rpd->buttons[i][j],
							  FALSE);

	if (rpd->rom_info->string_prefs[SNAPSHOT_FILE]) {
		gtk_entry_set_text
			(GTK_ENTRY (rpd->entry[SNAPSHOT_FILE]),
			 rpd->rom_info->string_prefs[SNAPSHOT_FILE]);
	}
	
	if (!rpd->rom_info->bool_prefs[LOADSNAPSHOT]) {
		gtk_widget_set_sensitive (rpd->entry[SNAPSHOT_FILE], FALSE);
		gtk_widget_set_sensitive (rpd->browse_snapfile, FALSE);
	}
}

void
rom_prefs_dialog_get_status (ROMPrefsData *rpd)
{
	gchar *text;
	gint   i, j, number;	

	F_NAME();
	
	for (i = 0; i < ROM_STR_PREFS; i++) {
		text = gtk_entry_get_text (GTK_ENTRY (rpd->entry[i]));
		if (strlen (text)) {
			g_free (rpd->rom_info->string_prefs[i]);
			rpd->rom_info->string_prefs[i] = g_strdup (text);
		}
	}
	
	for (i = 0; i < MIX_BOOL_PREFS+ROM_BOOL_PREFS; i++)
		rpd->rom_info->bool_prefs[i] =
			GTK_TOGGLE_BUTTON (rpd->check[i])->active;

	for (i = 0; i < MIX_NUM_PREFS; i++) {
		if (i == FILTER_NUMBER) 
			rpd->rom_info->number_prefs[i] =
			       	g_list_index (GTK_MENU_SHELL (gtk_option_menu_get_menu 
						 (GTK_OPTION_MENU 
						  (rpd->spin[i])))->children, 
						GTK_MENU_ITEM (gtk_menu_get_active 
							(GTK_MENU
							 (gtk_option_menu_get_menu
							  (GTK_OPTION_MENU 
							   (rpd->spin[i]))))));
		else 
			rpd->rom_info->number_prefs[i] =
				gtk_spin_button_get_value_as_float
				(GTK_SPIN_BUTTON (rpd->spin[i]));
	}
		
	for (i = 0; i < NUM_OF_JOYS; i++)
		rpd->rom_info->bool_prefs[JOYMAP0+i] =
			GTK_TOGGLE_BUTTON (rpd->check[JOYMAP0+i])->active;

	for (i = 0; i < NUM_OF_JOYS; i++) {
		if (rpd->check[JOYMAP0+i])
			for (j = 0; j < NUM_OF_BUTTONS; j++) {
				number = gtk_spin_button_get_value_as_int
					(GTK_SPIN_BUTTON (rpd->buttons[i][j]));
				rpd->rom_info->buttons[i][j] = (gshort) number;
			}
	}
	
	for (i = 0; i < MAX_CODES; i++) {
		if (rpd->rom_info->gamegenie_code[i])
			g_free (rpd->rom_info->gamegenie_code[i]);
		gtk_label_get (GTK_LABEL (rpd->code[0][i]), &text);

		if (strcmp (text, "-"))
			rpd->rom_info->gamegenie_code[i] = g_strdup (text);
		else
			rpd->rom_info->gamegenie_code[i] = NULL;
		
		if (rpd->rom_info->actionreplay_code[i])
			g_free (rpd->rom_info->actionreplay_code[i]);
		gtk_label_get (GTK_LABEL (rpd->code[1][i]), &text);

		if (strcmp (text, "-"))
			rpd->rom_info->actionreplay_code[i] = g_strdup (text);
		else
			rpd->rom_info->actionreplay_code[i] = NULL;
		
		if (rpd->rom_info->goldfinger_code[i])
			g_free (rpd->rom_info->goldfinger_code[i]);
		gtk_label_get (GTK_LABEL (rpd->code[2][i]), &text);

		if (strcmp (text, "-"))
			rpd->rom_info->goldfinger_code[i] = g_strdup (text);
		else
			rpd->rom_info->goldfinger_code[i] = NULL;
	}
	
	text = gtk_entry_get_text (GTK_ENTRY (rpd->entry[SNAPSHOT_FILE]));
	
	if (rpd->rom_info->bool_prefs[LOADSNAPSHOT] && strlen (text)) {
		if (rpd->rom_info->string_prefs[SNAPSHOT_FILE])
			g_free (rpd->rom_info->string_prefs[SNAPSHOT_FILE]);
		rpd->rom_info->string_prefs[SNAPSHOT_FILE] = g_strdup (text);
	}
}

void
rom_prefs_apply_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	gint    i, l, new_pos = -1;
	GList   *list;
	gchar   *name, *company, *rom_file, *img_file;
	gboolean repeat = FALSE;

	F_NAME();
	
	if (!rpd->changed)
		return;

	if (!check_rom_info (rpd->entry))
       	{
		rpd->nodata = TRUE;
		no_rom_file (rpd->rom_prefs_dialog);
		return;
	}

	if (!strlen (gtk_entry_get_text
		     (GTK_ENTRY (rpd->entry[SNAPSHOT_FILE]))) &&
	    GTK_TOGGLE_BUTTON (rpd->check[LOADSNAPSHOT])->active) {
		rpd->nodata = TRUE;
		no_snapfile (rpd->rom_prefs_dialog);
		return;
	}
	
	name     = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (rpd->entry[NAME])));
	company  = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (rpd->entry[COMPANY])));
	rom_file = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (rpd->entry[ROM_FILE])));
	img_file = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (rpd->entry[IMAGE_FILE])));

	l = strlen (name);

	for (i = 0; i < l; i++)
		if (name[i] == '/') {
			g_free (name);
			g_free (company);
			g_free (rom_file);
			g_free (img_file);
			no_correct_name (rpd->rom_prefs_dialog);
			return;
		}

	if (strcmp (name, rpd->rom_info->string_prefs[NAME])) {
		list = g_list_first (rpd->data->roms_list);
		l    = g_list_length (list);
		
		for (i = 0; i < l; i++) {
			if (!strcmp
			    (((ROMInfo*)(list->data))->string_prefs[NAME],
			     name))
				repeat = TRUE;
			list = g_list_next (list);
		}
	}

	if (repeat) {
		rom_repeated (name, rpd->rom_prefs_dialog);
		g_free (name);
		g_free (company);
		g_free (rom_file);
		g_free (img_file);
		return;
	}
	
	rom_prefs_dialog_get_status (rpd);
	
	save_rom (rpd->rom_info);

	rpd->changed = FALSE;

	gnome_dialog_set_sensitive (GNOME_DIALOG (rpd->rom_prefs_dialog),
				    0, FALSE);
	gnome_dialog_set_sensitive (GNOME_DIALOG (rpd->rom_prefs_dialog),
				    1, FALSE);

	list = g_list_first (rpd->data->roms_list);

	l = g_list_length (list);

	gtk_clist_set_text (GTK_CLIST (rpd->data->clist),
			    rpd->data->current_rom, 0, name);

	gtk_clist_sort (GTK_CLIST (rpd->data->clist));

	g_free (name);
	g_free (company);
	g_free (rom_file);
	g_free (img_file);
	
	for (i = 0; i < l; i++) {
		gtk_clist_get_text (GTK_CLIST (rpd->data->clist),
				    i, 0, &name);
		if (!strcmp (name, rpd->rom_info->string_prefs[NAME]))
			new_pos = i;
	}

	rpd->data->current_rom = new_pos;
	
	gtk_clist_select_row (GTK_CLIST (rpd->data->clist),
			      rpd->data->current_rom, 0);
	
}

void
rom_prefs_close_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	GtkWidget *warning;

	F_NAME();
	
	if (rpd->changed && !rpd->nodata) {
		warning = gnome_message_box_new
			(_("You changed the ROM preferences.\n"
			 "Are you sure?"),
			 GNOME_MESSAGE_BOX_WARNING,
			 GNOME_STOCK_BUTTON_YES,
			 GNOME_STOCK_BUTTON_NO,
			 NULL);

			gtk_window_set_modal (GTK_WINDOW (warning), TRUE);
			gtk_window_set_transient_for
				(GTK_WINDOW (warning),
				 GTK_WINDOW (rpd->rom_prefs_dialog));
		
		gnome_dialog_button_connect (GNOME_DIALOG (warning),
					     0, GTK_SIGNAL_FUNC
					     (rom_prefs_destroy_cb),
					     (gpointer) rpd);
		gnome_dialog_button_connect (GNOME_DIALOG (warning),
					     1, GTK_SIGNAL_FUNC
					     (destroy_data_cb),
					     (gpointer) warning);

		gtk_widget_show (warning);
	}
	
	if (!rpd->changed && !rpd->nodata)
		rom_prefs_destroy_cb (NULL, rpd);	
}

void
rom_prefs_help_cb (GtkWidget *widget, GtkWidget *notebook)
{
	gint part;
	
	F_NAME();
	
	part = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook))+1;

	readme_cb (NULL, part);
}

void
rom_prefs_destroy_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	F_NAME();
	
	if (rpd->changed)
		save_rom (rpd->rom_info);

	rpd->data->dialog_open = FALSE;
	
	gtk_widget_destroy (rpd->rom_prefs_dialog);
}

gboolean
rom_prefs_delete_cb (GtkWidget *widget)
{
	ROMPrefsData *rpd =
		(ROMPrefsData*) gtk_object_get_data (GTK_OBJECT (widget),
						     "rom-prefs");
	
	F_NAME();
	
	rom_prefs_close_cb (NULL, rpd);
	
	return TRUE;
}

void
rom_prefs_free_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	F_NAME();
	
	g_free (rpd);
}

void
rom_repeated (gchar *name, GtkWidget *parent)
{
	GtkWidget *error;
	gchar     *msg;

	F_NAME();


	msg = g_strdup_printf (_("Sorry, the ROM \"%s\"already exists."),
			       name);
		
	error = gnome_message_box_new(msg,
				      GNOME_MESSAGE_BOX_ERROR, 
				      GNOME_STOCK_BUTTON_OK,
				      NULL);

	gtk_window_set_modal (GTK_WINDOW (error), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (error),
				      GTK_WINDOW (parent));
	
	gtk_widget_show (error);

	g_free (msg);
}

void
no_rom_file (GtkWidget *parent) 
{
	GtkWidget *error;

	F_NAME();
	
	error = gnome_message_box_new(_("You must fill the \"ROM File\" entry."),
				      GNOME_MESSAGE_BOX_ERROR, 
				      GNOME_STOCK_BUTTON_OK,
				      NULL);

	gtk_window_set_modal (GTK_WINDOW (error), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (error),
				      GTK_WINDOW (parent));
	
	gtk_widget_show (error);	
}

void
no_snapfile (GtkWidget *parent)
{
	GtkWidget *error;

	F_NAME();
	
	error = gnome_message_box_new(_("The \"Load snapshot file\" option\n"
					"is selected, but the \"Snapshot "
					"file\" entry is empty."),
				      GNOME_MESSAGE_BOX_ERROR, 
				      GNOME_STOCK_BUTTON_OK,
				      NULL);

	gtk_window_set_modal (GTK_WINDOW (error), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (error),
				      GTK_WINDOW (parent));
	
	gtk_widget_show (error);
}

void
no_correct_name (GtkWidget *parent)
{
	GtkWidget *error;

	F_NAME();
	
	error = gnome_message_box_new(_("Sorry: the character '/' can't be\n"
					"in the name string."),
				      GNOME_MESSAGE_BOX_ERROR, 
				      GNOME_STOCK_BUTTON_OK,
				      NULL);

	gtk_window_set_modal (GTK_WINDOW (error), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (error),
				      GTK_WINDOW (parent));
	
	gtk_widget_show (error);
}

void
rom_load_defaults_cb (GtkWidget *widget, ROMPrefsData *rpd)
{
	GlobalPrefs *prefs = g_new (GlobalPrefs, 1);

	
	F_NAME();

	prefs = read_conf_file ();
	rom_get_defaults (rpd->rom_info, prefs);	
	g_free (prefs);
	
	rom_prefs_dialog_set_status (rpd);	
	
}


/* romprefs.c ends here. */
