;;; semi-def.el --- definition module for WEMI -*- coding: iso-8859-4; -*-

;; Copyright (C) 1995,96,97,98,99,2000 Free Software Foundation, Inc.

;; Author: MORIOKA Tomohiko <tomo@m17n.org>
;; Keywords: definition, MIME, multimedia, mail, news

;; This file is part of WEMI (Widget based Emacs MIME Implementation).

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or (at
;; your option) any later version.

;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Code:

(eval-when-compile (require 'cl))

(require 'custom)

(defconst mime-user-interface-product ["WEMIKO" (1 14 0) "Zoomastigophora"]
  "Product name, version number and code name of MIME-kernel package.")

(autoload 'mule-caesar-region "mule-caesar"
  "Caesar rotation of current region." t)

(autoload 'widget-convert-button "wid-edit")

;;; @ constants
;;;

(defconst mime-echo-buffer-name "*MIME-echo*"
  "Name of buffer to display MIME-playing information.")

(defconst mime-temp-buffer-name " *MIME-temp*")


;;; @ button
;;;

(define-widget 'mime-button 'link
  "Widget for MIME button."
  :action 'mime-button-action)

(defun mime-button-action (widget &optional event)
  (let ((function (widget-get widget :mime-button-callback))
	(data (widget-get widget :mime-button-data)))
    (when function
      (funcall function data))))

(defun mime-create-widget-button (string function data)
  "Insert STRING as a widget button with a callback FUNCTION and DATA.
Under XEmacs, the function `mime-create-xpm-button' might be identical
to the function `mime-create-widget-button' if the feature `xpm' is not
provided or the TTY frame is used."
  (save-restriction
    (narrow-to-region (point)(point))
    ;; Maybe we should introduce button formatter such as
    ;; `gnus-mime-button-line-format'.
    (insert "[" string "]")
    ;; XEmacs -- when `widget-glyph-enable' is non nil, widget values are not
    ;; guaranteed to be underlain.
    (widget-convert-button 'mime-button (point-min)(point-max)
			   :mime-button-callback function
			   :mime-button-data data)
    (insert "\n")))

(static-when (featurep 'xemacs)
  (defcustom mime-xpm-button-shadow-thickness 3
    "A number of pixels should be used for the shadows on the edges of
the buttons."
    :group 'mime
    :type 'integer)

  (defcustom mime-xpm-button-foreground "Yellow"
    "A color used to display the text."
    :group 'mime
    :type 'string)

  (defcustom mime-xpm-button-background "#a0a0d0"
    "A background color the text will be displayed upon."
    :group 'mime
    :type 'string)

  (defvar mime-xpm-button-glyph-cache nil)

  (if (featurep 'xpm)
      (defun mime-create-xpm-button (string function data)
	"Insert STRING as an XPM button with a callback FUNCTION.
It might be identical to the function `mime-create-widget-button'
if the TTY frame is used."
	;; `device-on-widow-system-p' must be checked at run-time.
	(if (device-on-window-system-p)
	    (let ((lines (split-string string "\n"))
		  line extent spec button
		  down-glyph up-glyph down-func up-func keymap)
	      (while lines
		(setq line (car lines)
		      lines (cdr lines)
		      extent (make-extent (point)
					  (progn
					    (insert "[" line "]")
					    (point)))
		      spec (list line
				 mime-xpm-button-shadow-thickness
				 mime-xpm-button-foreground
				 mime-xpm-button-background)
		      button (cdr (assoc spec mime-xpm-button-glyph-cache)))
		(unless button
		  (set-alist 'mime-xpm-button-glyph-cache spec
			     (setq button (apply (function xpm-button-create)
						 spec))))
		(set-extent-properties extent '(invisible t intangible t))
		(setq extent (make-extent (point) (point))
		      down-glyph (make-glyph (car (cdr button)))
		      up-glyph (make-glyph (car button))
		      down-func `(lambda (event)
				   (interactive "e")
				   (set-extent-begin-glyph ,extent
							   ,down-glyph))
		      up-func `(lambda (event)
				 (interactive "e")
				 (mouse-set-point event)
				 (set-extent-begin-glyph ,extent ,up-glyph)
				 (,function))
		      keymap (make-sparse-keymap))
		(set-extent-begin-glyph extent up-glyph)
		(define-key keymap 'button1 down-func)
		(define-key keymap 'button2 down-func)
		(define-key keymap 'button1up up-func)
		(define-key keymap 'button2up up-func)
		(set-extent-property extent 'keymap keymap)
		(insert "\n")))
	  (mime-create-widget-button string function)))
    (fset 'mime-create-xpm-button 'mime-create-widget-button)))

(defcustom mime-create-button-function 'mime-create-widget-button
  "A function called to create the content button."
  :group 'mime
  :type (list
	 'cons
	 :convert-widget
	 (function
	  (lambda (widget)
	    (list
	     'radio
	     :args
	     (append
	      '((const :tag "Widget button" mime-create-widget-button))
	      (static-when (featurep 'xemacs)
		'((const :tag "Xpm button" mime-create-xpm-button)))
	      '((function :tag "Other"))))))))

(defsubst mime-insert-button (string function &optional data)
  "Insert STRING as button with callback FUNCTION and DATA."
  (funcall mime-create-button-function string function data))


;;; @ for URL
;;;

(defcustom mime-browse-url-regexp
  (concat "\\(http\\|ftp\\|file\\|gopher\\|news\\|telnet\\|wais\\|mailto\\):"
	  "\\(//[-a-zA-Z0-9_.]+:[0-9]*\\)?"
	  "[-a-zA-Z0-9_=?#$@~`%&*+|\\/.,]*[-a-zA-Z0-9_=#$@~`%&*+|\\/]")
  "Regexp to match URL in text body."
  :group 'mime
  :type 'regexp)

(defcustom mime-browse-url-function (function browse-url)
  "Function to browse URL."
  :group 'mime
  :type 'function)

(define-widget 'mime-url-link 'url-link
  "A link to an www page.")

(defsubst mime-add-url-buttons ()
  "Add URL-buttons for text body."
  (goto-char (point-min))
  (while (re-search-forward mime-browse-url-regexp nil t)
    (widget-convert-button 'mime-url-link (match-beginning 0)(match-end 0)
			   (match-string-no-properties 0))))


;;; @ menu
;;;

(defmacro mime-popup-menu-bogus-filter-constructor (menu)
  ;; #### Kludge for FSF Emacs-style menu.
  (let ((bogus-menu (make-symbol "bogus-menu")))
    `(let (,bogus-menu selection function)
       (easy-menu-define ,bogus-menu nil nil ,menu)
       (setq selection (x-popup-menu t ,bogus-menu))
       (when selection
	 (setq function (lookup-key ,bogus-menu (apply #'vector selection)))
	 ;; If a callback entry has no name, easy-menu wraps its value.
	 ;; See `easy-menu-make-symbol'.
	 (if (eq t (compare-strings "menu-function-" 0 nil (symbol-name function) 0 14))
	     (car (last (symbol-function function)))
	   function)))))

;;; While XEmacs can have both X and tty frames at the same time with
;;; gnuclient, we shouldn't emulate in text-mode here.

(static-if (featurep 'xemacs)
    (defalias 'mime-popup-menu-popup 'popup-menu)
  (defun mime-popup-menu-popup (menu &optional event)
    (let ((function (mime-popup-menu-bogus-filter-constructor menu)))
      (when (symbolp function)
	(funcall function)))))

(static-if (featurep 'xemacs)
    (defun mime-popup-menu-select (menu &optional event)
      (let ((selection (get-popup-menu-response menu event)))
	(event-object selection)))
  (defun mime-popup-menu-select (menu &optional event)
    (mime-popup-menu-bogus-filter-constructor menu)))


;;; @ Other Utility
;;;

(defvar mime-condition-type-alist
  '((preview . mime-preview-condition)
    (action . mime-acting-condition)))

(defvar mime-condition-mode-alist
  '((with-default . ctree-set-calist-with-default)
    (t . ctree-set-calist-strictly)))

(defun mime-add-condition (target-type condition &optional mode file)
  "Add CONDITION to database specified by TARGET-TYPE.
TARGET-TYPE must be 'preview or 'action.
If optional argument MODE is 'strict or nil (omitted), CONDITION is
added strictly.
If optional argument MODE is 'with-default, CONDITION is added with
default rule.
If optional argument FILE is specified, it is loaded when CONDITION is
activate."
  (let ((sym (cdr (assq target-type mime-condition-type-alist))))
    (if sym
	(let ((func (cdr (or (assq mode mime-condition-mode-alist)
			     (assq t mime-condition-mode-alist)))))
	  (if (fboundp func)
	      (progn
		(funcall func sym condition)
		(if file
		    (let ((method (cdr (assq 'method condition))))
		      (autoload method file))))
	    (error "Function for mode `%s' is not found." mode)))
      (error "Variable for target-type `%s' is not found." target-type))))


;;; @ end
;;;

(provide 'semi-def)

;;; semi-def.el ends here
