/* spo_idmef.c */
 
/*
** Copyright (C) 2000, 2001 Joe McAlerney, Silicon Defense, Inc. 
**                    <joey@silicondefense.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef ENABLE_IDMEF

#include "spo_idmef.h"
#include "idmefxml.h"

extern char *file_name;  /* this is the file name from rules.c, generally used
                            for error messages */

extern int file_line;    /* this is the file line number from rules.c that is
                            used to indicate file lines for error messages */

/* #define IDMEF_DEBUG */

void SetupIDMEFRule()
{
  /* map the keyword to an initialization/processing function */
  RegisterPlugin("idmef", IDMEFRuleInit);

#ifdef DEBUG
    printf("Plugin: IDMEFRule Initialized\n");
#endif
}

/****************************************************************************
 * 
 * Function: IDMEFRuleInit(char *, OptTreeNode *)
 *
 * Purpose: Parse the rule argument and attach it to the rule data struct, 
 *          then attach the detection function to the function list
 *
 * Arguments: data => rule arguments/data
 *            otn => pointer to the current rule option list node
 *
 * Returns: void function
 *
 ****************************************************************************/
void IDMEFRuleInit(char *data, OptTreeNode *otn, int protocol)
{
  /* allocate the data structure and attach it to the
     rule's data struct list */
  otn->ds_list[PLUGIN_IDMEF] = (IDMEFRuleData *) calloc(sizeof(IDMEFRuleData), sizeof(char));

  /* this is where the keyword arguments are processed and placed into the 
     rule option's data structure */
  ParseIDMEFRule(data, otn);
}


/*
 * Function: SetupIDMEF()
 *
 * Purpose: Registers the output plugin keyword and initialization 
 *          function into the output plugin list.  This is the function that
 *          gets called from InitOutputPlugins() in plugbase.c.
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupIDMEF()
{
    /* link the preprocessor keyword to the init function in 
       the preproc list */
    RegisterOutputPlugin("idmef", NT_OUTPUT_LOG, IDMEFInit);

#ifdef DEBUG
    printf("Output plugin: IDMEF is setup...\n");
#endif
}

/****************************************************************************
 *
 * Function: ParseIDMEFRule(char *)
 *
 * Purpose: Figure out what type of IDMEF message needs to be created
 *
 * Arguments: type => a char * representing the type of message to be created
 *
 * Returns: void function
 *
 ***************************************************************************/
void ParseIDMEFRule(char *type, OptTreeNode *otn_tmp)
{

  if (! type || ! (*type))
     return;
 
  ((IDMEFRuleData *) otn_tmp->ds_list[PLUGIN_IDMEF])->idmef_msg_type = 0;

  while (isspace((int)*type)) type++;

  if (!strcasecmp(type, "web"))
    ((IDMEFRuleData *) otn_tmp->ds_list[PLUGIN_IDMEF])->idmef_msg_type = IDMEF_WEB;
  else if (!strcasecmp(type, "overflow"))
    ((IDMEFRuleData *) otn_tmp->ds_list[PLUGIN_IDMEF])->idmef_msg_type = IDMEF_OVERFLOW;
  else if (!strcasecmp(type, "default"))
    ((IDMEFRuleData *) otn_tmp->ds_list[PLUGIN_IDMEF])->idmef_msg_type = IDMEF_DEFAULT;
  else
  {
    FatalError( "ERROR %s (%d): invalid IDMEF message type\n", file_name, file_line);
  }

}

/*
 * Function: IDMEFInit(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void IDMEFInit(u_char *args)
{
  xmlSubstituteEntitiesDefault(0);

  /* Initialize the IDMEFPluginSettings struct */
  ips.output = NULL;
  ips.logto = NULL;
  ips.dtd = NULL;
  ips.analyzerid = NULL;
  ips.category = NULL;
  ips.name = NULL;
  ips.location = NULL;
  ips.address = NULL;
  ips.netmask = NULL;
  ips.address_cat = NULL;

  ips.alert_id_fn = DEFAULT_ALERTID_FN;
  ips.indent = 0;
  ips.def.active = 1;
  ips.def.payload = PAYLOAD_NONE;
  ips.web.active = 1;
  ips.web.payload = PAYLOAD_NONE;
  ips.overflow.active = 1;
  ips.overflow.payload = PAYLOAD_NONE;
  ips.anomrep.active = 1;
  ips.anomrep.payload = PAYLOAD_NONE;
  ips.portscan.active = 1;
  ips.portscan.payload = PAYLOAD_NONE;

  /*  ips.spec = NULL;  input plugin supplied info */

  /* parse the argument list from the rules file */
  ParseIDMEFArgs(args);
 
  /* Sanity check the plugin settings */
  SanityCheckIDMEFArgs();

  /* Initialize some globals */
  idmef_alertid = 1;

  /* set up IDMEF XML lib globals */
  globalsInit(ips.dtd);

  /* Setup the output facility */
  if(!strncasecmp(ips.output,"log",3))
  {
     pv.log_plugin_active = 1;
     AddFuncToOutputList(IDMEF, NT_OUTPUT_LOG, NULL);
  }
  else
  {
     pv.alert_plugin_active = 1;
     AddFuncToOutputList(IDMEF, NT_OUTPUT_ALERT, NULL);
  }

  /* Open the log file if it was specified */
  idmef_log_file = fopen(ips.logto,"a");
  if (!idmef_log_file) 
      FatalError("log_idmef: IDMEF log file open error (%s)\n", ips.logto);
  

  /* attempt to retrieve a stored alert id */
  idmef_alertid = getStoredAlertID(ips.alert_id_fn);
  if(idmef_alertid == 0)
     FatalError("idmef: Error retreiving the stored alert id\n");
  else if(idmef_alertid == 1)
    fprintf(stderr,"idmef: No stored alert id.  Continuing with alert id = 1\n");

  AddFuncToCleanExitList(IDMEFCleanExit, NULL);
  AddFuncToRestartList(IDMEFCleanExit, NULL);
}

/*
 * Function: ParseIDMEFArgs(char *)
 *
 * Purpose: Process the preprocessor arguements from the rules file and 
 *          initialize the preprocessor's data struct.
 *
 * Arguments: args => argument list
 *            We expect args to be in the format:
 *
 *            key1=valueA1[,valueB1,valueC1,...] key2=valueA2[,valueB2, \
 *            valueC2,...] key3=valueA3[,valueB3,valueC3,...] ...
 *                          
 * Returns: void function
 *
 * Details: The argument string is broken down into seperate key/list pairs.
 *          The key/lists are held together with a pointer to char *'s.
 *          Each one is fed into yet another parsing function.  The structure
 *          came about as a result of the limitations of strtok. I'm rather 
 *          proud of the way this works, and was happy to find that it did.
 */
void ParseIDMEFArgs(char *args)
{
    int i, numToks;
    char **toks;

    toks = mSplit(args, " ", MAX_PLUGIN_ARGUMENT_COUNT, &numToks, '\\');

    ParseIP(toks[0], &ips.homenet);     /* grab the homenet */

    for(i=1; i < numToks; i++)          /* parse each one */
    {
      ParseIndividualArgs(toks[i]);
    }
}

/*
 * Function: ParseIndividualArgs(char *)
 *
 * Purpose: Process the individual key/list arguments passed into the
 *          preprocessor.
 *
 * Arguments: args => argument list
 *            We expect args to be in the format:
 *
 *            name1=valueA1[,valueB1,valueC1,...]
 *
 * Returns: void function
 */
void ParseIndividualArgs(char *args)
{
    char *key, *val;
 
    key = strtok(args, "=");
    val = strtok(NULL, ",");

    if(!strncasecmp(key,"output",6)) /* 0 means a match */
    {
       if(strlen(val) > 0)  /* validate our output mechanism */
       {
	 if(strncasecmp(val,"log",3))
	   if(strncasecmp(val,"alert",5))
	   { 
	     FatalError("IDMEF: Invalid output facility, use \"log\" or \"alrt\"\n");
           }

       ips.output = val;
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDMEF output facility = %s\n", ips.output);
#endif
       }
       else
	 FatalError("IDMEF: IDMEF output facility not specified\n");
    }
    else if(!strncasecmp(key,"logto",5))
    {
       ips.logto = val;
       if(strlen(ips.logto) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDMEF log file = %s\n", ips.logto);
#endif
       }
       else
	 FatalError("IDMEF: IDMEF log file not specified\n");
    }
    else if(!strncasecmp(key,"dtd",3))
    {
       ips.dtd = val;
       if(strlen(ips.dtd) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDMEF XML dtd = %s\n", ips.dtd);
#endif
       }
       else
	 FatalError("IDMEF: IDMEF XML dtd file not specified\n");
    }
    else if(!strncasecmp(key,"analyzerid",10))
    {
       ips.analyzerid = val;
       if(strlen(ips.analyzerid) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDMEF analyzerid = %s\n", ips.analyzerid);
#endif
       }
       else
	 FatalError("IDMEF: IDMEF Analyzer ID not specified\n");
    }
    else if(!strncasecmp(key,"category",8))
    {
       ips.category = val;
       if(strlen(ips.category) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS category = %s\n", ips.category);
#endif
       }
       else
	 ips.category = NULL;
    }
    else if(!strncasecmp(key,"name",4))
    {
       ips.name = val;
       if(strlen(ips.name) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS name = %s\n", ips.name);
#endif
       }
       else
	 ips.name = NULL;   /* we'll have to use getenv(hostname) */
    }
    else if(!strncasecmp(key,"location",8))
    {
       ips.location = val;
       if(strlen(ips.location) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS location = %s\n", ips.location);
#endif
       }
       else
	 ips.location = NULL;
    }
    else if(!strncasecmp(key,"address_cat",11))
    {
       ips.address_cat = val;
       if(strlen(ips.address_cat) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS address_cat = %s\n", ips.address_cat);
#endif
       }
       else
	 ips.address_cat = NULL;
    }
    else if(!strncasecmp(key,"address",7))
    {
       ips.address = val;
       if(strlen(ips.address) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS address = %s\n", ips.address);
#endif
       }
       else
	 ips.address = NULL;
    }
    else if(!strncasecmp(key,"netmask",7))
    {
       ips.netmask = val;
       if(strlen(ips.netmask) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS netmask = %s\n", ips.netmask);
#endif
       }
       else
	 ips.netmask = NULL;
    }
    else if(!strncasecmp(key,"homenet_cat",11))
    {
       ips.homenet_cat = val;
       if(strlen(ips.homenet_cat) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS homenet_cat = %s\n", ips.homenet_cat);
#endif
       }
       else
	 ips.homenet_cat = NULL;
    }
    else if(!strncasecmp(key,"homenet_loc",11))
    {
       ips.homenet_loc = val;
       if(strlen(ips.homenet_loc) > 0)
       {
#ifdef IDMEF_DEBUG
 	 printf("IDMEF: IDS homenet_loc = %s\n", ips.homenet_loc);
#endif
       }
       else
	 ips.homenet_loc = NULL;
    }
    else if(!strncasecmp(key,"default",7)) /* remember, 0 means a match */
    {
        while(val != NULL)
	{
	   if(!strncasecmp(val,"disable",7))
	   {
	      ips.def.active = 0;
#ifdef IDMEF_DEBUG
	      printf("IDMEF: default msg's disabled\n");
#endif
           }
	   else if(!strncasecmp(val,"hex",3))
	   {
	      ips.def.payload = PAYLOAD_HEX;
#ifdef IDMEF_DEBUG
	      printf("IDMEF: log default msg payload in hex\n");
#endif
	   }
	   else if(!strncasecmp(val,"base64",6))
	   {
	      ips.def.payload = PAYLOAD_BASE64;
#ifdef IDMEF_DEBUG
	      printf("IDMEF: log default msg payload in base64\n");
#endif
           }
	   else if(!strncasecmp(val,"ascii",5))
	   {
	      ips.def.payload = PAYLOAD_ASCII;
#ifdef IDMEF_DEBUG
	      printf("IDMEF: log default msg payload in ascii\n");
#endif
           }
           else
 	      FatalError("IDMEF: Invalid value for \"default\": %s\n",val);
           val = strtok(NULL, ",");
        }
    }
    else if(!strncasecmp(key,"web",3)) /* remember, 0 means a match */
    {
        while(val != NULL)
	{
	   if(!strncasecmp(val,"disable",7))
	   {
	      ips.web.active = 0;
#ifdef IDMEF_DEBUG
	      printf("IDMEF: web msg's disabled\n");
#endif
           }
	   else if(!strncasecmp(val,"hex",3))
	   {
	      ips.web.payload = PAYLOAD_HEX;
#ifdef IDMEF_DEBUG      
              printf("IDMEF: log web msg payload in hex\n");
#endif
	   }
	   else if(!strncasecmp(val,"base64",6))
	   {
	      ips.web.payload = PAYLOAD_BASE64;
#ifdef IDMEF_DEBUG
              printf("IDMEF: log web msg payload in base64\n");
#endif
           }
	   else if(!strncasecmp(val,"ascii",5))
	   {
	      ips.web.payload = PAYLOAD_ASCII;
#ifdef IDMEF_DEBUG
              printf("IDMEF: log web msg payload in ascii\n");
#endif
           }
           else
 	      FatalError("IDMEF: Invalid value for \"web\": %s\n",val);
           val = strtok(NULL, ",");
        }
     }
    else if(!strncasecmp(key,"overflow",8)) /* remember, 0 means a match */
    {
        while(val != NULL)
	{
	   if(!strncasecmp(val,"disable",7))
           {
	      ips.overflow.active = 0;
#ifdef IDMEF_DEBUG
              printf("IDMEF: overflow msg's disabled\n");
#endif
           }
	   else if(!strncasecmp(val,"hex",3))
	   {
	      ips.overflow.payload = PAYLOAD_HEX;
#ifdef IDMEF_DEBUG
              printf("IDMEF: log overflow msg payload in hex\n");
#endif
	   }
	   else if(!strncasecmp(val,"base64",6))
	   {
	      ips.overflow.payload = PAYLOAD_BASE64;
#ifdef IDMEF_DEBUG 
              printf("IDMEF: log overflow msg payload in base64\n");
#endif
           }
	   else if(!strncasecmp(val,"ascii",5))
	   {
	      ips.overflow.payload = PAYLOAD_ASCII;
#ifdef IDMEF_DEBUG
              printf("IDMEF: log overflow msg payload in ascii\n");
#endif
           }
           else
 	      FatalError("IDMEF: Invalid value for \"overflow\": %s\n",val);
           val = strtok(NULL, ",");
	}
    }
    else if(!strncasecmp(key,"anomrep",7)) /* remember, 0 means a match */
    {
        while(val != NULL)
	{
	     if(!strncasecmp(val,"disable",7))
	     {
	        ips.anomrep.active = 0;
#ifdef IDMEF_DEBUG
		printf("IDMEF: anomrep msg's disabled\n");
#endif
	     }
	     else if(!strncasecmp(val,"hex",3))
	     {
	        ips.anomrep.payload = PAYLOAD_HEX;
#ifdef IDMEF_DEBUG		
		printf("IDMEF: log anomrep msg payload in hex\n");
#endif
	     }
	     else if(!strncasecmp(val,"base64",6))
	     {
	        ips.anomrep.payload = PAYLOAD_BASE64;
#ifdef IDMEF_DEBUG
		printf("IDMEF: log anomrep msg payload in base64\n");
#endif
	     }
	     else if(!strncasecmp(val,"ascii",5))
	     {
	        ips.anomrep.payload = PAYLOAD_ASCII;
#ifdef IDMEF_DEBUG
		printf("IDMEF: log anomrep msg payload in ascii\n");
#endif
	     }
	     else
	        FatalError("IDMEF: Invalid value for \"anomrep\": %s\n",val);
	 
             val = strtok(NULL, ",");
        }
    }
    else if(!strncasecmp(key,"portscan",8)) /* remember, 0 means a match */
    {
        while(val != NULL)
	{
	     if(!strncasecmp(val,"disable",7))
	     {
	        ips.portscan.active = 0;
#ifdef IDMEF_DEBUG
		printf("IDMEF: portscan msg's disabled\n");
#endif
	     }
	     else if(!strncasecmp(val,"hex",3))
	     {
	        ips.portscan.payload = PAYLOAD_HEX;
#ifdef IDMEF_DEBUG		
		printf("IDMEF: log portscan msg payload in hex\n");
#endif
	     }
	     else if(!strncasecmp(val,"base64",6))
	     {
	        ips.portscan.payload = PAYLOAD_BASE64;
#ifdef IDMEF_DEBUG
		printf("IDMEF: log portscan msg payload in base64\n");
#endif
	     }
	     else if(!strncasecmp(val,"ascii",5))
	     {
	        ips.portscan.payload = PAYLOAD_ASCII;
#ifdef IDMEF_DEBUG
		printf("IDMEF: log portscan msg payload in ascii\n");
#endif
	     }
	     else
	        FatalError("IDMEF: Invalid value for \"portscan\": %s\n",val);
	 
             val = strtok(NULL, ",");
        }
    }
    else if(!strncasecmp(key,"indent",6))
    {
       if(!strncasecmp(val,"true",4))
       {
	 xmlKeepBlanksDefault(0);
	 ips.indent = 1;
#ifdef IDMEF_DEBUG
         printf("IDMEF: Indented output: true\n");
#endif
       }
    }
    else if(!strncasecmp(key,"alert_id",8))
    {
       ips.alert_id_fn = val;
       if(strlen(ips.alert_id_fn) > 0)
       {
#ifdef IDMEF_DEBUG
         printf("IDMEF: IDS alert_id file = %s\n", ips.alert_id_fn);
#endif
       }
       else
       {
         ips.alert_id_fn = DEFAULT_ALERTID_FN;
#ifdef IDMEF_DEBUG
         printf("IDMEF: Defaulting to IDS alert_id file = %s\n", ips.alert_id_fn);
#endif
       }
    } 
}

/*
 * Function: SanityCheckIDMEFArgs()
 *
 * Purpose: Checks to see if the arguments given to the idmef plugin
 *          were valid in themselves, and valid with eachother.
 *
 * Returns: void function
 */

void SanityCheckIDMEFArgs()
{
  if(ips.output == NULL)
    FatalError("IDMEF: You must supply an output facility in the output plugin arguments using the syntax \"output=[log,alert]\"\n");

  if(ips.logto == NULL)
    FatalError("IDMEF: You must supply a file to log IDMEF alerts to in the output plugin arguments using the syntax \"logto=<filename>\"\n");

  if(ips.dtd == NULL)
    FatalError("IDMEF: You must supply the location of the IDMEF XML dtd in the output plugin arguments using the syntax\" dtd=<filename>\"\n");

  if(ips.analyzerid == NULL)
    FatalError("IDMEF: You must supply a unique analyzer name in the output plugin arguments for this IDS using the syntax \"analyzerid=<somename>\"\n");

  if(ips.address_cat == NULL)
  {
     if(ips.address != NULL)
     {
        FatalError("IDMEF: You must supply an address category argument to the idmef output plugin in the format \"address_cat=<somecategory>\" if you want to specify an address.\n");
     }
     else if(ips.netmask != NULL)
      FatalError("IDMEF: You must supply an address category and an address argument to the idmef output plugin if you want to specify a netmask.  The valid syntax is \"address_cat=<somecategory>\" and \"address=<someaddress>\" respectively.\n");
  }
  else
  {
    if(ips.address == NULL)
    {
      FatalError("IDMEF: You must supply an address argument to the idmef output plugin in the format \"address=<someaddress>\" if you want to specify an address category (address_cat)\n");
    }
  }

}

/*
 * Function: IDMEF(Packet *, char *, void *, input_id, output_msg_info **)
 *
 * Purpose: This is the main preprocessor function.  It builds an IDMEF
 *          message with the packet that set off an IDMEF rule.
 *
 * Arguments: p   => The Packet
 *            msg => The msg attribute from the rule that was set off
 *            arg => Something we don't use
 *
 * Returns: void function
 */

void IDMEF(Packet *p, char *msg, void *arg, Event *event)
{

    /* We won't make an IDMEF message if the msg field is NULL */
    if(msg == NULL) 
       return;

    /* Heck, same for event at this point too */
    if(event == NULL)
      return;

    switch(event->sig_generator)
    {
       case GENERATOR_SNORT_ENGINE:
          BuildRuleMessage(p, msg);
          break;
       case GENERATOR_SPP_PORTSCAN:
          BuildPortscanMessage(msg);
          break;
       case GENERATOR_SPP_SPADE:
 	  BuildSpadeMessage(p, msg);
          break;    
       default:
#ifdef IDMEF_DEBUG
       fprintf(stderr, "IDMEF(): Unknown caller type, returning\n");
#endif
    }

    return;
}

/*
 * Function: BuildRuleMessage(Packet *p, char *msg)
 *
 * Purpose: Constructs the IDMEF message, calling upon henchmen functions
 *          to do most of the dirty work.
 *
 * Arguments:  p =>    The current packet
 *             msg =>  The msg attribute from the rule that was set off  
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 */

void BuildRuleMessage(Packet *p, char *msg)
{
    xmlNodePtr idmef_msg;
    IDMEFMsgOptions *cur = NULL;
    char *payload;
    idmef_output_spec_msg spec;

    if(p == NULL)
	 return;

    if(otn_tmp->ds_list[PLUGIN_IDMEF] == NULL)
    {
#ifdef IDMEF_DEBUG
 	 fprintf(stderr,"IDMEF(): not an IDMEF rule, returning\n");
#endif
       return;
    }

    /* We are not expecting anything specifying output from the rules engine.
       Use the default text_file output, and to the location specified by the
       "logto" value in the plugin arguments  */
       
    spec.outtype = text_file;
    spec.outdet.logfile = idmef_log_file;

    if (!(otn_tmp && ((IDMEFRuleData *) otn_tmp->ds_list[PLUGIN_IDMEF])->idmef_msg_type))
    {
#ifdef IDMEF_DEBUG
 	 fprintf(stderr,"BuildRuleMessage(): IDMEF message information not available\n");
#endif
         return;
    }
    
    /* Determine which message type this is.  If the message type
       is inactive, just return. */

    switch(((IDMEFRuleData *) otn_tmp->ds_list[PLUGIN_IDMEF])->idmef_msg_type)
    {
       case IDMEF_DEFAULT:
	  if(!ips.def.active)
	    return;
	  cur = &ips.def;
          break;
       case IDMEF_WEB:
          if(!ips.web.active)
            return;
	  cur = &ips.web;
	  break;
       case IDMEF_OVERFLOW:
	  if(!ips.overflow.active)
	    return;
	  cur = &ips.overflow;
	  break;
       default:              /* we should never get here */
	  FatalError("IDMEF: Invalid message type\n");
    }

    /* Format the payload if specified */

    payload = NULL;

    switch(cur->payload)
    {
	case PAYLOAD_NONE:
	   payload = NULL;
	   break;
        case PAYLOAD_HEX:
	   payload = hex(p->data, p->dsize);
	   break;
        case PAYLOAD_BASE64:
	   payload = base64(p->data, p->dsize);
	   break;
        case PAYLOAD_ASCII:
	   payload = ascii(p->data, p->dsize);
	   break;
        default:
	   payload = NULL;
    }

    /* Build the IDMEF message */

    idmef_msg = BuildRuleMessageTree(cur, p, payload);


   /****************************** - RULES - ****************************
    *                                                                   *
    *      Output the message according to rule specifications          *
    *                                                                   *
    *********************************************************************/

    if(idmef_msg != NULL)
    {
#ifdef IDMEF_DEBUG
       validateCurrentDoc();
#endif
       outputMessage(spec);   /* output the current document with the 
				 specified facility */
    }
    else
    {
       /* The message was not made correctly, but lets not exit. */
       FreeGlobals(p, payload);  
       return;
    }
    incAndFlush(); /* increment alert ID, and flush the log file */
    FreeGlobalsAndMessage(p, payload);  

    return;
}

/*
 * Function: BuildPortscanMessage(Packet *p, char *msg)
 *
 * Purpose: Constructs the IDMEF message, calling upon henchmen functions
 *          to do most of the dirty work.
 *
 * Arguments:  p =>    The current packet
 *             msg =>  The msg attribute from the rule that was set off  
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 */

void BuildPortscanMessage(char *msg)
{
   xmlNodePtr idmef_msg;
   IDMEFMsgOptions *cur = NULL;
   idmef_output_spec_msg spec;

   if(!ips.portscan.active) /* is this disabled? */
      return;
   cur = &ips.portscan; 

   spec.outtype = text_file;
   spec.outdet.logfile = idmef_log_file;

   idmef_msg = BuildPortscanMessageTree(cur, msg);

   /* Do more checking here */

   if(idmef_msg != NULL)
   {
#ifdef IDMEF_DEBUG
      validateCurrentDoc();
#endif
      outputMessage(spec);
   }
   else
   {
     /* The message was not made correctly, but lets not exit. */
      fprintf(stderr,"The IDMEF message was returned null\n");
      FreeGlobals(NULL, NULL);  
      return;
   }
   incAndFlush(); /* increment alert ID, and flush the log file */
   FreeGlobalsAndMessage(NULL, NULL);

   return;
}

/*
 * Function: BuildSpadeMessage(Packet *p, char *msg)
 *
 * Purpose: Constructs the IDMEF message, calling upon henchmen functions
 *          to do most of the dirty work.
 *
 * Arguments:  p =>      The current packet
 *             msg =>    The default message sent by SPADE  
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 */

void BuildSpadeMessage(Packet *p, char *msg)
{
   xmlNodePtr idmef_msg;
   IDMEFMsgOptions *cur = NULL;
   idmef_output_spec_msg spec;
   /* 
     extra_fields *extra= NULL;
     int i;
   */

   if(p == NULL)
	return;

   if(!ips.anomrep.active) /* is this disabled? */
      return;
   cur = &ips.anomrep; 

   spec.outtype = text_file;
   spec.outdet.logfile = idmef_log_file;

   /* Build the IDMEF message */
   idmef_msg = BuildSpadeMessageTree(cur, p, msg, NULL);

   /*********************************************************************
    *                                                                   *
    *      Output the message according to Spade's specifications       *
    *                                                                   *
    *********************************************************************/

   if(idmef_msg != NULL)
   {
#ifdef IDMEF_DEBUG
   	validateCurrentDoc();
#endif
        outputMessage(spec);   /* output the current document with the 
                                  specified facility */
   }
   else
   {
        /* The message was not made correctly, but lets not exit. */
        fprintf(stderr,"The IDMEF message was returned null\n");
	FreeGlobals(p, NULL);  
	return;
   }
   incAndFlush(); /* increment alert ID, and flush the log file */
   FreeGlobalsAndMessage(p, NULL);  

   return;

}


/*
 * Function: BuildRuleMessageTree(char *)
 *
 * Purpose: Constructs the IDMEF message, calling upon henchmen functions
 *          to do most of the dirty work.
 *
 * Arguments: cur =>     The current IDMEF message options
 *            p =>       The current packet
 *            payload => The packet payload
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 *
 * Note: This is sort of messy right now, because libidmef does not sort
 *       subelements when they are inserted into elements.  Consequently,
 *       we are left doing a procedural programming excercise.  If we go
 *       ahead and build stuff out of order, the XML message will not validate
 *       against the DTD.
 */

xmlNodePtr BuildRuleMessageTree(IDMEFMsgOptions *cur, Packet *p, char *payload)
{
   xmlNodePtr theMessage, theAnalyzer, theSource, theTarget;
   ReferenceData *rd;
   
   int have_classification = 0;  /* flag to keep track of required element */

   if(p == NULL)
     return NULL;

   /* Initialize the XML doc */
   if(!createCurrentDoc( XML_VERSION))
   {
      fprintf(stderr,"ERROR: spo_idmef.c: createCurrentDoc returned 0\n");
      return NULL;
   }
      
   /****************************** - RULES - ****************************
    *                                                                   *
    *          Initialize values to be used in the XML tree             *
    *                                                                   *
    *********************************************************************/

   /* convert ports into strings */

   if((p->iph->ip_proto == IPPROTO_TCP) || 
      (p->iph->ip_proto == IPPROTO_UDP))
   {
      sport = calloc(MAX_PORT_DIGITS,sizeof(char));
      dport = calloc(MAX_PORT_DIGITS,sizeof(char));

      sprintf(sport,"%d",p->sp);
      sprintf(dport,"%d",p->dp);
   }
    
   idmef_alertid_str = intToString(idmef_alertid);

   /****************************** - RULES - ****************************
    *                                                                   *
    *          Start constructing the IDMEF XML message                 *
    *                                                                   *
    *********************************************************************/

   if((theAnalyzer = BuildAnalyzer(p)) == NULL)
   {
#ifdef IDMEF_DEBUG
      fprintf(stderr,"IDMEF: Problem building Analyzer node\n");
#endif
      return NULL;
   }

   if((theSource = BuildSource(p)) == NULL)
   {
#ifdef IDMEF_DEBUG
      fprintf(stderr,"IDMEF: Problem building Source node\n");
#endif
      return NULL;
   }

   if((theTarget = BuildTarget(p)) == NULL)
   {
#ifdef IDMEF_DEBUG
      fprintf(stderr,"IDMEF: Problem building Target node\n");
#endif
      return NULL;
   }

  /* Build as much as we can, up to the Target */

  theMessage = newIDMEF_Message(
     newAttribute("version",IDMEF_MESSAGE_VERSION),
     newAlert(
        newSimpleElement("ident",idmef_alertid_str),
	newSimpleElement("impact","unknown"),
	theAnalyzer,
        newCreateTime(NULL),                /* Time values set here */
        theSource,
        theTarget,
        NULL
     ),
     NULL
  );

  /* Extract reference information for Classification url's.  We may end up
     with more than one Classification, hence it gets it's own section, and 
     destroys the opportunity to build our message in one fell swoop. */

  for(rd = (ReferenceData *)otn_tmp->ds_list[PLUGIN_REFERENCE_NUMBER];
      rd != NULL; rd = rd->next)
  {
     if(rd->url == NULL)  /* we are required to show something*/
        rd->url = NO_URL;

     if(strncasecmp(rd->system, "bugtraq", 7) == 0)
     {
        /* Add Bugtraq information to this message if it exists */
        addElement(getElement(theMessage,"Alert"),
                   newClassification(
		      newAttribute("origin","bugtraqid"),
		      newSimpleElement("name",otn_tmp->message),
		      newSimpleElement("url",rd->url),
		      NULL)
                   );
     }
     else if(strncasecmp(rd->system, "cve", 3) == 0)
     {
        addElement(getElement(theMessage,"Alert"),
                   newClassification(
	   	      newAttribute("origin","cve"),
		      newSimpleElement("name",otn_tmp->message),
		      newSimpleElement("url",rd->url),
		   NULL)
                );
     }
     else
     {
        addElement(getElement(theMessage,"Alert"),
                   newClassification(
	   	      newAttribute("origin","vendor-specific"),
		      newSimpleElement("name",otn_tmp->message),
		      newSimpleElement("url",rd->url),
		   NULL)
                );
     }
     have_classification = 1;
  }

  if(!have_classification)
  {
      addElement(getElement(theMessage,"Alert"),
                   newClassification(
		      newSimpleElement("name",otn_tmp->message),
		      newSimpleElement("url",NO_URL),
		   NULL)
                );
  }
  have_classification = 1;

  /* Add newAdditionalData to include the Packet Payload if it exists */
  if(payload != NULL)
  {
    addElement(getElement(theMessage,"Alert"),
	        newAdditionalData(
		   newAttribute("meaning","Packet Payload"),
		   newAttribute("type","string"),
		   newSimpleElement("value",payload),
		   NULL)
               );
  }

  return theMessage;
}


/*
 * Function: BuildPortscanMessageTree(IDMEFMsgOptions *cur, char *msg)
 *
 * Purpose: Constructs the IDMEF message, calling upon henchmen functions
 *          to do most of the dirty work.
 *
 * Arguments: cur =>     The current IDMEF message options
 *            msg =>     The Default message provided by SPADE
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 *
 * Note: This is sort of messy right now, because libidmef does not sort
 *       subelements when they are inserted into elements.  Consequently,
 *       we are left doing a procedural programming excercise.  If we go
 *       ahead and build stuff out of order, the XML message will not validate
 *       against the DTD.
 */

xmlNodePtr BuildPortscanMessageTree(IDMEFMsgOptions *cur, char *msg)
{
   xmlNodePtr theMessage, theAnalyzer;
    
   if(msg == NULL)
     return NULL;     

   /* Initialize the XML doc */
   if(!createCurrentDoc( XML_VERSION))
   {
      fprintf(stderr,"ERROR: spo_idmef.c: createCurrentDoc returned 0\n");
      return NULL;
   }

   /************************* - PORTSCAN - ******************************
    *                                                                   *
    *          Initialize values to be used in the XML tree             *
    *                                                                   *
    *********************************************************************/

   idmef_alertid_str = intToString(idmef_alertid);

   /************************* - PORTSCAN - ******************************
    *                                                                   *
    *          Start constructing the IDMEF XML message                 *
    *                                                                   *
    *********************************************************************/

   if((theAnalyzer = BuildAnalyzer(NULL)) == NULL)
   {
#ifdef IDMEF_DEBUG
      fprintf(stderr,"IDMEF: Problem building Analyzer node\n");
#endif
      return NULL;
   }

   /* Build the Portscan message  - This message will contain more information
      when we can pick the fruit of the portscan plugin */

   theMessage = newIDMEF_Message(
       newSimpleElement("version",IDMEF_MESSAGE_VERSION),
       newAlert(
         newSimpleElement("ident",idmef_alertid_str),
  	 newSimpleElement("impact","unknown"),
	 theAnalyzer,
         newCreateTime(NULL),
         newClassification(
            newAttribute("origin","vendor-specific"),
            newSimpleElement("name",msg),
            newSimpleElement("url",PORTSCAN_CLASSIFICATION_URL),
	    NULL
         ),
         NULL
      ),
      NULL
   );

   return theMessage;
}

/*
 * Function: BuildSpadeMessageTree(IDMEFMsgOptions *cur,
 *                                 Packet *p,
 *                                 char *msg,
 *                                 extra_fields *extra)
 *
 * Purpose: Constructs the IDMEF message, calling upon henchmen functions
 *          to do most of the dirty work.
 *
 * Arguments: cur =>     The current IDMEF message options
 *            p =>       The current packet
 *            msg =>     The Default message provided by SPADE
 *
 * Proposed Arguments (not currently used):
 *
 *            extra =>   extra fields provided by the SPADE plugin
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 *
 * Note: This is sort of messy right now, because libidmef does not sort
 *       subelements when they are inserted into elements.  Consequently,
 *       we are left doing a procedural programming excercise.  If we go
 *       ahead and build stuff out of order, the XML message will not validate
 *       against the DTD.
 */

xmlNodePtr BuildSpadeMessageTree(IDMEFMsgOptions *cur, Packet *p, char *msg, void *extra)
{
   xmlNodePtr theMessage, theAnalyzer, theSource, theTarget;
    
   /* Spade must be sending us threshold adjusting information.  We are not
      interested in this. */
   if(p == NULL)
     return NULL;

   /* Initialize the XML doc */
   if(!createCurrentDoc( XML_VERSION))
   {
      fprintf(stderr,"ERROR: spo_idmef.c: createCurrentDoc returned 0\n");
      return NULL;
   }

   /*************************** - SPADE - *******************************
    *                                                                   *
    *          Initialize values to be used in the XML tree             *
    *                                                                   *
    *********************************************************************/

   if(msg == NULL)
     msg = SPADE_CLASSIFICATION_NAME;

   /* convert ports into strings */
   if((p->iph->ip_proto == IPPROTO_TCP) || 
      (p->iph->ip_proto == IPPROTO_UDP))
   {
      sport = calloc(MAX_PORT_DIGITS,sizeof(char));
      dport = calloc(MAX_PORT_DIGITS,sizeof(char));

      sprintf(sport,"%d",p->sp);
      sprintf(dport,"%d",p->dp);
   }

   idmef_alertid_str = intToString(idmef_alertid);

   /*************************** - SPADE - *******************************
    *                                                                   *
    *          Start constructing the IDMEF XML message                 *
    *                                                                   *
    *********************************************************************/

   if((theAnalyzer = BuildAnalyzer(p)) == NULL)
   {
#ifdef IDMEF_DEBUG
      fprintf(stderr,"IDMEF: Problem building Analyzer node\n");
#endif
      return NULL;
   }

   if((theSource = BuildSource(p)) == NULL)
   {
#ifdef IDMEF_DEBUG
      fprintf(stderr,"IDMEF: Problem building Source node\n");
#endif
      return NULL;
   }

   if((theTarget = BuildTarget(p)) == NULL)
   {
#ifdef IDMEF_DEBUG
      fprintf(stderr,"IDMEF: Problem building Target node\n");
#endif
      return NULL;
   }

   /* Build as much as we can, up to the Analyzer */

   theMessage = newIDMEF_Message(
       newSimpleElement("version",IDMEF_MESSAGE_VERSION),
       newAlert(
         newSimpleElement("ident",idmef_alertid_str),
	 newSimpleElement("impact","unknown"),
	 theAnalyzer,
         newCreateTime(NULL),
         theSource,
         theTarget,
         newClassification(
            newAttribute("origin","vendor-specific"),
            newSimpleElement("name",msg),
            newSimpleElement("url",SPADE_CLASSIFICATION_URL),
	    NULL
         ),
         NULL
      ),
      NULL
   );

   /* Add newAdditionalData to include the anomaly score if appropriate */
   /* NOTE: We're probably going to use the Event structure to help us
            do this eventually. */
   /*
   if(extra != NULL)
   {
	char score[20];
	snprintf(score,20,"%.5f",extra->spade.anomscore);
	addElement(getElement(theMessage,"Alert"),
				newAdditionalData(
				   newAttribute("meaning","anomaly score"),
				   newAttribute("type","real"),
				   newSimpleElement("value",score),
				   NULL)
			   );
   }
   */

   return theMessage;
}

/*
 * Function: BuildAnalyzer(Packet *)
 *
 * Purpose: Constructs the Analyzer node for the IDMEF message.
 *
 * Arguments: p =>       The current packet
 *
 * Returns: An xmlNodePtr to the Analyzer node.
 */

xmlNodePtr BuildAnalyzer(Packet *p)
{
  xmlNodePtr theAnalyzer, theNode, theAddress;

  theAnalyzer = newAnalyzer(
	           newSimpleElement("analyzerid",ips.analyzerid),
                   NULL
		);

  theNode = newNode(NULL);
  theAddress = newAddress(NULL);

  /* Add Analyzer category, name and location if available.  If the user
     did not supply a name for the Analyzer, we use the HOSTNAME environment
     variable */

  if(ips.category != NULL)
     setAttribute(theNode,newAttribute("category",ips.category));

  if(ips.location != NULL)
     addElement(theNode,newSimpleElement("location",ips.location));

  if(ips.name != NULL)
     addElement(theNode,newSimpleElement("name",ips.name));
  else
     addElement(theNode,newSimpleElement("name",getenv("HOSTNAME")));


  /* Add Address information if available */
  if(ips.address_cat != NULL)
     setAttribute(theAddress, newAttribute("category",ips.address_cat));

  if(ips.address != NULL)
     addElement(theAddress, newSimpleElement("address",ips.address));
 
  if(ips.netmask != NULL)
     addElement(theAddress, newSimpleElement("netmask",ips.netmask));

  /* Put things together */
  if(theAddress->children != NULL)  /* did we add a sub element to Address? */
    addElement(theNode, theAddress);
  else
    xmlFreeNode(theAddress);

  if(theNode->children != NULL)     /* did we add a sub element to Node? */
    addElement(theAnalyzer, theNode);
  else
    xmlFreeNode(theNode);

  return theAnalyzer;
}

/*
 * Function: BuildSource(Packet *)
 *
 * Purpose: Constructs the Source node for the IDMEF message.
 *
 * Arguments: p => The current packet
 *
 * Returns: An xmlNodePtr to the Source node.
 */

xmlNodePtr BuildSource(Packet *p)
{
  xmlNodePtr theSource, theAddress, theNode;

  theSource = newSource(NULL);

  theNode = newNode(NULL);
  
  theAddress =  newAddress(
                   newSimpleElement("category","ipv4-addr"),
                   newSimpleElement("address",inet_ntoa(p->iph->ip_src)),
                   NULL
	           );

  /* Add Homenet specific stuff.  If the target is in our homenet, include
     network category and physical location information if they exist. */

  if ((p->iph->ip_src.s_addr & ips.homenet.netmask) == ips.homenet.ip_addr)
  {
    /* User defined values for homenet category and location */

     if(ips.homenet_cat != NULL)
        setAttribute(theNode,newAttribute("category",ips.homenet_cat));

     if(ips.homenet_loc != NULL)
        addElement(theNode,newSimpleElement("location",ips.homenet_loc));
  }

 /* Since the order in which we add things matters, we add the Address last. */
  addElement(theNode,theAddress);

  /* ... and finally, the Node to the Source */
  addElement(theSource,theNode);
  
  /* We need to add a Service element to include port information if this
     was a TCP or UDP packet */

  if((p->iph->ip_proto == IPPROTO_TCP) || (p->iph->ip_proto == IPPROTO_UDP))
  {
     addElement(theSource,
                newService(
                     newSimpleElement("port",sport),
                     NULL)
                );
  }

  return theSource;
}

/*
 * Function: BuildTarget(Packet *)
 *
 * Purpose: Constructs the Target node for the IDMEF message.
 *
 * Arguments: p => The current packet
 *
 * Returns: An xmlNodePtr to the Target node.
 */

xmlNodePtr BuildTarget(Packet *p)
{
  xmlNodePtr theTarget, theAddress, theNode;

  theTarget = newTarget(NULL);
  
  theNode = newNode(NULL);

  theAddress =  newAddress(
                   newSimpleElement("category","ipv4-addr"),
                   newSimpleElement("address",inet_ntoa(p->iph->ip_dst)),
                   NULL
	           );

  /* Add Homenet specific stuff.  If the target is in our homenet, include
     network category and physical location information if they exist. */

  if ((p->iph->ip_dst.s_addr & ips.homenet.netmask) == ips.homenet.ip_addr)
  {
    /* User defined values for homenet category and location */

     if(ips.homenet_cat != NULL)
        setAttribute(theNode,newAttribute("category",ips.homenet_cat));

     if(ips.homenet_loc != NULL)
        addElement(theNode,newSimpleElement("location",ips.homenet_loc));
  }

 /* Since the order in which we add things matters, we add the Address last. */
  addElement(theNode,theAddress);

  /* ... and finally, the Node to the Target */
  addElement(theTarget,theNode);
 
  /* We need to add a Service element to include port information if this
     was a TCP or UDP packet */

  if((p->iph->ip_proto == IPPROTO_TCP) || (p->iph->ip_proto == IPPROTO_UDP))
  {
     addElement(theTarget,
                newService(
                     newSimpleElement("port",dport),
                     NULL)
                );
  }

  return theTarget;
}

void outputMessage(idmef_output_spec_msg spec)
{
   if (spec.outtype == text_file) 
   {
     /*       printf("IDMEF: writing to a file\n"); */
       printCurrentMessage(spec.outdet.logfile);
   } 
   else  // outtype == tcp_conn
   { 
       printf("IDMEF: writing to a socket\n");
       tcpCurrentMessage(spec.outdet.sock_dest);
   }
   
   return;
}

/*
 * Function: IDMEFCleanExit(int signal, void *arg)
 *
 * Purpose: Makes preperations for the program to exit.
 *
 * Arguments: signal => the signal (we don't really care what it for now)
 *            arg    => an argument
 *
 * Returns: void function
 */

void IDMEFCleanExit(int signal, void *arg)
{
#ifdef DEBUG
   printf("IDMEFCleanExit\n");
#endif
     
#ifdef IDMEF_DEBUG
   printf("IDMEF: Saving alert ID %lu to %s\n", idmef_alertid, ips.alert_id_fn);
#endif
   if(!saveAlertID(idmef_alertid, DEFAULT_ALERTID_FN))
     FatalError("IDMEF: Error trying to save last alert ID (%lu) to %s\n",
		idmef_alertid, DEFAULT_ALERTID_FN);

   fflush(idmef_log_file);
   fclose(idmef_log_file);   

   if(doc != NULL)
     xmlFreeDoc(doc);

}

/*
 * Function: FreeGlobalsAndMessage(Packet *p)
 *
 * Purpose: Free the components of the IDMEF xml message.
 *
 * Arguments: 
 *            p =>       The current packet
 *            payload => The packet payload
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 */

void FreeGlobalsAndMessage(Packet *p, char *payload)
{
#ifdef IDMEF_DEBUG
   fprintf(stderr,"** spo_idmef.c: In FreeGLobalsAndMessage\n");
#endif
    clearCurrentDoc();
    FreeGlobals(p, payload);
    /* xmlMemShow(stdout, 50); */
}


/*
 * Function: FreeGlobals(Packet *p)
 *
 * Purpose: Free the components of the IDMEF xml message.
 *
 * Arguments: 
 *            p =>       The current packet
 *            payload => The packet payload
 *
 * Returns: An xmlNodePtr to the root element of the IDMEF Message
 */

void FreeGlobals(Packet *p, char *payload)
{
  free(idmef_alertid_str);

  if(p != NULL)
  {
     if((p->iph->ip_proto == IPPROTO_TCP) || (p->iph->ip_proto == IPPROTO_UDP))
     {
        free(sport);
        free(dport);
     }
  }

  if(payload != NULL)
    free(payload);

  /* xmlMemShow(stdout, 50); */
}

/*
 * Function: incAndFlush()
 *
 * Purpose: Increment the alert ID and flush the log file.
 *
 * Returns: void function
 */

void incAndFlush()
{
   idmef_alertid++;

   fflush(idmef_log_file);
}

/* This is used for testing purposes only. */

void tcpCurrentMessage(struct sockaddr_in *sock_dest) {
	int mySocket;

	xmlChar *mess; // get string for "doc" from libxml
	int size;
	xmlDocDumpMemory(doc, &mess, &size);
	printf("going to write to %p: %s\n",sock_dest,mess);
	
	if ((mySocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
    	perror("spo_idmef: socket");
	    return;
	}
	printf("sock_dest->sin_family=%d\n",sock_dest->sin_family);
    printf("sock_dest->sin_port=%d\n",ntohs(sock_dest->sin_port));
    printf("sock_dest->sin_addr=%s\n",inet_ntoa(sock_dest->sin_addr));
	printf("socket descriptor: %d\n", mySocket);
	if (connect(mySocket, sock_dest, sizeof(struct sockaddr_in)) == -1) {
    	close(mySocket);
    	perror("spo_idmef: connect");
    	exit(1);
    }
 	write(mySocket, mess, size);
	close(mySocket);
}

#endif
