/*************************************************************************
 *
 *  $RCSfile: inetmsg.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2003/04/17 16:50:40 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#define _INETCOREMSG_CXX "$Revision: 1.2 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _SOLAR_H
#include <tools/solar.h>
#endif
#ifndef _DATETIME_HXX
#include <tools/datetime.hxx>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif
#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif

#ifndef _INETCOREMSG_HXX
#include "inetmsg.hxx"
#endif
#ifndef _INETCORESTRM_HXX
#include "inetstrm.hxx"
#endif

#include <stdio.h>

/*=======================================================================
 *
 * INetCoreMessage (Message Container) Implementation.
 *
 * References:
 *   RFC  822 - Format of Electronic Mail Messages (STD 11).
 *   RFC 1036 - Standard for Interchange of USENET Messages.
 *   RFC 1123 - Requirements for Internet Hosts (STD 3).
 *   RFC 1521 - Multipurpose Internet Mail Extensions (Draft Standard).
 *   RFC 1522 - MIME Part II - Message Header Extension for Non-ASCII Text.
 *   RFC 1920 - Internet Official Protocol Standards (STD 1).
 *
 *=====================================================================*/
#define INETCOREMSG_STREAM_MAGIC   ((UINT32)0xfefefefe)
#define INETCOREMSG_STREAM_SEEKREL (-((long)(sizeof(UINT32))))

#if ((SUPD > 505) && (SUPD < 514)) /* NOT SO5.0 COMPATIBLE */
#define INETCOREMSG_STREAM_VERSION_NUMBER 2
#elif (SUPD > 363)
#define INETCOREMSG_STREAM_VERSION_NUMBER 1
#else
#define INETCOREMSG_STREAM_VERSION_NUMBER 0
#endif /* SUPD */

#define INETCOREMSG_STREAM_VERSION ((UINT32)INETCOREMSG_STREAM_VERSION_NUMBER)

//=======================================================================

inline sal_Bool ascii_isDigit( sal_Unicode ch )
{
    return ((ch >= 0x0030) && (ch <= 0x0039));
}

inline sal_Bool ascii_isLetter( sal_Unicode ch )
{
    return (( (ch >= 0x0041) && (ch <= 0x005A)) || ((ch >= 0x0061) && (ch <= 0x007A)));
}

/*=======================================================================
 *
 * INetCoreMessage Implementation.
 *
 *=====================================================================*/
INetCoreMessage::INetCoreMessage (void)
    : m_nDocSize   (0),
      m_pDocStream (NULL),
      m_pDocLB     (NULL)
{
}

INetCoreMessage::INetCoreMessage (const INetCoreMessage& rMsg)
{
    // (NYI).
}

INetCoreMessage&
INetCoreMessage::operator= (const INetCoreMessage& rMsg)
{
    if (this != &rMsg)
    {
        // (NYI).
    }
    return *this;
}

INetCoreMessage::~INetCoreMessage (void)
{
}

SvStream& INetCoreMessage::operator<< (SvStream& rStrm) const
{
    rStrm << INETCOREMSG_STREAM_MAGIC;
    rStrm << INETCOREMSG_STREAM_VERSION;

    rStrm.WriteByteString (m_aDocFilename, RTL_TEXTENCODING_UTF8);
    rStrm << m_nDocSize;

    return rStrm;
}

SvStream& INetCoreMessage::operator>> (SvStream& rStrm)
{
    UINT32 nMagic = 0, nVersion = 0;
    rStrm >> nMagic;
    if (nMagic == INETCOREMSG_STREAM_MAGIC)
        rStrm >> nVersion;
    else
        rStrm.SeekRel (INETCOREMSG_STREAM_SEEKREL);

    rStrm.ReadByteString (m_aDocFilename, RTL_TEXTENCODING_UTF8);
    rStrm >> m_nDocSize;

    return rStrm;
}

/*========================================================================
 *
 * INetCoreRFC822Message Implementation.
 *
 *======================================================================*/
INetCoreRFC822Message::INetCoreRFC822Message (void)
    : INetCoreMessage ()
{
}

INetCoreRFC822Message::INetCoreRFC822Message (
    const INetCoreRFC822Message& rMsg)
    // : INetCoreMessage (rMsg)
{
    // (NYI).
}

INetCoreRFC822Message&
INetCoreRFC822Message::operator= (const INetCoreRFC822Message& rMsg)
{
    if (this != &rMsg)
    {
#if 0 /* (NYI) */
        INetCoreMessage::operator= (rMsg);
#endif /* (NYI) */
    }
    return *this;
}

INetCoreRFC822Message::~INetCoreRFC822Message (void)
{
}

SvStream& INetCoreRFC822Message::operator<< (SvStream& rStrm) const
{
    INetCoreMessage::operator<< (rStrm);

    rStrm << INETCOREMSG_STREAM_MAGIC;
    rStrm << INETCOREMSG_STREAM_VERSION;

    rStrm.WriteByteString (m_aBCC);
    rStrm.WriteByteString (m_aCC);
    rStrm.WriteByteString (m_aComments);
    rStrm.WriteByteString (m_aDate);
#if (INETCOREMSG_STREAM_VERSION_NUMBER > 0)
    rStrm.WriteByteString (m_aEncoding);
#endif
    rStrm.WriteByteString (m_aFrom);
    rStrm.WriteByteString (m_aInReplyTo);
    rStrm.WriteByteString (m_aKeywords);
    rStrm.WriteByteString (m_aMessageID);
    rStrm.WriteByteString (m_aReceived);
    rStrm.WriteByteString (m_aReferences);
    rStrm.WriteByteString (m_aReplyTo);
    rStrm.WriteByteString (m_aReturnPath);
    rStrm.WriteByteString (m_aSubject);
    rStrm.WriteByteString (m_aSender);
    rStrm.WriteByteString (m_aTo);
    rStrm.WriteByteString (m_aXMailer);
#if (INETCOREMSG_STREAM_VERSION_NUMBER > 0)
    rStrm.WriteByteString (m_aXPriority);
#endif
    rStrm.WriteByteString (m_aReturnReceiptTo);

    return rStrm;
}

SvStream& INetCoreRFC822Message::operator>> (SvStream& rStrm)
{
    INetCoreMessage::operator>> (rStrm);

    UINT32 nMagic = 0, nVersion = 0;
    rStrm >> nMagic;
    if (nMagic == INETCOREMSG_STREAM_MAGIC)
        rStrm >> nVersion;
    else
        rStrm.SeekRel (INETCOREMSG_STREAM_SEEKREL);

    rStrm.ReadByteString (m_aBCC);
    rStrm.ReadByteString (m_aCC);
    rStrm.ReadByteString (m_aComments);
    rStrm.ReadByteString (m_aDate);
    if (nVersion > 0) rStrm.ReadByteString (m_aEncoding);
    rStrm.ReadByteString (m_aFrom);
    rStrm.ReadByteString (m_aInReplyTo);
    rStrm.ReadByteString (m_aKeywords);
    rStrm.ReadByteString (m_aMessageID);
    rStrm.ReadByteString (m_aReceived);
    rStrm.ReadByteString (m_aReferences);
    rStrm.ReadByteString (m_aReplyTo);
    rStrm.ReadByteString (m_aReturnPath);
    rStrm.ReadByteString (m_aSubject);
    rStrm.ReadByteString (m_aSender);
    rStrm.ReadByteString (m_aTo);
    rStrm.ReadByteString (m_aXMailer);
    if (nVersion > 0) rStrm.ReadByteString (m_aXPriority);
    rStrm.ReadByteString (m_aReturnReceiptTo);

    return rStrm;
}

/*
 * <Generate|Parse>DateField and local helper functions.
 *
 * GenerateDateField.
 * Generates a String from Date and Time objects in format:
 *   Wkd, 00 Mon 0000 00:00:00 [GMT]            (rfc822, rfc1123)
 *
 * ParseDateField.
 * Parses a String in (implied) GMT format into class Date and Time objects.
 * Four formats are accepted:
 *
 *  [Wkd,] 1*2DIGIT Mon 2*4DIGIT 00:00:00 [GMT]  (rfc1123)
 *  [Wkd,] 00 Mon 0000 00:00:00 [GMT])           (rfc822, rfc1123)
 *   Weekday, 00-Mon-00 00:00:00 [GMT]           (rfc850, rfc1036)
 *   Wkd Mon 00 00:00:00 0000 [GMT]              (ctime)
 *   1*DIGIT                                     (delta seconds)
 *
 */
/*
 * GenerateDateField.
 */
BOOL INetCoreRFC822Message::GenerateDateField (
    const Date& rDate, const Time& rTime, UniString& rDateFieldW) const
{
    // Check arguments.
    if (!rDate.IsValid () ||
        (rTime.GetSec()  > 59) ||
        (rTime.GetMin()  > 59) ||
        (rTime.GetHour() > 23)    ) return FALSE;

    // Months and Weekdays.
    const char *months[12] = {
        "Jan", "Feb", "Mar", "Apr", "May", "Jun",
        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
    };

    const char *wkdays[7] = {
        "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"
    };

    // Prepare output string.
    ByteString rDateField;

    // Insert Date.
    rDateField += wkdays[(USHORT)(rDate.GetDayOfWeek())];
    rDateField += ", ";

    USHORT nNum = rDate.GetDay();
    if (nNum < 10) rDateField += '0';
    rDateField += ByteString::CreateFromInt32 (nNum);
    rDateField += ' ';

    rDateField += months[(USHORT)(rDate.GetMonth() - 1)];
    rDateField += ' ';

    rDateField += ByteString::CreateFromInt32 (rDate.GetYear());
    rDateField += ' ';

    // Insert Time.
    nNum = rTime.GetHour();
    if (nNum < 10) rDateField += '0';
    rDateField += ByteString::CreateFromInt32 (nNum);
    rDateField += ':';

    nNum = rTime.GetMin();
    if (nNum < 10) rDateField += '0';
    rDateField += ByteString::CreateFromInt32 (nNum);
    rDateField += ':';

    nNum = rTime.GetSec();
    if (nNum < 10) rDateField += '0';
    rDateField += ByteString::CreateFromInt32 (nNum);
    rDateField += " GMT";

    // Done.
    rDateFieldW = UniString (rDateField, RTL_TEXTENCODING_ASCII_US);
    return TRUE;
}

/*
 * ParseDateField and local helper functions.
 */
static USHORT ParseNumber (const ByteString& rStr, USHORT& nIndex)
{
    USHORT n = nIndex;
    while ((n < rStr.Len()) && ascii_isDigit (rStr.GetChar(n))) n++;

    ByteString aNum (rStr.Copy (nIndex, (n - nIndex)));
    nIndex = n;

    return (USHORT)(aNum.ToInt32());
}

static USHORT ParseMonth (const ByteString& rStr, USHORT& nIndex)
{
    USHORT n = nIndex;
    while ((n < rStr.Len()) && ascii_isLetter (rStr.GetChar(n))) n++;

    ByteString aMonth (rStr.Copy (nIndex, 3));
    nIndex = n;

    const char *months[12] = {
        "Jan", "Feb", "Mar", "Apr", "May", "Jun",
        "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
    };

    USHORT i;
    for (i = 0; i < 12; i++)
        if (aMonth.CompareIgnoreCaseToAscii (months[i]) == 0) break;
    return (i + 1);
}

static USHORT MakeYear (USHORT nYear)
{
    if (nYear < 100)
    {
        USHORT nCurrent = Date().GetYear();
        USHORT nCentury = nCurrent / 100;
        nCurrent %= 100;
        if (nCurrent < 50)
        {
            if (nYear <= nCurrent)
                nYear += nCentury * 100;
            else if (nYear < nCurrent + 50)
                nYear += nCentury * 100;
            else
                nYear += (nCentury - 1) * 100;
        }
        else
        {
            if (nYear >= nCurrent)
                nYear += nCentury * 100;
            else if (nYear >= nCurrent - 50)
                nYear += nCentury * 100;
            else
                nYear += (nCentury + 1) * 100;
        }
    }
    return nYear;
}

BOOL INetCoreRFC822Message::ParseDateField (
    const UniString& rDateFieldW, Date& rDate, Time& rTime) const
{
    ByteString rDateField (rDateFieldW, RTL_TEXTENCODING_ASCII_US);
    if (rDateField.Len() == 0) return FALSE;

    if (rDateField.Search (':') != STRING_NOTFOUND)
    {
        // Some DateTime format.
        USHORT nIndex = 0;

        // Skip over <Wkd> or <Weekday>, leading and trailing space.
        while ((nIndex < rDateField.Len()) &&
               (rDateField.GetChar(nIndex) == ' '))
            nIndex++;

        while ((nIndex < rDateField.Len()) &&
               (ascii_isLetter (rDateField.GetChar(nIndex)) ||
                (rDateField.GetChar(nIndex) == ',')))
            nIndex++;

        while ((nIndex < rDateField.Len()) &&
               (rDateField.GetChar(nIndex) == ' '))
            nIndex++;

        if (ascii_isLetter (rDateField.GetChar(nIndex)))
        {
            // Format: ctime().
            if ((rDateField.Len() - nIndex) < 20) return FALSE;

            rDate.SetMonth (ParseMonth  (rDateField, nIndex)); nIndex++;
            rDate.SetDay   (ParseNumber (rDateField, nIndex)); nIndex++;

            rTime.SetHour  (ParseNumber (rDateField, nIndex)); nIndex++;
            rTime.SetMin   (ParseNumber (rDateField, nIndex)); nIndex++;
            rTime.SetSec   (ParseNumber (rDateField, nIndex)); nIndex++;

            USHORT nYear = ParseNumber (rDateField, nIndex);
            nYear        = MakeYear (nYear);
            rDate.SetYear  (nYear);
        }
        else
        {
            // Format: RFC1036 or RFC1123.
            if ((rDateField.Len() - nIndex) < 17) return FALSE;

            rDate.SetDay   (ParseNumber (rDateField, nIndex)); nIndex++;
            rDate.SetMonth (ParseMonth  (rDateField, nIndex)); nIndex++;

            USHORT nYear  = ParseNumber (rDateField, nIndex);  nIndex++;
            nYear         = MakeYear (nYear);
            rDate.SetYear  (nYear);

            rTime.SetHour (ParseNumber (rDateField, nIndex)); nIndex++;
            rTime.SetMin  (ParseNumber (rDateField, nIndex)); nIndex++;
            rTime.SetSec  (ParseNumber (rDateField, nIndex)); nIndex++;

            if ((rDateField.GetChar(nIndex) == '+') ||
                (rDateField.GetChar(nIndex) == '-')    )
            {
                // Offset from GMT: "(+|-)HHMM".
                BOOL   bEast   = (rDateField.GetChar(nIndex++) == '+');
                USHORT nOffset = ParseNumber (rDateField, nIndex);
                if (nOffset > 0)
                {
                    Time aDiff (0);
                    aDiff.SetHour (nOffset / 100);
                    aDiff.SetMin  (nOffset % 100);

                    DateTime aDT (rDate, rTime);
                    if (bEast)
                        aDT -= aDiff;
                    else
                        aDT += aDiff;

                    rDate.SetDate (aDT.GetDate());
                    rTime.SetTime (aDT.GetTime());
                }
            }
        }
    }
    else if (rDateField.IsNumericAscii())
    {
        // Format: delta seconds.
        Time aDelta (0);
        aDelta.SetTime (rDateField.ToInt32() * 100);

        DateTime aDT;
        aDT += aDelta; // -=
        aDT.ConvertToUTC();

        rDate.SetDate (aDT.GetDate());
        rTime.SetTime (aDT.GetTime());
    }
    else
    {
        // Junk.
        return FALSE;
    }

    return (rDate.IsValid () &&
            !((rTime.GetSec()  > 59) ||
              (rTime.GetMin()  > 59) ||
              (rTime.GetHour() > 23)    ));
}

/*=======================================================================
 *
 * INetCoreMIMEMessage Implementation.
 *
 *=====================================================================*/
INetCoreMIMEMessage::INetCoreMIMEMessage (void)
    : INetCoreRFC822Message (),
      m_pParent   (NULL),
      m_nChildren (0)
{
}

INetCoreMIMEMessage::INetCoreMIMEMessage (const INetCoreMIMEMessage& rMsg)
    // : INetCoreRFC822Message (rMsg)
{
    // (NYI).
}

INetCoreMIMEMessage&
INetCoreMIMEMessage::operator= (const INetCoreMIMEMessage& rMsg)
{
    if (this != &rMsg)
    {
#if 0 /* (NYI) */
        INetCoreRFC822Message::operator= (rMsg);
#endif /* (NYI) */
    }
    return *this;
}

INetCoreMIMEMessage::~INetCoreMIMEMessage (void)
{
#ifdef _PB_EXPERIMENTAL_

    // Cleanup child list.
    INetCoreMIMEMessage *pChild;
    while (pChild = (INetCoreMIMEMessage *)aChildren.Remove()) delete pChild;

#endif /* _PB_EXPERIMENTAL_ */
}

SvStream& INetCoreMIMEMessage::operator<< (SvStream& rStrm) const
{
    INetCoreRFC822Message::operator<< (rStrm);

    rStrm << INETCOREMSG_STREAM_MAGIC;
    rStrm << INETCOREMSG_STREAM_VERSION;

    rStrm << m_nChildren;
    rStrm.WriteByteString (m_aBoundary);

    rStrm.WriteByteString (m_aMIMEVersion);
#if (INETCOREMSG_STREAM_VERSION_NUMBER > 0)
    rStrm.WriteByteString (m_aContentBase);
#endif
    rStrm.WriteByteString (m_aContentDescription);
    rStrm.WriteByteString (m_aContentDisposition);
    rStrm.WriteByteString (m_aContentID);
#if (INETCOREMSG_STREAM_VERSION_NUMBER > 0)
    rStrm.WriteByteString (m_aContentLocation);
#endif
    rStrm.WriteByteString (m_aContentType);
    rStrm.WriteByteString (m_aContentTransferEncoding);

    return rStrm;
}

SvStream& INetCoreMIMEMessage::operator>> (SvStream& rStrm)
{
    INetCoreRFC822Message::operator>> (rStrm);

    UINT32 nMagic = 0, nVersion = 0;
    rStrm >> nMagic;
    if (nMagic == INETCOREMSG_STREAM_MAGIC)
        rStrm >> nVersion;
    else
        rStrm.SeekRel (INETCOREMSG_STREAM_SEEKREL);

    rStrm >> m_nChildren;
    rStrm.ReadByteString (m_aBoundary);

    rStrm.ReadByteString (m_aMIMEVersion);
    if (nVersion > 0) rStrm.ReadByteString (m_aContentBase);
    rStrm.ReadByteString (m_aContentDescription);
    rStrm.ReadByteString (m_aContentDisposition);
    rStrm.ReadByteString (m_aContentID);
    if (nVersion > 0) rStrm.ReadByteString (m_aContentLocation);
    rStrm.ReadByteString (m_aContentType);
    rStrm.ReadByteString (m_aContentTransferEncoding);

    return rStrm;
}

INetCoreMIMEMessageStream*
INetCoreMIMEMessage::CreateMessageStream (void) const
{
    return new INetCoreMIMEMessageStream;
}

BOOL INetCoreMIMEMessage::EnableAttachChild (
    INetCoreMessageContainerType eType)
{
    // Check context.
    if (IsContainer()) return FALSE;

    // Setup Content-Type header field.
    switch (eType)
    {
        case INETCOREMSG_MESSAGE_RFC822:
            m_aContentType = "message/rfc822";
            break;

        case INETCOREMSG_MULTIPART_ALTERNATIVE:
            m_aContentType = "multipart/alternative";
            break;

        case INETCOREMSG_MULTIPART_DIGEST:
            m_aContentType = "multipart/digest";
            break;

        case INETCOREMSG_MULTIPART_PARALLEL:
            m_aContentType = "multipart/parallel";
            break;

        case INETCOREMSG_MULTIPART_RELATED:
            m_aContentType = "multipart/related";
            break;

        default:
            m_aContentType = "multipart/mixed";
            break;
    }

    // Setup boundary for multipart types.
    if (IsMultipart())
    {
        // Generate a unique boundary from current time.
        sal_Char pTail[16 + 1];
        Time aCurTime;

        DBG_ASSERT( sizeof(pTail) <= 17,
                    "INetCoreMIMEMessage::EnableAttachChild: buffer to small!");

        snprintf (pTail, sizeof(pTail), "%08X%08X",
                  aCurTime.GetTime(), (ULONG)this);
        m_aBoundary = "------------=_4D48";
        m_aBoundary += pTail;

        // Append boundary as ContentType parameter.
        m_aContentType += "; boundary=\"";
        m_aContentType += m_aBoundary;
        m_aContentType += '\"';
    }

    // Set additional header fields.
    m_aMIMEVersion = "1.0";
    m_aContentTransferEncoding = "7bit";

    // Done.
    return TRUE;
}

BOOL INetCoreMIMEMessage::AttachChild (
    INetCoreMIMEMessage& rChildMsg)
{
    if (IsContainer() && rChildMsg.m_aContentType.Len())
    {
        rChildMsg.m_pParent = this;
        m_aChildren.Insert (&rChildMsg, LIST_APPEND);
        m_nChildren = m_aChildren.Count();

        return TRUE;
    }
    return FALSE;
}

BOOL INetCoreMIMEMessage::DetachChild (
    ULONG nIndex, INetCoreMIMEMessage& rChildMsg) const
{
    if (IsContainer())
    {
        // Check document stream.
        SvStream *pIS = GetDocumentStream();
        if (pIS == NULL) return FALSE;
        pIS->Seek (STREAM_SEEK_TO_BEGIN);

        // Initialize message buffer.
        char pMsgBuffer[1024];
        char *pMsgRead, *pMsgWrite;
        pMsgRead = pMsgWrite = pMsgBuffer;

        // Initialize message parser stream.
        INetCoreMIMEMessageStream *pMsgStream = NULL;

        // Check for "multipart/uvw" or "message/xyz".
        if (IsMultipart())
        {
            // Multipart message body. Check index.
            // if (!(m_nChildren > nIndex)) return FALSE;

            //Initialize multipart delimiters.
            ByteString aDelim ("--");
            aDelim += m_aBoundary;

            ByteString aClose (aDelim);
            aClose += "--";

            // Initialize token buffer.
            char pTokBuffer[512];
            char *pTokRead, *pTokWrite;
            pTokRead = pTokWrite = pTokBuffer;

            // Initialize control variables.
            int  nCurIndex = -1;
            BOOL bEOL = FALSE;

            // Go!
            while (nCurIndex < (int)(nIndex + 1))
            {
                if ((pMsgRead - pMsgWrite) > 0)
                {
                    // Bytes still in buffer.
                    if (bEOL)
                    {
                        // Check for 2nd line break character.
                        if ((*pMsgWrite == '\r') || (*pMsgWrite == '\n'))
                            *pTokWrite++ = *pMsgWrite++;

                        // Check current index.
                        if (nCurIndex == (int)nIndex)
                        {
                            // Found requested part index.
                            if (pMsgStream == NULL)
                            {
                                // Create message parser stream.
                                pMsgStream = rChildMsg.CreateMessageStream();
                                pMsgStream->SetTargetMessage (&rChildMsg);
                            }
                            else
                            {
                                // Put message down-stream.
                                int status = pMsgStream->Write (
                                    pTokBuffer,
                                    (pTokWrite - pTokBuffer), NULL);
                                if (status != INETCORESTREAM_STATUS_OK)
                                {
                                    // Cleanup.
                                    delete pMsgStream;
                                    pMsgStream = NULL;

                                    // Finish.
                                    return (!(status ==
                                              INETCORESTREAM_STATUS_ERROR));
                                }
                            }
                        }

                        // Reset to <Begin-of-Line>.
                        pTokRead = pTokWrite = pTokBuffer;
                        bEOL = FALSE;
                    }
                    else if ((*pMsgWrite == '\r') || (*pMsgWrite == '\n'))
                    {
                        /*
                         * Found any line break character.
                         * Compare buffered line with part/close delimiter.
                         */
                        USHORT nTokLen = pTokWrite - pTokBuffer;
                        if (nTokLen >= aDelim.Len())
                        {
                            if ((aDelim.CompareTo (pTokBuffer, aDelim.Len())
                                 == COMPARE_EQUAL) ||
                                (aClose.CompareTo (pTokBuffer, aClose.Len())
                                 == COMPARE_EQUAL))
                            {
                                // Increment current part index.
                                nCurIndex++;
                            }
                        }
                        *pTokWrite++ = *pMsgWrite++;
                        bEOL = TRUE;
                    }
                    else
                    {
                        // Insert into line buffer.
                        *pTokWrite++ = *pMsgWrite++;

                        // Check for buffer overflow.
                        USHORT nTokLen = pTokWrite - pTokBuffer;
                        if ((nTokLen + 1) == sizeof (pTokBuffer))
                            bEOL = TRUE;
                    }
                }
                else
                {
                    // Buffer empty. Reset to <Begin-of-Buffer>.
                    pMsgRead = pMsgWrite = pMsgBuffer;

                    // Read document.
                    ULONG nRead = pIS->Read (pMsgBuffer, sizeof (pMsgBuffer));
                    if (nRead > 0)
                    {
                        // Set read pointer.
                        pMsgRead = (pMsgBuffer + nRead);
                    }
                    else
                    {
                        // Premature end.
                        if (pMsgStream)
                        {
                            // Assume end of requested part.
                            nCurIndex++;
                        }
                        else
                        {
                            // Requested part not found.
                            return FALSE;
                        }
                    }
                }
            } // while (nCurIndex < (nIndex + 1))
        }
        else
        {
            // Encapsulated message body. Create message parser stream.
            pMsgStream = rChildMsg.CreateMessageStream();
            pMsgStream->SetTargetMessage (&rChildMsg);

            // Initialize control variables.
            INetCoreMIMEMessageStreamState eState = INETCOREMSG_MIME_BEGIN;

            // Go!
            while (eState == INETCOREMSG_MIME_BEGIN)
            {
                if ((pMsgRead - pMsgWrite) > 0)
                {
                    // Bytes still in buffer. Put message down-stream.
                    int status = pMsgStream->Write (
                        pMsgBuffer, (pMsgRead - pMsgWrite), NULL);
                    if (status != INETCORESTREAM_STATUS_OK)
                    {
                        // Cleanup.
                        delete pMsgStream;
                        pMsgStream = NULL;

                        // Finish.
                        return (!(status == INETCORESTREAM_STATUS_ERROR));
                    }

                    pMsgWrite = pMsgBuffer + (pMsgRead - pMsgWrite);
                }
                else
                {
                    // Buffer empty. Reset to <Begin-Of-Buffer>.
                    pMsgRead = pMsgWrite = pMsgBuffer;

                    // Read document.
                    ULONG nRead = pIS->Read (pMsgBuffer, sizeof (pMsgBuffer));
                    if (nRead > 0)
                    {
                        // Set read pointer.
                        pMsgRead = pMsgBuffer + nRead;
                    }
                    else
                    {
                        // Mark we're done.
                        eState = INETCOREMSG_MIME_END;
                    }
                }
            } // while (eState == INETCOREMSG_MIME_BEGIN)
        } // if (IsMultipart())

        // Done.
        if (pMsgStream)
        {
            delete pMsgStream;
            pMsgStream = NULL;
        }
        return TRUE;
    }
    return FALSE;
}

/*
 * GetDefaultContentType_Impl.
 */
ByteString INetCoreMIMEMessage::GetDefaultContentType_Impl (void) const
{
    ByteString aDCT ("text/plain; charset=us-ascii");
    if (m_pParent)
    {
        ByteString aPCT (m_pParent->m_aContentType);
        if (aPCT.Len() == 0)
            aPCT = m_pParent->GetDefaultContentType_Impl();
        if (aPCT.CompareIgnoreCaseToAscii ("multipart/digest", 17) == 0)
            return ByteString ("message/rfc822");
    }
    return aDCT;
}

/*=======================================================================
 *
 * INetCoreNewsMessage Implementation.
 *
 *=====================================================================*/
INetCoreNewsMessage::INetCoreNewsMessage (void)
    : INetCoreMIMEMessage ()
{
}

INetCoreNewsMessage::INetCoreNewsMessage (const INetCoreNewsMessage& rMsg)
    // : INetCoreMIMEMessage (rMsg)
{
    // (NYI).
}

INetCoreNewsMessage&
INetCoreNewsMessage::operator= (const INetCoreNewsMessage& rMsg)
{
    if (this != &rMsg)
    {
#if 0 /* (NYI) */
        INetCoreMIMEMessage::operator= (rMsg);
#endif /* (NYI) */
    }
    return *this;
}

INetCoreNewsMessage::~INetCoreNewsMessage (void)
{
}

SvStream& INetCoreNewsMessage::operator<< (SvStream& rStrm) const
{
    INetCoreMIMEMessage::operator<< (rStrm);

    rStrm << INETCOREMSG_STREAM_MAGIC;
    rStrm << INETCOREMSG_STREAM_VERSION;

    rStrm.WriteByteString (m_aNewsgroups);
    rStrm.WriteByteString (m_aPath);

    rStrm.WriteByteString (m_aApproved);
#if (INETCOREMSG_STREAM_VERSION_NUMBER > 1)
    rStrm.WriteByteString (m_aBytes);
#endif
    rStrm.WriteByteString (m_aControl);
    rStrm.WriteByteString (m_aDistribution);
    rStrm.WriteByteString (m_aExpires);
    rStrm.WriteByteString (m_aFollowupTo);
    rStrm.WriteByteString (m_aLines);
    rStrm.WriteByteString (m_aOrganization);
    rStrm.WriteByteString (m_aSummary);
    rStrm.WriteByteString (m_aXref);
    rStrm.WriteByteString (m_aXNewsreader);

    return rStrm;
}

SvStream& INetCoreNewsMessage::operator>> (SvStream& rStrm)
{
    INetCoreMIMEMessage::operator>> (rStrm);

    UINT32 nMagic = 0, nVersion = 0;
    rStrm >> nMagic;
    if (nMagic == INETCOREMSG_STREAM_MAGIC)
        rStrm >> nVersion;
    else
        rStrm.SeekRel (INETCOREMSG_STREAM_SEEKREL);

    rStrm.ReadByteString (m_aNewsgroups);
    rStrm.ReadByteString (m_aPath);

    rStrm.ReadByteString (m_aApproved);
#if (SUPD > 505)
    if (nVersion > 1) rStrm.ReadByteString (m_aBytes);
#endif /* (SUPD > 505) */
    rStrm.ReadByteString (m_aControl);
    rStrm.ReadByteString (m_aDistribution);
    rStrm.ReadByteString (m_aExpires);
    rStrm.ReadByteString (m_aFollowupTo);
    rStrm.ReadByteString (m_aLines);
    rStrm.ReadByteString (m_aOrganization);
    rStrm.ReadByteString (m_aSummary);
    rStrm.ReadByteString (m_aXref);
    rStrm.ReadByteString (m_aXNewsreader);

    return rStrm;
}

INetCoreMIMEMessageStream*
INetCoreNewsMessage::CreateMessageStream (void) const
{
    return new INetCoreNewsMessageStream;
}

