/*!========================================================================

  @file         RTE_SecurityAttributesNT.cpp
  @ingroup      Runtime
  @author       StefanP

  @brief        Local Communication

  @since        2003-09-15  11:50
  @sa           

  ==========================================================================

  \if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2003-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
7
    You should have received a copy of the GNU General Public License
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
  \endif
 ============================================================================*/

#if defined (_WIN32)
/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/
/*
#define ASSERT_ARGUMENT_OPTION  SAPDBERR_CRASH
#define ASSERT_STATE_OPTION     SAPDBERR_CRASH
#define ASSERT_RANGE_OPTION     SAPDBERR_CRASH
*/

#include    "gos00.h"

#include    "RunTime/RTE_SecurityAttributesNT.hpp"
#include    "RunTime/MemoryManagement/RTEMem_Allocator.hpp" 
#include    "RunTime/RTE_Messages.hpp"





/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/


/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTION PROTOTYPES                                        *
 *===========================================================================*/
RTE_SecurityAttributes   *RTE_SecurityAttributes::m_Instance = NULL; 


/*===========================================================================*
 *  METHODS                                                                  *
 *===========================================================================*/

 
RTE_SecurityAttributes & RTE_SecurityAttributes::Instance()
{
    SAPDBErr_MessageList    messageList;

    SAPDBTRACE_ROUTINE_DEBUG("RTE_SecurityAttributes::Instance", Runtime_Trace, 9);

    if (NULL == m_Instance)
    {
        m_Instance = new (RTEMem_Allocator::Instance()) RTE_SecurityAttributes ();
        if (NULL == m_Instance)
        {
            RTE_Crash (SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SYS_ALLOC_INST, "RTE_SecurityAttributes"));
        }

        if (!m_Instance->Initialize (messageList))
        {
            messageList = messageList + SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SECURITY_INIT_ATTRIBUTES);
            RTE_Crash (messageList);
        }
    }

    return *m_Instance;
}

//--------------------------------------------------------------------------------

SAPDB_Bool RTE_SecurityAttributes::Initialize (SAPDBErr_MessageList        &   messageList)
{
    ACCESS_RIGHTS_REC       Access;
    RTE_SystemRc            rc = RTE_SYSTEMRC_NO_ERROR;


    SAPDBTRACE_ROUTINE_DEBUG("RTE_SecurityAttributes::Initialize", Runtime_Trace, 9);

    // --- initialize Windows NT security
    rc = sql49c_set_sync_access ();
    if (RTE_SYSTEMRC_NO_ERROR != rc)
    {
        messageList = SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SECURITY_SET_SYNCHRONIZE_ACCESS, 
            SAPDB_ToString (rc));
        return false;
    }

    //World Access
    m_WorldSA.nLength           = sizeof(m_WorldSA);
    m_WorldSA.bInheritHandle    = FALSE;

    memset (&Access, 0, sizeof (Access));   // - this must be done first!
    Access.World.ulAccess     = SPECIFIC_RIGHTS_ALL | STANDARD_RIGHTS_ALL;

    rc = sql49c_alloc_and_init_SD( &Access, &m_WorldSA.lpSecurityDescriptor );
    if (RTE_SYSTEMRC_NO_ERROR != rc)
    {
        messageList = SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SECURITY_INIT_DESCRIPTOR, 
            "World", SAPDB_ToString (rc));
        return false;
    }

    //Lock Access
    m_LockSA.nLength            = sizeof(m_LockSA);
    m_LockSA.bInheritHandle     = FALSE;

    memset ( &Access, 0, sizeof (Access) );   // - this must be done first!
    rc = sql49c_alloc_and_init_SD(&Access, &m_LockSA.lpSecurityDescriptor);
    if (RTE_SYSTEMRC_NO_ERROR != rc)
    {
        messageList = SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SECURITY_INIT_DESCRIPTOR, 
            "Lock", SAPDB_ToString (rc));
        return false;
    }

    return true;
}

//--------------------------------------------------------------------------------

void    RTE_SecurityAttributes::Release  ()
{
    SAPDBTRACE_METHOD_DEBUG ("RTE_SecurityAttributes::Release", Runtime_Trace, 9);

    if (NULL != m_WorldSA.lpSecurityDescriptor)
    {
        FREE_MEM (m_WorldSA.lpSecurityDescriptor);
        m_WorldSA.lpSecurityDescriptor = NULL;
    }

    if (NULL != m_LockSA.lpSecurityDescriptor)
    {
        FREE_MEM (m_LockSA.lpSecurityDescriptor);
        m_LockSA.lpSecurityDescriptor = NULL;
    }

    return;
}

//--------------------------------------------------------------------------------

SAPDB_Bool  RTE_SecurityAttributes::Lock  (HANDLE const                       hObject,
                                           SAPDBErr_MessageList &             messageList)
const
{
    RTE_SystemRc                                rc = RTE_SYSTEMRC_NO_ERROR;    

    SAPDBTRACE_METHOD_DEBUG ("RTE_SecurityAttributes::Lock", Runtime_Trace, 9);

    if (!SetKernelObjectSecurity (hObject,
                                  DACL_SECURITY_INFORMATION,
                                  m_LockSA.lpSecurityDescriptor))
    {
        rc = GetLastError();
        messageList = SAPDBErr_MessageList (RTE_CONTEXT, RTEERR_SECURITY_LOCK_OBJ,
                                            SAPDB_ToString (hObject), SAPDB_ToString (rc));
        return false;
    }

    return true;
}

#endif