/*
 *  Copyright (C) 2002 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla.h"
#include "embed.h"
#include "glade.h"
#include "prefs_strings.h"
#include "eel-gconf-extensions.h"
#include "state.h"
#include "xlink.h"
#include "context.h"
#include "js_console.h"
#include "toolbar.h"
#include "misc_gui.h"

#include <string.h>
#include <stdlib.h>
#include <gtk/gtktext.h>
#include <gtk/gtkrc.h>
#include <gtk/gtkstyle.h>
#include <gtk/gtktoolbar.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-dock.h>
#include <libgnomeui/gtkpixmapmenuitem.h>

typedef struct
{
	GtkWidget *dialog;

	/* menubar */
	GtkWidget *menubar_dockitem;
	GtkWidget *show_menubar;
	GtkWidget *show_toolbar;
	GtkWidget *show_entry;
	GtkWidget *show_all;
	GtkWidget *show_errors;
	GtkWidget *show_warnings;
	GtkWidget *show_messages;
	GtkWidget *cut;
	GtkWidget *copy;
	GtkWidget *paste;
	GtkWidget *selectall;

	/* toolbar */
	GtkWidget *toolbar;
	GtkWidget *toolbar_dockitem;
	GtkWidget *all;
	GtkWidget *errors;
	GtkWidget *warnings;
	GtkWidget *messages;

	/* entrybar */
	GtkWidget *entrybar_dockitem;
	GtkWidget *evaluate_entry;

	/* content */
	GtkWidget *text;

	/* associated embed for doing js evaluations in */
	GaleonEmbed *embed;
} JSConsole;

/**
 * Msg: this structure contains information for a JavaScript message
 */
typedef struct
{
	gchar *type;     /* error type string, for example Error  */
	gchar *text;     /* the error message itself              */
	guint column;    /* the column that should be highlighted */
	GdkColor *color; /* the color used for the type string    */
} Msg;

/* function prototypes */
static GtkWidget *js_console_edit_find_focus (void);
static void js_console_appearance_menu_cb (GtkRadioMenuItem *menuitem, JSConsole *dummy);
static void js_console_append_msg (Msg *msg, gpointer data);
static void js_console_free_msg (Msg *msg, gpointer data);
static void js_console_filter_except (gint show);
static void js_console_clear (void);
gboolean js_console_delete_cb (GtkWidget *window, GdkEventAny *event,
		               JSConsole *dummy);
void js_console_close_cb (GtkWidget *widget, JSConsole *dummy);
void js_console_show_menubar_cb (GtkCheckMenuItem *widget, JSConsole *dummy);
void js_console_show_toolbar_cb (GtkCheckMenuItem *widget, JSConsole *dummy);
void js_console_show_entry_cb (GtkCheckMenuItem *widget, JSConsole *dummy);
gboolean js_console_edit_expose_event_cb (GtkWidget *editmenu, GdkEvent *event,
				          GaleonWindow *window);
void js_console_cut_cb (GtkMenuItem *menuitem, JSConsole *dummy);
void js_console_copy_cb (GtkMenuItem *menuitem, JSConsole *dummy);
void js_console_paste_cb (GtkMenuItem *menuitem, JSConsole *dummy);
void js_console_select_all_cb (GtkMenuItem *menuitem, JSConsole *dummy);
void js_console_clear_clicked_cb (GtkWidget *widget, JSConsole *dummy);
void js_console_toggled_cb (GtkRadioButton *widget, JSConsole *dummy);
gboolean js_console_toolbar_button_press_event_cb (GtkWidget *widget,
						   GdkEventButton *event,
						   JSConsole *dummy);
void js_console_eval_clicked_cb (GtkWidget *widget, JSConsole *dummy);
gboolean js_console_text_button_press_event_cb (GtkWidget *entry,
				                GdkEventButton *event,
				                JSConsole *dummy);
void js_console_show_menu_cb (GtkRadioMenuItem *menuitem, JSConsole *dummy);

/* colors for the console messages */
#define ERROR_COLOR   "#FF0000" /* red    */
#define WARNING_COLOR "#BDAE7B" /* orange */
#define MESSAGE_COLOR "#006400" /* green  */

/* globals */
static JSConsole *console  = NULL; /* the global console structure           */

static GdkColor  *bg       = NULL; /* text background color, from gtk theme  */
static GdkColor  *fg       = NULL; /* text foreground color, from gtk theme  */
static GdkColor  *err_clr  = NULL; /* color used in text widget for errors   */
static GdkColor  *wrng_clr = NULL; /* color used in text widget for warnings */
static GdkColor  *msg_clr  = NULL; /* color used in text widget for messages */

static GdkFont   *font     = NULL; /* font used in the text widget           */

static gboolean  virgin    = TRUE; /* has the dialog been fu^H^Hused before  */

static GList     *all      = NULL; /* list of all Msg structures             */
static GList     *errors   = NULL; /* list of error Msg structures           */
static GList     *warnings = NULL; /* list of warnings Msg structures        */
static GList     *messages = NULL; /* list of messages Msg structures        */

/**
 * js_console_init: initialize javascript console
 */
void
js_console_init (void)
{
	GladeXML *gxml;
	GtkStyle *style;
	GtkWidget *tmp;
	GtkReliefStyle button_relief =
		(gnome_preferences_get_toolbar_relief_btn () ?
		 GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);
	
	/* this should be called just once */
	g_assert (console == NULL);

	/* allocate a new console */
	console = g_new0 (JSConsole, 1);

	/* build dialog */
	gxml = glade_widget_new ("galeon.glade", 
				 "js_console",
				 &(console->dialog), console);
	
	/* store widget pointers in JSConsole structure */
	/* menubar */
	console->menubar_dockitem = glade_xml_get_widget
		(gxml, "js_console_menubar_dockitem");
	console->cut          = glade_xml_get_widget 
		(gxml, "js_console_cut");
	console->copy         = glade_xml_get_widget 
		(gxml, "js_console_copy");
	console->paste        = glade_xml_get_widget 
		(gxml, "js_console_paste");
	console->selectall    = glade_xml_get_widget 
		(gxml, "js_console_select_all");
	console->show_menubar = glade_xml_get_widget 
		(gxml, "js_console_show_menubar");
	console->show_toolbar = glade_xml_get_widget 
		(gxml, "js_console_show_toolbar");
	console->show_entry   = glade_xml_get_widget 
		(gxml, "js_console_show_entry");
	console->show_all     = glade_xml_get_widget
		(gxml, "js_console_show_everything");
	console->show_errors  = glade_xml_get_widget
		(gxml, "js_console_only_errors");
	console->show_warnings= glade_xml_get_widget
		(gxml, "js_console_only_warnings");
	console->show_messages= glade_xml_get_widget
		(gxml, "js_console_only_messages");

	/* toolbar */
	console->toolbar_dockitem = glade_xml_get_widget
		(gxml, "js_console_toolbar_dockitem");
	console->toolbar      = glade_xml_get_widget
		(gxml, "js_console_toolbar");
	console->all          = glade_xml_get_widget 
		(gxml, "js_console_all_radio");
	console->errors       = glade_xml_get_widget 
		(gxml, "js_console_errors_radio");
	console->warnings     = glade_xml_get_widget 
		(gxml, "js_console_warnings_radio");
	console->messages     = glade_xml_get_widget 
		(gxml, "js_console_messages_radio");

	/* entrybar */
	console->entrybar_dockitem = glade_xml_get_widget
		(gxml, "js_console_eval_dockitem");
	console->evaluate_entry = glade_xml_get_widget
		(gxml, "js_console_entry");

	/* content */
	console->text         = glade_xml_get_widget 
		(gxml, "js_console_text");

	/* setup the menubar */
	toolbar_set_dock_style (console->menubar_dockitem, TOOLBAR_TYPE_MENUBAR,
				0);
	
	/* setup the main toolbar */
	toolbar_set_dock_style (console->toolbar_dockitem, TOOLBAR_TYPE_TOOLBAR,
			        eel_gconf_get_integer
					(CONF_TOOLBAR_JS_CONSOLE_STYLE));

	/* setup the evaluate toolbar */
	toolbar_set_dock_style (console->entrybar_dockitem, TOOLBAR_TYPE_NONE,
				0);

	/* setup the "Evaluate" button */
	tmp = glade_xml_get_widget
		(gxml, "js_console_eval_button");
	gtk_button_set_relief (GTK_BUTTON (tmp), button_relief);
	GTK_WIDGET_UNSET_FLAGS (tmp, GTK_CAN_FOCUS);

	/* get the GtkStyle from the current theme */
	style = gtk_rc_get_style (console->text);
	if (style == NULL)
	{
		style = gtk_style_new ();
	}
	else
	{
		gtk_style_ref (style);
	}

	/* get fore & background colors */
	bg = &(style->base[GTK_STATE_NORMAL]);
	fg = &(style->fg[GTK_STATE_NORMAL]);

	/* get font */
	font = style->font;

	/* allocate colors */
	err_clr = gdk_color_copy (bg);
	gdk_color_parse (ERROR_COLOR, err_clr);
	wrng_clr = gdk_color_copy (bg);
	gdk_color_parse (WARNING_COLOR, wrng_clr);
	msg_clr = gdk_color_copy (bg);
	gdk_color_parse (MESSAGE_COLOR, msg_clr);

	/* we dont need the style anymore */
	gtk_style_unref (style);

	/* reset bar states (FIXME they should be restored) */
	eel_gconf_set_boolean (CONF_STATE_JS_CONSOLE_SHOW_MENUBAR, TRUE);
	eel_gconf_set_boolean (CONF_STATE_JS_CONSOLE_SHOW_TOOLBAR, TRUE);
	eel_gconf_set_boolean (CONF_STATE_JS_CONSOLE_SHOW_ENTRYBAR, TRUE);

	/* load saved filtering state */
	misc_gui_gtk_radio_button_set (GTK_RADIO_BUTTON (console->all),
	      	      eel_gconf_get_integer (CONF_STATE_JS_CONSOLE_SHOW));
	
	/* dispose gxml object */
	gtk_object_unref (GTK_OBJECT (gxml));
}

/**
 * js_console_exit: shutdown javascript console
 */
void
js_console_exit (void)
{
	/* if visible, save states & hide */
	if (GTK_WIDGET_VISIBLE (console->dialog))
	{
		state_save_window_state (console->dialog->window, "js_console");
		state_save_dock_layout (console->dialog, "js_console");

		gtk_widget_hide (console->dialog);
	}
}

/**
 * js_console_show: show javascript console
 */
void
js_console_show (GaleonEmbed *embed)
{
	g_assert (console != NULL);

	/* update associated embed */
	console->embed = embed;

	if (GTK_WIDGET_VISIBLE (console->dialog))
	{
		/* window is already shown, raise */
		gdk_window_raise (console->dialog->window);
	}
	else
	{
		/* window isnt shown */
		if (virgin)
		{
			state_load_window_state (console->dialog, "js_console");
			state_load_dock_layout (console->dialog, "js_console");
			/* some XXX */
			virgin = FALSE;
		}
	
		/* show the console */
		gtk_widget_show (console->dialog);
	}
}

/**
 * js_console_append: append a message to the js console
 */
void
js_console_append (const gchar *text, JSConsoleMessageType type, guint column)
{
	gint show = eel_gconf_get_integer (CONF_STATE_JS_CONSOLE_SHOW);
	Msg *msg;

	/* console should be initialized */
	g_assert (console != NULL);

	/* allocate new Msg */
	msg = g_new0 (Msg, 1);

	/* set the column error point */
	msg->column = column;

	/* set the error text */
	if (text[strlen (text) - 1] != '\n')
	{
		/* no newline at the end of the text, append one */
		msg->text = g_strconcat (text, "\n", NULL);
	}
	else
	{
		msg->text = g_strdup (text);
	}
	
	/* set the type text & color */
	switch (type)
	{
	case JS_CONSOLE_ERROR:
		/* this is an error, set string, color, and append
		 * to the list of errors */
		msg->type = g_strdup (_("Error: "));
		msg->color = err_clr;
		errors = g_list_append (errors, msg);
		break;
	case JS_CONSOLE_WARNING:
		/* this is an warning, set string, color, and append
		 * to the list of warnings */
		msg->type = g_strdup (_("Warning: "));
		msg->color = wrng_clr;
		warnings = g_list_append (warnings, msg);
		break;
	case JS_CONSOLE_MESSAGE:
		/* this is an message, set string, color, and append
		 * to the list of messages */
		msg->type = g_strdup ("");
		msg->color = msg_clr;
		messages = g_list_append (messages, msg);
		break;
	default:
		/* shouldn't get here */
		g_warning ("unknown js message type\n");
		break;
	}

	/* append it to the list of all messages */
	all = g_list_append (all, msg);

	/* if it isnt currently filtered, show the message */
	if (show == 0 || show == (gint) type)
	{
		js_console_append_msg (msg, NULL);
	}
}

/**
 * js_console_delete_cb: close dialog
 */
gboolean
js_console_delete_cb (GtkWidget *window, GdkEventAny *event,
		      JSConsole *dummy)
{
	/* save states, and hide */
	js_console_exit ();

	/* do not actually destroy the dialog */
	return TRUE;
}

/**
 * js_console_close_cb: close in the menu clicked
 */
void
js_console_close_cb (GtkWidget *widget, JSConsole *dummy)
{
	/* save states, and hide */
	js_console_exit ();
}

/**
 * js_console_show_menubar_cb: show/hide menubar
 */
void
js_console_show_menubar_cb (GtkCheckMenuItem *widget, JSConsole *dummy)
{
	if (widget->active)
	{
		/* show bar */
		gtk_widget_show (console->menubar_dockitem);
	}
	else
	{
		/* hide bar */
		gtk_widget_hide (console->menubar_dockitem);
		gtk_widget_queue_resize (console->dialog);
	}

	/* set state */
	eel_gconf_set_boolean (CONF_STATE_JS_CONSOLE_SHOW_MENUBAR,
			       widget->active);
}

/**
 * js_console_show_toolbar_cb: show/hide toolbar
 */
void
js_console_show_toolbar_cb (GtkCheckMenuItem *widget, JSConsole *dummy)
{
	if (widget->active)
	{
		/* show bar */
		gtk_widget_show (console->toolbar_dockitem);
	}
	else
	{
		/* hide bar */
		gtk_widget_hide (console->toolbar_dockitem);
		gtk_widget_queue_resize (console->dialog);
	}

	/* set state */
	eel_gconf_set_boolean (CONF_STATE_JS_CONSOLE_SHOW_TOOLBAR,
			       widget->active);
}

/**
 * js_console_show_entry_cb: show/hide entrybar
 */
void
js_console_show_entry_cb (GtkCheckMenuItem *widget, JSConsole *dummy)
{
	if (widget->active)
	{
		/* show bar */
		gtk_widget_show (console->entrybar_dockitem);
	}
	else
	{
		/* hide bar */
		gtk_widget_hide (console->entrybar_dockitem);
		gtk_widget_queue_resize (console->dialog);
	}

	/* set state */
	eel_gconf_set_boolean (CONF_STATE_JS_CONSOLE_SHOW_ENTRYBAR,
			       widget->active);
}

/*
 * js_console_edit_find_focus: find which widget is currently in focus
 */
static GtkWidget *
js_console_edit_find_focus (void)
{
	if (GTK_WIDGET_HAS_FOCUS (console->text))
	{
		/* the text widget is in focus */
		return console->text;
	}

	if (GTK_WIDGET_HAS_FOCUS (console->evaluate_entry))
	{
		/* the evaluation entry is in focus */
		return console->evaluate_entry;
	}

	/* no widget is in focus */
	return NULL;
}

/**
 * js_console_edit_expose_event_cb: edit menu exposed
 */
gboolean
js_console_edit_expose_event_cb (GtkWidget *editmenu, 
				 GdkEvent *event,
				 GaleonWindow *window)
{
	gboolean can_cut, can_copy, can_paste, can_selectall;
	GtkWidget *focus_widget;
	
	can_cut       = FALSE;
	can_copy      = FALSE;
	can_paste     = FALSE;
	can_selectall = FALSE;

	/* find widget in focus */
	focus_widget = js_console_edit_find_focus ();
	
	if (focus_widget != NULL && GTK_IS_EDITABLE (focus_widget))
	{
		GtkEditable *editable = GTK_EDITABLE (focus_widget);

		/* get properties from the GtkEditable */
		if (editable->has_selection && editable->editable)
		{
			can_cut = TRUE;
		}
		if (editable->has_selection)
		{
			can_copy = TRUE;
		}
		if (editable->editable && galeon_x_clipboard_present()) 
		{
			can_paste = TRUE;
		}
		
		/* we can select everything, as long as we found
		 * a widget (which we did, at this point) */
		can_selectall = TRUE;
	}

	/* setup sensitivity according to properties */
	gtk_widget_set_sensitive (console->cut, can_cut);
	gtk_widget_set_sensitive (console->copy, can_copy);
	gtk_widget_set_sensitive (console->paste, can_paste);
	gtk_widget_set_sensitive (console->selectall, can_selectall);

	return FALSE;
}

/**
 * js_console_cut_cb: Cut selected text into clipboard
 */
void
js_console_cut_cb (GtkMenuItem *menuitem, JSConsole *dummy)
{
	GtkEditable *editable = GTK_EDITABLE (js_console_edit_find_focus ());

	gtk_editable_cut_clipboard (editable);
}

/**
 * js_console_copy_cb: Copy selected text into clipboard
 */
void
js_console_copy_cb (GtkMenuItem *menuitem, JSConsole *dummy)
{
	GtkEditable *editable = GTK_EDITABLE (js_console_edit_find_focus ());

	gtk_editable_copy_clipboard (editable);
}

/**
 * js_console_paste_cb: Paste selected text into clipboard
 */
void
js_console_paste_cb (GtkMenuItem *menuitem, JSConsole *dummy)
{
	GtkEditable *editable = GTK_EDITABLE (js_console_edit_find_focus ());

	gtk_editable_paste_clipboard (editable);
}

/**
 * js_console_select_all_cb: select all text in the current editable
 */
void
js_console_select_all_cb (GtkMenuItem *menuitem, JSConsole *dummy)
{
	GtkEditable *editable = GTK_EDITABLE (js_console_edit_find_focus ());

	gtk_editable_select_region (editable, 0, -1);
}

/**
 * js_console_clear_clicked_cb: clear clicked
 */
void
js_console_clear_clicked_cb (GtkWidget *widget, JSConsole *dummy)
{
	/* clear the text widget */
	js_console_clear ();
}

/**
 * js_console_toggled_cb: one of the 4 radiobuttons clicked
 */
void
js_console_toggled_cb (GtkRadioButton *widget, JSConsole *dummy)
{
	/* filter evyerhing except the clicked button */
	js_console_filter_except (misc_gui_gtk_radio_button_get (widget));
}

/**
 * js_console_filter_except: filter everything except (show)
 */
static void
js_console_filter_except (gint show)
{
	/* do not filter unless we're not already doing so */
	static gboolean block = FALSE;
	if (block) return;
	block = TRUE;
	
	/* save state */
	eel_gconf_set_integer (CONF_STATE_JS_CONSOLE_SHOW, show);

	/* clear the text widget */
	gtk_editable_delete_text (GTK_EDITABLE (console->text), 0, -1);

	/* do the requested filtering */
	switch (show)
	{
	case 0:
		/* show all messages of this type */
		g_list_foreach (all, (GFunc) js_console_append_msg, NULL);

		/* update the ui */
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (console->show_all), TRUE);
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (console->all), TRUE);
		break;
	case 1:
		/* show all messages of this type */
		g_list_foreach (errors, (GFunc) js_console_append_msg, NULL);

		/* update the ui */
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (console->show_errors), TRUE);
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (console->errors), TRUE);
		break;
	case 2:
		/* show all messages of this type */
		g_list_foreach (warnings, (GFunc) js_console_append_msg, NULL);

		/* update the ui */
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (console->show_warnings), TRUE);
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (console->warnings), TRUE);
		break;
	case 3:
		/* show all messages of this type */
		g_list_foreach (messages, (GFunc) js_console_append_msg, NULL);

		/* update the ui */
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (console->show_messages), TRUE);
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (console->messages), TRUE);
		break;
	}

	/* unblock */
	block = FALSE;
}

/**
 * js_console_toolbar_button_press_event_cb: button pressed on the toolbar,
 * popup toolbar appearance context
 */
gboolean
js_console_toolbar_button_press_event_cb (GtkWidget *widget, 
					  GdkEventButton *event,
					  JSConsole *dummy)
{
	if (event->button == 3)
	{
		/* 3rd button clicked, display contextmenu */
		GtkMenu *popup = GTK_MENU (gtk_menu_new ());
		GSList *radiogroup = NULL;
		gint style = 0;
	
		/* get current state */
		style = eel_gconf_get_integer (CONF_TOOLBAR_JS_CONSOLE_STYLE);
	
		/* build menu */
		context_menu_add_radio_item 
			(popup, _("_Icons only"),
			 js_console_appearance_menu_cb, console,
			 (style == GTK_TOOLBAR_ICONS), &radiogroup);
		context_menu_add_radio_item 
			(popup, _("_Text only"),
			 js_console_appearance_menu_cb, console,
			 (style == GTK_TOOLBAR_TEXT), &radiogroup);
		context_menu_add_radio_item 
			(popup, _("Text _under icons"),
			 js_console_appearance_menu_cb, console,
			 (style == GTK_TOOLBAR_BOTH), &radiogroup);
	
		/* run */
		gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup), NULL,
						 NULL, event, NULL);

		/* dispose menu object */
		gtk_object_unref (GTK_OBJECT (popup));
	}

	return TRUE;
}

/**
 * js_console_appearance_menu_cb: an item clicked in the tb appearance context
 */
static void
js_console_appearance_menu_cb (GtkRadioMenuItem *menuitem, JSConsole *dummy)
{
	/* get the selected menuitem */
	GSList *list = g_slist_reverse
		(g_slist_copy (gtk_radio_menu_item_group (menuitem)));
	gint index = g_slist_index (list, menuitem);
	
	g_slist_free (list);

	/* set the toolbar style */
	gtk_toolbar_set_style (GTK_TOOLBAR (console->toolbar), index);
	
	/* save the state */
        eel_gconf_set_integer (CONF_TOOLBAR_JS_CONSOLE_STYLE, index);
	
	/* ensure the toolbar has the right size */
	gtk_widget_queue_resize (console->dialog);
}

/**
 * js_console_eval_clicked_cb: Evaluate js code in entry
 */
void
js_console_eval_clicked_cb (GtkWidget *widget, JSConsole *dummy)
{
	gchar *text;

	/* get text from entry */
	text = gtk_editable_get_chars
		(GTK_EDITABLE (console->evaluate_entry), 0, -1);

	/* check whether the associated embed still exists, if not, pick
	 * a random one */
	if (!embed_exists (console->embed))
	{
		if (!all_embeds)
		{
			/* no embeds are present, so we cant do anything. */
			return;
		}

		/* update the associated embed */
		console->embed = all_embeds->data;
	}

	/* evaluate javascript in selected embed */
	mozilla_evaluate_js (console->embed, text);

	/* free the retreived text */
	g_free (text);
}

/**
 * js_console_text_button_press_event_cb: popup textwidget contextmenu
 */
gboolean
js_console_text_button_press_event_cb (GtkWidget *entry,
				       GdkEventButton *event,
				       JSConsole *dummy)
{
	if (event->button == 3)
	{
		/* 3rd button clicked, display contextmenu */
		gint show = eel_gconf_get_integer (CONF_STATE_JS_CONSOLE_SHOW);
		GtkEditable *editable = GTK_EDITABLE (entry);
		GSList *group = NULL;
		GtkWidget *menu;
		gint action;

		/* context menu */
		static GnomeUIInfo menu_uiinfo[] =
		{
			GNOMEUIINFO_ITEM_STOCK (N_("Copy"),
					NULL, NULL, GNOME_STOCK_MENU_COPY),
			GNOMEUIINFO_SEPARATOR,
			GNOMEUIINFO_ITEM_STOCK (N_("Select all"),
					NULL, NULL, GNOME_STOCK_MENU_BLANK),
			GNOMEUIINFO_ITEM_STOCK (N_("Clear"),
					NULL, NULL, GNOME_STOCK_MENU_BLANK),
			GNOMEUIINFO_SEPARATOR,
			GNOMEUIINFO_TOGGLEITEM (N_("Show menubar"),
					NULL, NULL, GNOME_STOCK_MENU_BLANK),
			GNOMEUIINFO_SEPARATOR,
			GNOMEUIINFO_END
		};
	
		/* setup context menu */
		menu = misc_gui_new_popup_menu_lock_accels (menu_uiinfo);
		
		/* check whether there's a seleciotn, if that's not the case
		 * make the copy item unsensitivie */
		if (!editable->has_selection)
		{
			gtk_widget_set_sensitive (menu_uiinfo[0].widget, FALSE);
		}
		
		/* do not show the "Show menubar" widgets if the menubar is
		 * already shown */
		if (eel_gconf_get_boolean (CONF_STATE_JS_CONSOLE_SHOW_MENUBAR))
		{
			gtk_widget_hide (menu_uiinfo[5].widget);
			gtk_widget_hide (menu_uiinfo[6].widget);
		}

		/* add radiobuttons for filtering */
		context_menu_add_radio_item
			(GTK_MENU (menu), _("Show everything"),
			 js_console_show_menu_cb, console,
			 (show == 0),
			 &group);
		context_menu_add_radio_item
			(GTK_MENU (menu), _("Show only errors"),
			 js_console_show_menu_cb, console,
			 (show == 1),
			 &group);
		context_menu_add_radio_item
			(GTK_MENU (menu), _("Show only warnings"),
			 js_console_show_menu_cb, console,
			 (show == 2),
			 &group);
		context_menu_add_radio_item
			(GTK_MENU (menu), _("Show only messages"),
			 js_console_show_menu_cb, console,
			 (show == 3),
			 &group);

		/* run */
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* do appropiate action */
		switch (action)
		{
		case 0:
			/* copy selection */
			gtk_editable_copy_clipboard (editable);
			break;
		case 2:
			/* select everything */
			gtk_editable_select_region (editable, 0, -1);
			break;
		case 3:
			/* clear the console */
			js_console_clear ();
			break;
		case 5:
			/* show the menubar */
			gtk_widget_show (console->menubar_dockitem);

			eel_gconf_set_boolean (CONF_STATE_JS_CONSOLE_SHOW_MENUBAR,
					       TRUE);

			GTK_CHECK_MENU_ITEM (console->show_menubar)->active = TRUE;

			break;
		}

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		
		/* gtk+ goofiness */
		gtk_signal_emit_stop_by_name (GTK_OBJECT (entry),
				              "button-press-event");

		return TRUE;
	}

	return FALSE;
}

/**
 * js_console_append_msg: append a message to the text widget
 */
static void
js_console_append_msg (Msg *msg, gpointer data)
{
	/* append message type text to console, with appropiate color */
	gtk_text_insert (GTK_TEXT (console->text),
			 font, msg->color, bg, msg->type,
			 strlen (msg->type));

	/* check whether we need to highlight something */
	if ((gint) msg->column > 0)
	{
		/* yes, we do */
		gchar *tmp;

		/* insert the first chunk of text normally */
		tmp = g_strndup (msg->text, msg->column);
		gtk_text_insert (GTK_TEXT (console->text),
				 font, fg, bg, tmp, strlen (tmp));
		g_free (tmp);

		/* now insert the highlighted chunk with appriopate
		 * highlighting (we use wrng_clr as background, which is
		 * red) */
		tmp = g_strndup (msg->text + msg->column, 1);
		gtk_text_insert (GTK_TEXT (console->text),
				 font, bg, wrng_clr, tmp, strlen (tmp));
		g_free (tmp);

		/* if there's anything left of the message, append that too */	
		if (strlen (msg->text) > msg->column)
		{
			gtk_text_insert (GTK_TEXT (console->text),
					 font, fg, bg,
					 msg->text + msg->column + 1,
					 strlen (msg->text + msg->column + 1));
		}
	}
	else
	{
		/* no, we don't. just insert the text without any more fuss */
		gtk_text_insert (GTK_TEXT (console->text),
				 font, fg, bg, msg->text,
				 strlen (msg->text));
	}
}

/**
 * js_console_free_msg: free a Msg struct 
 */
static void
js_console_free_msg (Msg *msg, gpointer data)
{
	g_free (msg->text);
	g_free (msg->type);
	msg->column = 0;
	g_free (msg);
}

/**
 * js_console_clear: clear console
 */
static void
js_console_clear (void)
{
	/* clear the console */
	gtk_editable_delete_text (GTK_EDITABLE (console->text), 0, -1);

	/* clear error message list */
	g_list_free (errors);
	errors = NULL;

	/* clear warning message list */
	g_list_free (warnings);
	warnings = NULL;

	/* clear message list */
	g_list_free (messages);
	messages = NULL;

	/* free all Msg structures */
	g_list_foreach (all, (GFunc) js_console_free_msg, NULL);
	g_list_free (all);
	all = NULL;
}

/**
 * js_console_show_menu_cb: some show only .. menuitem clicked
 */ 
void
js_console_show_menu_cb (GtkRadioMenuItem *menuitem, JSConsole *dummy)
{
	/* filter everything except the selected radioitem */
	js_console_filter_except (misc_gui_radio_menuitem_index (menuitem));
}
