//<copyright>
// 
// Copyright (c) 1994
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
// original version:
// Copyright (c) 1987, 1988, 1989, 1990, 1991 Stanford University
// Copyright (c) 1991 Silicon Graphics, Inc.
// 
//</copyright>


//<file>
//
// Name:        labels.C
//
// Purpose:     interface to UnderlinedLabel
//
// Created:      9 Sep 94   Michael Pichler
//
// $Id: labels.C,v 1.5 1996/10/09 14:17:05 jschipf Exp $
//
// $Log: labels.C,v $
// Revision 1.5  1996/10/09 14:17:05  jschipf
// add multilinugal support
//
// Revision 1.4  1996/09/02 15:30:17  bmarsch
// Removed memory leak
//
// Revision 1.3  1996/02/15 17:40:08  bmarsch
// Includes from OS/ were moved to hyperg/OS/
//
// Revision 1.2  1996/01/18 16:28:06  bmarsch
// Removed warnings
//
//</file>


#include <ctype.h>
#include <string.h>

#include "labels.h"

#include <InterViews/brush.h>
#include <InterViews/canvas.h>
#include <InterViews/color.h>
#include <InterViews/font.h>
#include <InterViews/hit.h>

#include <IV-look/kit.h>
#include <hyperg/OS/string.h>

#ifdef iv_nls
#include <NLS/wstring.h>
#endif


UnderlinedLabel::UnderlinedLabel (
  const char* str,
#ifndef iv_nls
      const Font* font,
#else
      const FontSet* fontset,
#endif
  const Color* color,
  int from, int to
)
#ifndef iv_nls
:Label (str, font, color)
#else
:WLabel (WString(str),fontset,color)
#endif
{
  if (from < 0)
    from = 0;
  from_ = from;
  to_ = to;

  brush_ = new Brush (0.0);  // thin brush
  Resource::ref (brush_);
}

#ifndef iv_nls
UnderlinedLabel::UnderlinedLabel(
  const char* str,
  const Font* font, const Color* color, char underline
):Label (str, font, color)
{
  // convert to lower case
  underline = tolower(underline);
  char* s = new char[strlen(str) + 1];
  strcpy(s, str);
  char* c = (char*) s;
  while (*c) *c = tolower(*c), c++;

  char* un = strchr(s, underline);
  if (un) {
    from_ = un - s;
    to_ = from_ + 1;
  }
  else
    from_ = to_ = 0;

  delete[] s;

  brush_ = new Brush(0.0);
  Resource::ref(brush_);
}
#else
UnderlinedLabel::UnderlinedLabel(
  const char* str,
  const FontSet* fontset, const Color* color, WChar underline
):WLabel (WString(str),fontset,color)
{
  // convert to lower case
  underline = tolower(underline.charcode());
  WString s(str);

  int idx=-1;
  for (int i=0;i<s.length();i++)
  {
    WChar wc=tolower(s[i].charcode());
    if (wc.charcode()==underline.charcode())
    {
      idx=i;
      break;
    }
  }

  if (idx!=-1)
  {
    from_ = idx;
    to_ = from_ + 1;
  }
  else
    from_ = to_ = 0;

  brush_ = new Brush(0.0);
  Resource::ref(brush_);
}
#endif

UnderlinedLabel::~UnderlinedLabel ()
{
  Resource::unref (brush_);
}


void UnderlinedLabel::draw (Canvas* c, const Allocation& a) const
{
  int len = string ()->length ();


#ifndef iv_nls
  Label::draw (c, a);
#else
  WLabel::draw (c,a);
#endif

  int to = to_;
  if (to < 0 || to > len)
    to = len;  // underline whole label

  if (from_ < to)
  {
    Coord des = descent ();
    Coord base = a.y () - des/2;
    Coord left = a.x () - 0.2;
    const Coord* cw = charWidths ();

    int i = from_;  // assert: from >= 0
    while (i--)
      left += *cw++;
    Coord right = left;
    i = to - from_;
    while (i--)
      right += *cw++;

    c->line (left, base, right, base, color (), brush_);
  }
} // draw


/*** kit for underlined labels ***/

UnderlinedLabel* UnderlinedLabel::udlLabel (
  WidgetKit& kit, const char* str,
  const char* style, const char* alias)
{
  if (style)
    kit.begin_style (style);
  if (alias)
    kit.alias (alias);

#ifndef iv_nls
  UnderlinedLabel* ulabel = new UnderlinedLabel (str, kit.font (), kit.foreground (), 0, -1);
#else
  UnderlinedLabel* ulabel = new UnderlinedLabel (str, kit.fontset (), kit.foreground (), 0, -1);
#endif
  if (style)
    kit.end_style ();

  return ulabel;
}


UnderlinedLabel* UnderlinedLabel::hotLabel (
  WidgetKit& kit, const char* str,
  const char* style, const char* alias,
  int no, int num)
{
  if (style)
    kit.begin_style (style);
  if (alias)
    kit.alias (alias);

#ifndef iv_nls
  UnderlinedLabel* ulabel = new UnderlinedLabel (str, kit.font (), kit.foreground (), no, no+num);
#else
  UnderlinedLabel* ulabel = new UnderlinedLabel (str, kit.fontset (), kit.foreground (), no, no+num);
#endif
  if (style)
    kit.end_style ();

  return ulabel;
}

UnderlinedLabel* UnderlinedLabel::hotLabel(
  WidgetKit& kit, const char* str, char underline,
  const char* style, const char* alias
)
{
  if (style)
    kit.begin_style(style);
  if (alias)
    kit.alias(alias);

#ifndef iv_nls
  UnderlinedLabel* ulabel = new UnderlinedLabel(
    str, kit.font(), kit.foreground(), underline
  );
#else
  UnderlinedLabel* ulabel = new UnderlinedLabel(
    str, kit.fontset(), kit.foreground(), underline
  );
#endif

  if (style)
    kit.end_style();

  return ulabel;
}
