//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        vrmledit.C
//
// Purpose:     implementation of editing functions of class VRMLScene
//
// Created:     14 Feb 96   Alexander Nussbaumer
//
// Changed:     24 Feb 97   Alexander Nussbaumer
//
// Changed:     23 Apr 97   Michael Pichler
//
// $Id: vrmledit.C,v 1.12 1997/04/30 10:48:25 mpichler Exp $
//
//</file>


#include "vrmlscene.h"
#include "scene3d.h"
#include "arrays.h"
#include "scenewin.h"
#include "vecutil.h"
#include "editdlg.h"
#include "camera.h"

#include <vrml/QvChildList.h>
#include <vrml/QvCone.h>
#include <vrml/QvCube.h>
#include <vrml/QvCylinder.h>
#include <vrml/QvDB.h>
#include <vrml/QvIndexedFaceSet.h>
#include <vrml/QvMaterial.h>
#include <vrml/QvMatrixTransform.h>
#include <vrml/QvNode.h>
#include <vrml/QvGroup.h>
#include <vrml/QvPerspectiveCamera.h>
#include <vrml/QvOrthographicCamera.h>
#include <vrml/QvRotation.h>
#include <vrml/QvScale.h>
#include <vrml/QvSeparator.h>
#include <vrml/QvSphere.h>
#include <vrml/QvState.h>
#include <vrml/QvSwitch.h>
#include <vrml/QvSFString.h>
#include <vrml/QvString.h>
#include <vrml/QvTransform.h>
#include <vrml/QvTranslation.h>
#include <vrml/QvTexture2.h>
#include <vrml/QvTexture2Transform.h>
#include <vrml/QvWWWAnchor.h>
#include <vrml/QvWWWInline.h>

#include <ge3d/ge3d.h>

#include <iostream.h>
#include <math.h>
#include <stdlib.h>

#include <hyperg/hyperg/message.h>
#include <hyperg/utils/verbose.h>
#include <hyperg/utils/str.h>

static const float epsilon = 0.01;


/****** general functions *******/


// return +/-1 for positive/negative no., 0 for 0.0

static float fsgnf (float f)
{
  if (f == 0)
    return 0.0;
  else
    return f < 0 ? -1.0 : 1.0;
} // fsgnf


/******** Manipulation functions *********/

// translate selected geometric node

void VRMLScene::translateSelected (const vector3D& trans)
{
  // TODO move condition of topseparator into Hg3dinputhandler (also rotation and scaling)
  if (!scene_->selectedNode () || sel_istopseparator_)   
    return;

  if (!transformation_allowed_)
    makeTransformForSelected ();

  // picking for constraints
  vector3D translate = trans;
  if (constraints_)
    pickTranslation (translate);

  sel_transform_->translation.setDefault (0);
  vector3D* translation = (vector3D*) sel_transform_->translation.value;
  inc3D (*translation, translate);

  ge3d_push_matrix ();
  sel_transform_->build (0);
  ge3d_pop_matrix ();

  updateTransformDialog (1);  // update translation

} // translateSelected


// rotate selected geometric node

void VRMLScene::rotateSelected (vector3D& axis, float angle)/* const */
{
  if (!scene_->selectedNode () || sel_istopseparator_)
    return;

  if (!transformation_allowed_)
    makeTransformForSelected ();

  sel_transform_->rotation.setDefault (0);

 // take actual transform (relative rotation)
  vector3D *qvaxis = (vector3D*) sel_transform_->rotation.axis;
  float *qvangle = &sel_transform_->rotation.angle;
/*
  // take values from original transform at time of press (absolut rotation) 
  vector3D* qvaxis = (vector3D*) &rotbasisaxis_.x;
  float *qvangle = &rotbasisangle_;
*/
  // normalise interactive and qv rotation axis
  float temp = norm3D (axis);
  if ((temp != 1.0) && temp)
  { temp = 1.0/temp;
    scl3D (axis, temp);
  }

  temp = norm3D (*qvaxis);
  if ((temp != 1.0) && temp)
  { temp = 1.0/temp;
    scl3D (*qvaxis, temp);
  }

  vector3D resaxis;
  float resangle = 0;
  multiplyRotations (*qvaxis, *qvangle, axis, angle, resaxis, resangle);

  // for absolute rotation
  qvaxis = (vector3D*) sel_transform_->rotation.axis;
  qvangle = &sel_transform_->rotation.angle;

  // put result back to qv-stucture
  *qvaxis = resaxis;
  *qvangle = resangle;

  ge3d_push_matrix ();
  sel_transform_->build (NULL);
  ge3d_pop_matrix ();

  updateTransformDialog (2); // update rotation

} // rotateSelected


// stores the actual rotation when the mose button is pressed

void VRMLScene::storeRotation ()
{
  if (!sel_transform_)
    return;

  rotbasisaxis_.x = sel_transform_->rotation.axis[0];
  rotbasisaxis_.y = sel_transform_->rotation.axis[1];
  rotbasisaxis_.z = sel_transform_->rotation.axis[2];
  rotbasisangle_ = sel_transform_->rotation.angle;

} // storeRotation


// scale selected geometric node
// should have a constant minscale for magic 0.04

void VRMLScene::scaleSelected (const vector3D& scale)
{
  if (!scene_->selectedNode () || sel_istopseparator_)
    return;

  if (!transformation_allowed_)
    makeTransformForSelected ();

  vector3D scaling = scale;
  if ((constraints_) && (scaling.x >= 0) && (scaling.y >= 0) && (scaling.z >= 0))
    pickScale (scaling);

  sel_transform_->scaleFactor.setDefault (0);

  vector3D* qvscale = (vector3D*) sel_transform_->scaleFactor.value;

  if ((qvscale->x + scaling.x <= 0.04) || (qvscale->y + scaling.y <= 0.04) || (qvscale->z + scaling.z <= 0.04))
    return;  // 0.4 is a value of experience (floating exception can be caused with a lower value)

  inc3D (*qvscale, scaling);

  // rebuild only changes
  ge3d_push_matrix ();
  sel_transform_->build (NULL);
  ge3d_pop_matrix ();

  updateTransformDialog (3);

} // scaleSelected


/* xxx */

// calculates the object extent coordinates in the same way as in wrlbuild.C,
// but for groupnodes, for those the Transform node is inside, 

// TODO check this: ask for transform is afterwards
void VRMLScene::calculateObjectExtent ()
{
  if (!scene_->selectedNode ())
    return;

  if (!selTransformInsideGroup ())
  { sel_omin_ = scene_->selectedNode ()->omin_;
    sel_omax_ = scene_->selectedNode ()->omax_;
    return;
  }

  QvGroup* selectedgroup = (QvGroup*)scene_->selectedNode ();
  int n = selectedgroup->getNumChildren ();
  QvState state;
  QvNode* child;

  emptyBoundingbox (sel_omin_, sel_omax_);
  ge3dPushIdentity ();

  for (int i = 0; i < n; i++)
  { // TODO does not work for top separator with camera node
    child = selectedgroup->getChild (i);
    if ((child != sel_transform_) && (child != camswitch_))
      child->build (&state);
    if (child->hasextent_)
      extendBoundingbox (child->wmin_, child->wmax_, sel_omin_, sel_omax_);
  }

  ge3d_pop_matrix ();

} // calculateObjectExtent


// recalculates for every groupnode below node, if it has an extent

void VRMLScene::calculateHasExtent (QvGroup* node)
{
  QvNode* child;
  int hasextent = 0,
      i = node->getNumChildren ();

  while (i--)
  {
    child = node->getChild (i);
    if (child->isGroupNode ())
      calculateHasExtent ((QvGroup*)child);
    hasextent += child->hasextent_;
  }

  if (hasextent)
    node->hasextent_ = 1;
  else
    node->hasextent_ = 0;

} // calculateHasExtent


/****** transform Path to nodepointer ******/

// returns the parent node according to the path
// no check, if the node is referenced multiple
// implemented in vrmlscene.C

// QvGroup* VRMLScene::getParentByPath (const IntArray* patharr) const
// {
//   if (!patharr || !root_)
//     return NULL;

//   int depth = patharr->count ();
//   if (!depth)
//     return NULL;
//   depth--;

//   int val;
//   const int* path = patharr->data ();
//   QvNode* go = root_;

//   while (depth--)
//   {
//     val = *path++;

//     if (go && go->isGroupNode ()  && (val < ((QvGroup*) go)->getNumChildren ()))
//       go = ((QvGroup*) go)->getChild (val);
//     else
//     { go = NULL;
//       break;
//     }
//   }
//   return (QvGroup*) go;

// } // getParentByPath


// get selected node, its parent and the childnumber of the selected node from path
// return 0, if somthing is not ok

int VRMLScene::getSelectionByPath (
  const IntArray* path, 
  QvGroup*& parent, QvNode*& node, int& whichnode, int& pathsize
) const
{
  if (!path)  // check input
    return 0;

  pathsize = path->count ();  // path size is deph of selected object in the scene graph
  if (!pathsize)
    return 0;

  whichnode = (*path)[pathsize - 1];  // child number of selected node

  parent = getParentByPath (path);  // parent of selected node
  if (!parent)
    return 0;

  node = parent->getChild (whichnode);
  if (!node)
    return 0;

  return 1;

} // getSelectionByPath


// get selected node from a path; nearly the same as above

QvNode* VRMLScene::getSelectedNodeByPath (const IntArray* path)
{
  if (!path)
    return 0;

  int pathsize = path->count ();  // path size is deph of selected object in the scene graph
  if (!pathsize)
    return 0;

  QvGroup* parent = (QvGroup*)getParentByPath (path);  // parent of selected node
  if (!parent)
    return 0;

  return parent->getChild ((*path)[pathsize - 1]);

} // getSelectedNodeByPath



/**** functions to handle property nodes ****/

// search the appropriate property node (i.e. QvMaterial) to the selected node
// the search begins on the level of the selected node and goes up to the root

QvNode* VRMLScene::searchPropertyNode (int nodetype, IntArray** propertypath)
{
  if (propertypath)
    *propertypath = 0;
  const IntArray* patharray = scene_->selectedPath ();
  if (!patharray)
    return 0;

  IntArray* path = new IntArray;
  *path = *patharray;  // local copy of the selected path

  QvGroup* parent;
  int i;
  int depth = path->count ();
  while (depth--)  // search though the levels above
  { parent = getParentByPath (path);
    i = (*path)[depth];
    path->remove (depth);
    while (i--)  // search nodes above in the same level
      if (parent->getChild (i)->nodeType () == nodetype)  // property node found
      { if (propertypath)
        { path->append (i);
          *propertypath = path;  // returned path
        }
        else
          delete path;
        return parent->getChild (i);  // of the found property node
      }
  }

  delete path;
  return 0;  // nothing found

} // searchPropertyNode


// search deepest group node of nodetype containing selected node
// returns the group node, sets *propertypath

QvNode* VRMLScene::searchGroupNode (int nodetype, IntArray** propertypath)
{
/*
  if (propertypath)
    *propertypath = 0;
*/
  const IntArray* patharray = scene_->selectedPath ();
  if (!patharray)
    return 0;

  IntArray* path = new IntArray;
  *path = *patharray;  // local copy of the selected path

  QvGroup* parent;
  int i;
  int depth = path->count ();

  while (depth--)  // search though the levels above
  { parent = getParentByPath (path);
    i = (*path)[depth];
    if (parent->getChild (i)->nodeType ()  == nodetype)
    { if (propertypath)
        *propertypath = path;
      else 
        delete path;
      return parent->getChild (i);
    }
    path->remove (depth);
  }
  
  delete path;
  return 0;  // nothing found

} // searchGroupNode



// returns the number of geometric nodes, for which the property node (nodetype) is relevant
// if the property node is a group node (wwwanchor) the search does not stop at the node
// in the list

int VRMLScene::numberOfRelevantNodes (QvGroup* parent, int firstnode, int nodetype)
{
  int number = 0;

  for (int i = firstnode; i < parent->getNumChildren (); i++)
  { 
    if ((parent->getChild (i)->nodeType () == nodetype) && parent->getChild (i)->isGroupNode ())
      continue;  // for wwwanchors, geometries after the the anchor are influenced from the selected
    if (parent->getChild (i)->nodeType () == nodetype)
      return number;  // 
    if (parent->getChild (i) == scene_->selectedNode ())
      return ++number;
    if (parent->getChild (i)->isGeometryNode ())
      number++;
    if (parent->getChild (i)->isGroupNode ())
      number += numberOfRelevantNodes ((QvGroup*)(parent->getChild (i)), 0, nodetype);
  }
  return number;
  
} // numberOfRelevantNodes


// checks whether the relevant property for the selected node is also relevant for other geometric nodes

int VRMLScene::multipleRelevantOfProperty (int nodetype, const IntArray* proppath)
{
  IntArray* path = new IntArray; // TODO should not be initialized, but should have a value
  if (!proppath)
    searchPropertyNode (nodetype, &path); 
  else
  { *path = *proppath;
    delete proppath;
  }

  QvGroup* parent = nil;
  QvNode* node = nil;
  int whichnode,
      size;   
  getSelectionByPath (path, parent, node, whichnode, size);
  int number = numberOfRelevantNodes (parent, whichnode + 1, node->nodeType ());

  return (number > 1);

} // multipleRelevantOfProperty


/***** property node selection and dialog updating ******/

// select a node from a path and do all updates

void VRMLScene::selectNodeByPath (IntArray* path)
{
  QvNode* node = getSelectedNodeByPath (path);
  if (!node) 
    return;

  scene_->selectNode (node, path);  // takes over path
  updateEditContext ();

} // selectNodeByPath



// updateEditContext is called after a new node has been selected

void VRMLScene::updateEditContext ()
{ 
  if (!scene_->manipulationAllowed ())
    return;

  if (!scene_->selectedNode ())
    unselectPropertyNodes ();

  if (scene_->selectedPath () && (scene_->selectedPath ()->count () < 2))
    sel_istopseparator_ = 1;
  else
    sel_istopseparator_ = 0;
      

  updateMaterialDialog ();
  updateTextureDialog ();
  updateWWWAnchorDialog ();
  updateViewpointDialog ();
  searchTransform ();
  updateTransformDialog ();
  updateStructureViewerDialog ();  // the scene graph may have changed
  calculateObjectExtent ();

}  // selectPropertyNodes


// if node node is selected, clear property nodes

void VRMLScene::unselectPropertyNodes ()
{
  sel_transform_ = 0;
  sel_material_ = 0;
  sel_texture_ = 0;
  sel_texturetransform_ = 0;
  sel_wwwanchor_ = 0;
  sel_transform_insidegroup_ = 0;
  sel_istopseparator_ = 0;
  transformation_allowed_ = 0;
  init3D (sel_omin_, 0, 0, 0);
  init3D (sel_omax_, 1, 1, 1);

} // unselectPropertyNodes



/* Material */

// is also called from MateriaDialog and after a new selection of a geometric node

void VRMLScene::updateMaterialDialog ()
{
  if (!materialdialog_ || !scene_->selectedNode ())
    return;

  sel_material_ = (QvMaterial*)searchPropertyNode (QvNodeType::QvMaterial);

  QvMaterial temp;
  QvMaterial* p = sel_material_;
  if (!sel_material_)  // use default values
    p = &temp;

  materialdialog_->update (p->ambientColor.values, p->diffuseColor.values, p->specularColor.values, 
                           p->emissiveColor.values, p->shininess.values, p->transparency.values);

} // updateMaterialDialog


// selects a material node, that matches only with the selection

int VRMLScene::selectMaterial ()
{
  if (!scene_->selectedNode () || !scene_->manipulationAllowed ())
    return 0;

  IntArray* mtlpath;
  sel_material_ = (QvMaterial*)searchPropertyNode (QvNodeType::QvMaterial, &mtlpath);

  if (sel_material_ && !multipleRelevantOfProperty (QvNodeType::QvMaterial, mtlpath))
    return 1;  // ok

  IntArray* selectedpath = new IntArray;
  *selectedpath = *scene_->selectedPath ();

  QvGroup* parent = nil;
  QvNode* selectednode = nil;
  int whichnode, 
      size;
  if (!getSelectionByPath (selectedpath, parent, selectednode, whichnode, size))
    return 0;

  if (numberOfRelevantNodes (parent, whichnode + 1, QvNodeType::QvMaterial))  
  { // separate selected node
    QvSeparator* newseparator = new QvSeparator;
  
    newseparator->getChildren ()->append (selectednode);  // ref
   
    parent->getChildren ()->remove (whichnode);  // unref selected node
    parent->getChildren ()->insert (whichnode, newseparator);
  
    parent = newseparator;
    whichnode = 0;
    selectedpath->append (whichnode);
  }

  // insert material before selected node
  sel_material_ = new QvMaterial;
  sel_material_->ref ();

  parent->getChildren ()->insert (whichnode, sel_material_);

  // update selection (node, transform, path)
  selectedpath->remove (size-1);
  selectedpath->append (whichnode+1);
  scene_->selectNode (selectednode, selectedpath);  // takes over path

  rebuild (0);
  return 1;

}  // selectMaterial


// apply material information to the selected geometric node(s)

int VRMLScene::applyMaterial (
  const float* ambientColor, const float* diffuseColor, const float* specularColor, const float* emissiveColor, 
  float shininess, float transparency)
{
  if (scene_->selectedNode () && scene_->selectedNode ()->isGroupNode ())
  { cerr << "Apply to Group Node is not implemented now.\n";
    return 0;
  }

  if (!selectMaterial ())  // sets sel_material_
    return 0;

  int i;
  for (i = 0; i < 3; i++)
    sel_material_->ambientColor.values[i] = ambientColor[i];
  for (i = 0; i < 3; i++)
    sel_material_->diffuseColor.values[i] = diffuseColor[i];
  for (i = 0; i < 3; i++)
    sel_material_->specularColor.values[i] = specularColor[i];
  for (i = 0; i < 3; i++)
    sel_material_->emissiveColor.values[i] = emissiveColor[i];
  sel_material_->shininess.values[0] = shininess;
  sel_material_->transparency.values[0] = transparency;

  sel_material_->ambientColor.setDefault(0);
  sel_material_->diffuseColor.setDefault(0);
  sel_material_->specularColor.setDefault(0);
  sel_material_->emissiveColor.setDefault(0);
  sel_material_->shininess.setDefault(0);
  sel_material_->transparency.setDefault(0);

  scene_->sceneManipulated (1);

  rebuild (0);

  return 1;  // ok

} // applyMaterial


/* Texture */

void VRMLScene::updateTextureDialog ()
{
  if (!texturedialog_)
    return;

  // check, if texture apply is possible
  int ret = 0;
  QvNode* node = scene_->selectedNode ();
  if (!node)
    ret = 1;
  else if (node->isGroupNode ())
    ret = 1;
  else if ((node->nodeType () == QvNodeType::QvIndexedFaceSet)
           && (((QvIndexedFaceSet*)node)->textureCoordIndex.values[0] == -1))  // no texture coordinates
    ret = 1;

  if (ret)
  {
    texturedialog_->clear ();
    return;
  }

  IntArray* mtlpath;

  // Texture2
  RString path("");
  int wraph = 0;
  int wrapv = 0;

  sel_texture_ = (QvTexture2*)searchPropertyNode (QvNodeType::QvTexture2, &mtlpath);
  if (sel_texture_)
  {
    path = sel_texture_->filename.value.getString ();
    wraph = sel_texture_->wrapS.value;
    wrapv = sel_texture_->wrapT.value;

    if (multipleRelevantOfProperty (QvNodeType::QvTexture2, mtlpath))
      sel_texture_ = 0;
  }

  // Texture2Transform
  float transx = 0, transy = 0;  // default values
  float rot    = 0;
  float scalex = 1, scaley = 1;

  sel_texturetransform_ = (QvTexture2Transform*)searchPropertyNode (QvNodeType::QvTexture2Transform, &mtlpath);
  if (sel_texturetransform_)
  {
    transx = sel_texturetransform_->translation.value[0],
    transy = sel_texturetransform_->translation.value[1],
    rot = sel_texturetransform_->rotation.value,
    scalex = sel_texturetransform_->scaleFactor.value[0],
    scaley = sel_texturetransform_->scaleFactor.value[1];

    if (multipleRelevantOfProperty (QvNodeType::QvTexture2Transform, mtlpath))
      sel_texturetransform_ = 0;
   }

  texturedialog_->update (path, wraph, wrapv, transx, transy, rot, scalex, scaley);

} // updateTextureDialog


// inserts the property node into the scene graph at the right
// position before the selected node
// now used for texture and texturetransform, could also be used for material
// TODO: also for group nodes 

int VRMLScene::insertPropertyNode (QvNode* propnode)
{
  if (!scene_->manipulationAllowed () || !scene_->selectedNode ())
    return 0; // not ok

  if (scene_->selectedNode ()->isGroupNode ())
  { cerr << "Apply to Group Node is not implemented.\n";
    return 0;
  }

  IntArray* selectedpath = new IntArray;
  *selectedpath = *scene_->selectedPath ();

  QvGroup* parent = nil;
  QvNode* selectednode = nil;
  int whichnode, 
      size;
  if (!getSelectionByPath (selectedpath, parent, selectednode, whichnode, size))
    return 0;
  
  int geometries = 0;
  int n = parent->getNumChildren ();
  for (int i = 0; i < n; i++)
    if (parent->getChild (i)->hasextent_)
      geometries++;


  if (geometries > 1)
  { // separate selected node
    QvSeparator* newseparator = new QvSeparator;
  
    newseparator->getChildren ()->append (selectednode);  // ref
   
    parent->getChildren ()->remove (whichnode);  // unref selected node
    parent->getChildren ()->insert (whichnode, newseparator);
  
    parent = newseparator;
    whichnode = 0;
    selectedpath->append (whichnode);
  }

  // insert property node
  parent->getChildren ()->insert (whichnode, propnode);

  // update selection (node, transform, path)
  selectedpath->remove (size-1);
  selectedpath->append (whichnode+1);
  scene_->selectNode (selectednode, selectedpath);  // takes over path

  return 1;

} // insertPropertyNode


// ensure selected node has its unique Texture2 and Texture2Transform nodes,
// set texture values for them

int VRMLScene::applyTexture (
  const char* filename, int wraps, int wrapt,
  float transx, float transy, float rot, float scalex, float scaley
)
{
  if (!scene_->manipulationAllowed () || !scene_->selectedNode ())
    return 0; // not ok

  if (scene_->selectedNode ()->isGroupNode ())
  { cerr << "Apply to Group Node is not implemented.\n";
    return 0;
  }

  if (!sel_texture_)
  { sel_texture_ = new QvTexture2;
    insertPropertyNode (sel_texture_);
  }

  // Texture2
  sel_texture_->changeURL (filename);
  sel_texture_->filename.setDefault (0);

  sel_texture_->wrapS.value = wraps;
  sel_texture_->wrapT.value = wrapt;
  sel_texture_->wrapS.setDefault (0);
  sel_texture_->wrapT.setDefault (0);

  // Texture2Transform
  if ((transx != 0) || (transy != 0) || (rot != 0) || (scalex != 1) || (scaley != 1))
  {
    if (!sel_texturetransform_)
    { sel_texturetransform_ = new QvTexture2Transform;
      insertPropertyNode (sel_texturetransform_);
    }

    sel_texturetransform_->translation.value[0] = transx;
    sel_texturetransform_->translation.value[1] = transy;
    sel_texturetransform_->rotation.value = rot;
    sel_texturetransform_->scaleFactor.value[0] = scalex;
    sel_texturetransform_->scaleFactor.value[1] = scaley;

    sel_texturetransform_->translation.setDefault (0);
    sel_texturetransform_->rotation.setDefault (0);
    sel_texturetransform_->scaleFactor.setDefault (0);
  }
  else if (sel_texturetransform_)
  { // default values
    sel_texturetransform_->translation.setDefault (1);
    sel_texturetransform_->rotation.setDefault (1);
    sel_texturetransform_->scaleFactor.setDefault (1);
  }

  scene_->sceneManipulated (1);
  rebuild (0);
  return 1; // redraw

} // applyTexture


/* WWWAnchor */

void VRMLScene::updateWWWAnchorDialog ()
{
  if (!wwwanchordialog_)
    return;

  sel_wwwanchor_ = (QvWWWAnchor*)searchGroupNode (QvNodeType::QvWWWAnchor);

  if (sel_wwwanchor_)
    wwwanchordialog_->update (sel_wwwanchor_->name.value.getString (), sel_wwwanchor_->description.value.getString ());
  else
    wwwanchordialog_->update ("", "");

} // updateWWWAnchorDialog


int VRMLScene::applyWWWAnchor (const char* url, const char* desc)
{
  if (/* !*url || */ !selectWWWAnchor ())
    return 0;

  sel_wwwanchor_->name.value = url;
  sel_wwwanchor_->name.setDefault (0);

  if (*desc)  // if description field is not empty
  { sel_wwwanchor_->description.value = desc;
    sel_wwwanchor_->description.setDefault (0);
  }
  else
    sel_wwwanchor_->description.setDefault (1);

  return 1;  // anchor applied

} // applyWWWAnchor



int VRMLScene::selectWWWAnchor ()
{
  sel_wwwanchor_ = 0;
  QvNode* selnode = scene_->selectedNode ();
  if (!selnode || !scene_->manipulationAllowed ())
    return 0;

  if (scene_->selectedPath ()->count () < 2) // do not use top separator
    return 0;
  
  // if the selected node is a wwwanchor, use it
  if (selnode->nodeType () == QvNodeType::QvWWWAnchor)
  { sel_wwwanchor_ = (QvWWWAnchor*)selnode;
    return 1;
  }

  // search wwwanchor in all parents along the path
  IntArray* path = new IntArray;
  sel_wwwanchor_ = (QvWWWAnchor*)searchGroupNode (QvNodeType::QvWWWAnchor, &path);
  
  // if a wwwanchor has been found
  if (sel_wwwanchor_)  
  { 
    int number = numberOfRelevantNodes (sel_wwwanchor_, 0, QvNodeType::QvWWWAnchor);
    if (number == 1)  // only the selected node in in the wwwanchor
      return 1;
  }

  sel_wwwanchor_ = new QvWWWAnchor;
  QvGroup* parent;
  int whichnode, size, i, n;
  *path = *scene_->selectedPath ();
  getSelectionByPath (path, parent, selnode, whichnode, size);

  if (selnode->isGroupNode ())
  { 
    // substitute the selected group node through the wwwanchor
    n = ((QvGroup*)selnode)->getNumChildren ();
    for (i = 0; i < n; i++)
      sel_wwwanchor_->getChildren ()->append (((QvGroup*)selnode)->getChild (i));

      scene_->selectNode (sel_wwwanchor_, 0);
      scene_->selectNode (sel_wwwanchor_, path);
  }
  else
  { 
    // check if the parent has more nodes with extents as the selected one
    n = parent->getNumChildren ();
    int extents = 0;
    for (i = 0; i < n; i++)
      if (parent->getChild (i)->hasextent_)
        extents++;

    if ((extents == 1) && (path->count () > 2)) 
    {
      // substitute the parent of the selected node through the wwwanchor
      for (i = 0; i < n; i++)
        sel_wwwanchor_->getChildren ()->append (parent->getChild (i));

      path->remove (path->count () - 1);
      parent = getParentByPath (path);
      whichnode = (*path)[path->count () - 1];
      delete path;
    }
    else
    {
      // substitute the selected node through the wwwanchor
      sel_wwwanchor_->getChildren ()->append (selnode);

      scene_->selectNode (sel_wwwanchor_, 0);
      scene_->selectNode (sel_wwwanchor_, path);
    }
  }

  // substitute the computed node through the wwwanchor
  parent->getChildren ()->remove (whichnode);
  parent->getChildren ()->insert (whichnode, sel_wwwanchor_);
  sel_wwwanchor_->hasextent_ = 1;
  rebuild (0);

  scene_->sceneManipulated (1);
  updateStructureViewerDialog ();

  return 1;

} // selectWWWAnchor


// deletes the wwwanchor of the selected node

void VRMLScene::deleteWWWAnchor ()
{
  if (!scene_->manipulationAllowed ())
    return;

  QvNode* selnode = scene_->selectedNode ();
  if (!selnode || !sel_wwwanchor_)  
    return;
  
  IntArray* path = new IntArray;
  // store location of wwwanchor in path
  if (selnode->nodeType () == QvNodeType::QvWWWAnchor)
  { // the selected node is a wwwanchor, substitute it through a separator
    *path = *scene_->selectedPath ();
  }  
  else
  {
    int number = numberOfRelevantNodes (sel_wwwanchor_, 0, QvNodeType::QvWWWAnchor);
    if (number > 1)
    { cerr << "wwwanchor cannot be deleted. other nodes also belongs to it.\n";
      delete path;
      return;
    }
    searchGroupNode (QvNodeType::QvWWWAnchor, &path);
  }
  

  // copy chidren to new separarator
  QvSeparator* newsep = new QvSeparator;
  int n = sel_wwwanchor_->getNumChildren ();
  for (int i = 0; i < n; i++)
    newsep->getChildren ()->append (sel_wwwanchor_->getChild (i));

  // exchange wwwanchor for a new separarator
  QvGroup* parent = (QvGroup*)getParentByPath (path);
  int which = (*path)[path->count () - 1];

  parent->getChildren ()->remove (which); // remove wwwanchor (unref)
  parent->getChildren ()->insert (which, newsep);
  rebuild (0);

  // update selection
  if (selnode == sel_wwwanchor_)
  { scene_->selectNode (newsep, 0);
    scene_->selectNode (newsep, path);
  }
  else 
    delete path;

  sel_wwwanchor_ = 0;
  updateStructureViewerDialog ();
  scene_->sceneManipulated (1);

} // deleteWWWAnchor


/* Structure Viewer */

// parse the scene graph and build a tree of all group and geometric nodes
// maps the scene graph to a linear list

void VRMLScene::parseSceneGraph (QvGroup* group, IntArray* path)
{
  if (!group || !path)
    return;

  QvNode* child;
  int numchildren = group->getNumChildren ();
  int i = -1;
  path->append (0);
  while (++i < numchildren)
  { 
    child = group->getChild (i);
    path->remove (path->count () - 1);
    path->append (i);

    if (child->hasextent_)
    { 
      int selection = 0;
      if (child == scene_->selectedNode ())
      { selection = child->refCount - 1;
        if (child == paste_)
          selection--;
      }
      structureviewerdialog_->append (
        qvNodeName[child->nodeType ()], child->getName ().getString (), path->count (), selection
      );

      if (scene_->selectedPath () && (*path == *scene_->selectedPath ()))
        structureviewerdialog_->selectItem (); // select last item
    }

    if (child->isGroupNode () && child->hasextent_)
      parseSceneGraph ((QvGroup*)child, path);
  }

  path->remove (path->count () - 1);

} // parseSceneGraph


// reconstruct the path of the number of the listitem of the structure viewer

int VRMLScene::getSelectedNodeByNumber (QvGroup* group, IntArray* path, int& number)
{
  QvNode* child;
  int numchildren = group->getNumChildren ();
  int i = -1;
  while (++i < numchildren)
  { child = group->getChild (i);
    if (child->hasextent_)
    { if (!number--)
      { path->append (i);
        return 1;  // node found
      }
      //number--;
    }

    if (child->isGroupNode () && child->hasextent_)
    { path->append (i);
      if (getSelectedNodeByNumber ((QvGroup*)child, path, number))
        return 1; // node found
      else
        path->remove (path->count () - 1);  // remove last element
    }
  }

  return 0;  // node not found

} // getSelectedNodeByNumber



// updates the structure viewer dialog

void VRMLScene::updateStructureViewerDialog ()
{
  if (!structureviewerdialog_ || !root_)
    return;

  structureviewerdialog_->storeAdjustable ();
  IntArray path;
  structureviewerdialog_->clear ();
  parseSceneGraph ((QvGroup*)root_, &path);
  structureviewerdialog_->useStoredAjustable ();

  RString name;
  if (paste_)
  { name = qvNodeName[paste_->nodeType ()];
    name += "  ";
    name += paste_->objName->getString ();
  }
  structureviewerdialog_->setPasteName (name);

  if (!scene_->selectedNode ())
  { name = "";
    structureviewerdialog_->setNodeNameField (name);
  } // otherwise it was set  with the selection of the listitem

} // updateStructureViewerDialog


// change the selection of the scene, after having changed it in the structure viewer dialog

int VRMLScene::applyNodeSelection (int item)
{

  IntArray* path = new IntArray;
  if (!getSelectedNodeByNumber ((QvGroup*)root_, path, item))
  { delete path;
    return 0;
  }

  storeForUndo ();
  selectNodeByPath (path);
  return 1;

} // applyNodeSelection


void VRMLScene::applyNodeName (int whichitem, RString& name)
{

  IntArray* path = new IntArray;
  if (!getSelectedNodeByNumber ((QvGroup*)root_, path, whichitem))
  { delete path;
    return;
  }

  // TODO-anuss: check, if name != "" when there are more than one references

  QvNode* node = getSelectedNodeByPath (path);
  QvName newname (name.string ());
  node->setName (newname);

  delete path;
  updateStructureViewerDialog ();
}

void VRMLScene::getNameOfSelectedNode (RString& name)
{
  QvNode* node = scene_->selectedNode ();

  if (node)
    name = node->objName->getString ();
  else
    name = "";

} // getNameOfSelectedNode


/* Transform */

void VRMLScene::updateTransformDialog (int what)
{
  if (!transformdialog_)
    return;

  if (!scene_->selectedNode ())
  { transformdialog_->clear ();
    return;
  }

  QvTransform defaulttransform;
  QvTransform* transform = sel_transform_;
  if (!transform)
    transform = &defaulttransform;

  vector3D translate, rotate, scale;
  float angle;

  if (transformdialog_->coordinateSystem () == TransformDialog::model)
  { // model coordinates
    translate = *(vector3D*) transform->translation.value;
    rotate = *(vector3D*) transform->rotation.axis;
    scale = *(vector3D*) transform->scaleFactor.value;
    angle = transform->rotation.angle;
    
    if (!what || (what == 1))
      transformdialog_->updateTranslation (translate);
    if (!what || (what == 2))
      transformdialog_->updateRotation (rotate, angle);
    if (!what || (what == 3))
      transformdialog_->updateScaling (scale);
  }
  else
  { // world coordiantes
    matrix4D objectmatrix;
    ge3dPushIdentity ();
    ge3dMultMatrix (camerainvmat_);
    ge3dMultMatrix (scene_->selectedNode ()->selectedTransformation_);
    ge3d_get_and_pop_matrix (objectmatrix);

    translate = *(vector3D*)objectmatrix[3];
    rotate.x = 0; // TODO calculate this
    rotate.y = 0;
    rotate.z = 0;
    angle = 0;

    vector3D temp;
    temp = *(vector3D*)objectmatrix[0];
    scale.x = norm3D (temp);
    temp = *(vector3D*)objectmatrix[1];
    scale.y = norm3D (temp);
    temp = *(vector3D*)objectmatrix[2];
    scale.z = norm3D (temp);

    if (!what || (what == 1))
      transformdialog_->updateTranslation (translate);
    if (!what || (what == 2))
      transformdialog_->clearRotation ();
    if (!what || (what == 3))
      transformdialog_->updateScaling (scale);
  }

} // updateTransformDialog


// searches the transform, that determines the selected node
// no separation, if it determines also other nodes

void VRMLScene::searchTransform ()
{
  QvGroup* parent = nil;
  QvNode* selectednode = nil;
  int whichnode, size;
  if (!getSelectionByPath (scene_->selectedPath (), parent, selectednode, whichnode, size))
    return;

  transformation_allowed_ = 0;
  sel_transform_insidegroup_ = 0;
  sel_transform_ = 0;
  int n = parent->getNumChildren ();
  QvNode* child;

  // transform node inside group node?
  if (selectednode->isGroupNode ())
  {
    QvGroup* selectedgroup = (QvGroup*) selectednode;
    int nsg = selectedgroup->getNumChildren ();  // n selected group
    int i = -1;
    while (++i < nsg)
    { child = selectedgroup->getChild (i);
      if (child->nodeType () == QvNodeType::QvTransform)
      { sel_transform_ = (QvTransform*)child;
        transformation_allowed_ = 1;  // may change this Transform node
        sel_transform_insidegroup_ = 1;
        return;  // transform node found
      }
      if (child->hasextent_)
        break;
    }
  }

  // geometric node after selected node?
  int i = whichnode;
  while (++i < n)
    if (parent->getChild (i)->hasextent_)
      return;  

  // look for the last transform node before the selected node
  i = whichnode;
  while (i--)
  { child = parent->getChild (i);
    if (child->hasextent_)
      return;

    if (child->nodeType () == QvNodeType::QvTransform)
    { sel_transform_ = (QvTransform*) child;
      transformation_allowed_ = 1;
      return;
    }
  }

} // searchTransform


// inserts a new transform for the selected node
// assumes, that a new transform node must be inserted
// (an existent one had been detected in searchTransform)

void VRMLScene::makeTransformForSelected ()
{
  QvGroup* parent = nil;
  QvNode* selectednode = nil;
  int whichnode, size;
  if (!getSelectionByPath (scene_->selectedPath (), parent, selectednode, whichnode, size))
    return;

  QvTransform* newtransform = new QvTransform;  // transform to insert
  transformation_allowed_ = 1;
  sel_transform_ = newtransform;

  if (selectednode->isGroupNode ())
  {
    parent = (QvGroup*)selectednode;
    parent->getChildren ()->insert (0, newtransform);  // insert at top

    sel_transform_insidegroup_ = 1;
    storeForUndo ();

    return;
  }

  // geometric node after selected node?
  int separate = 0;
  int n = parent->getNumChildren ();
  int i = whichnode;
  while (++i < n)
    if (parent->getChild (i)->hasextent_)
    { separate = 1;
      break;
    }

  IntArray* path = new IntArray;
  *path = *scene_->selectedPath ();

  if (separate)
  {
    QvSeparator* newseparator = new QvSeparator;
    newseparator->getChildren ()->append (newtransform);
    newseparator->getChildren ()->append (selectednode);  // ref
    parent->getChildren ()->remove (whichnode);  // unref
    parent->getChildren ()->insert (whichnode, newseparator);
  
    // update selected path
    path->append (1);
    scene_->selectNode (selectednode, path);
    updateStructureViewerDialog ();
  }
  else
  {
    parent->getChildren ()->insert (whichnode, newtransform);

    // update selected path
    path->set_value (size - 1, whichnode + 1);
    scene_->selectNode (selectednode, path);
  }

  rebuild (0); // TODO newseparator->build () ?

  // update edit context (calculateObjectExtent is up to date)
  sel_transform_insidegroup_ = 0;
  storeForUndo ();

} // makeTransformForSelected


// select transformnode according to selected node
// old code; now transformation only created on object manipulation, not already on object selection

// void VRMLScene::selectTransform ()
// {
//   if (!scene_->selectedNode () || !scene_->selectedPath () || !scene_->manipulationAllowed ())
//     return;

//   IntArray *path = new IntArray;
//   *path = *scene_->selectedPath ();

//   QvGroup* parent = nil;
//   QvNode* selectednode = nil;
//   int whichnode, 
//       size;
//   if (!getSelectionByPath (path, parent, selectednode, whichnode, size) | !selectednode->hasextent_)
//   { delete path;
//     return;
//   }

//   sel_transform_ = nil;
//   sel_transform_insidegroup_ = 0;

//   QvChildList* childlist = parent->getChildren ();
//   int i;

//   // if the selected node is a groupnode, use the transformationnode inside
//   if (selectednode->isGroupNode ())
//   {
//     parent = (QvGroup*) selectednode;
//     childlist = parent->getChildren ();

//     // search for the first geometry or group node in the selected group
//     for (i = 0;  i < parent->getNumChildren ();  i++)
//       if (parent->getChild (i)->hasextent_)
//       { whichnode = i;
//         break;
//       }

//     if (whichnode >= parent->getNumChildren ())
//     { sel_transform_ = nil;
//       delete path;
//       return;  // no node witch extension is inside the selected group
//     }  
//     sel_transform_insidegroup_ = 1;  // selected transform is inside the selected group
//   }
//   else
//     // if geometry or group node afterwards, ...
//     for (i = whichnode + 1; i < parent->getNumChildren ();  i++)
//       if (parent->getChild (i)->hasextent_)
//       { 
//         // separate the selected node and insert a transformation to it
//         QvSeparator* newseparator = new QvSeparator;
//         QvTransform* newtransform = new QvTransform;
    
//         QvChildList* sepchildren = newseparator->getChildren ();
//         sepchildren->append (newtransform);
//         sepchildren->append (selectednode);  // ref
   
//         childlist->remove (whichnode);  // unref
//         childlist->insert (whichnode, newseparator);
  
//         sel_transform_ = newtransform;
//         path->append (1);
//         scene_->selectNode (selectednode, path);
//         rebuild (0);

//         storeForUndo ();
//         transformation_allowed_ = 1;
//         return;
//       }

//   // take the last transformation node before the selected node (path)
//   for (i = whichnode - 1;  i >= 0;  i--)
//   {
//     if ((parent->getChild (i)->isGeometryNode ()) || parent->getChild (i)->isGroupNode ())
//       break;

//     if (parent->getChild (i)->nodeType () == QvNodeType::QvTransform)
//     { sel_transform_ = (QvTransform*) parent->getChild (i);
//       delete path;
//       transformation_allowed_ = 1;
//       return;
//     }
//   }

//   // insert transformation before the selected node
//   QvTransform* newtransform = new QvTransform;

//   childlist->insert (whichnode, newtransform);
//   sel_transform_ = newtransform;
//   if (!sel_transform_insidegroup_)  // then path of the selected node has changed
//     path->set_value (size - 1, whichnode + 1);

//   scene_->selectNode (selectednode, path);
//   rebuild (0);
//   transformation_allowed_ = 1;

// } // selectTransform


// apply transformation of Transform dialog

int VRMLScene::applyTransform (
  const vector3D* translate, const vector3D* rotate, const float* angle, const vector3D* scale
)
{
  if (!transformdialog_)
    return 0; // pathologic

  if (!sel_transform_)
  { transformdialog_->clear ();
    return 0;
  }

  storeForUndo ();

  *(vector3D*)sel_transform_->translation.value = *translate;
  *(vector3D*)sel_transform_->rotation.axis = *rotate;
  *(vector3D*)sel_transform_->scaleFactor.value = *scale;
  sel_transform_->rotation.angle = *angle;

  rebuild (0);

  return 1; // redraw

} // applyTransform



/**** geometric node selection ***/


// viewSelectedNode - center selected node in view
// returns redraw flag

int VRMLScene::viewSelectedNode ()
{
  if (!scene_->selectedNode ())
    return 0;

  point3D wmin, wmax;
  calculateCameraMatrix ();
  calculateObjectExtent ();

  // transform bounding box into world coordinates
  // code from getBoundingboxWorld
  const matrix4D& selectedtransformation = scene_->selectedNode ()->selectedTransformation_;
  ge3dPushIdentity ();
  ge3dMultMatrix (camerainvmat_);
  ge3dMultMatrix (selectedtransformation);  // after the actual transformrotation
  computeBoundingbox (sel_omin_, sel_omax_, wmin, wmax);
  ge3d_pop_matrix ();

  point3D center = wmin;
  inc3D (center, wmax);
  scl3D (center, 0.5);

  float size = (wmax.x - wmin.x + wmax.y - wmin.y + wmax.z - wmin.z) / 3;

  point3D pos, look, up;  // see viewAll
  center.z += 3 * size;  // rule of thumb, zoom out somewhat more than in viewAll
  init3D (pos, center.x, center.y, center.z);
  init3D (look, center.x, center.y, center.z - 1);  // looking along z (to -z)
  init3D (up, 0, 1, 0);

  camera_->setposlookup (pos, look, &up);  // default up vector (y axis)

  return 1;

} // viewSelectedNode



// viewAll - bring whole scene in view

void VRMLScene::viewAll ()
{
  point3D center;
  scene_->getCenter (center);

  point3D pos, look, up;
  center.z += 2 * scene_->size ();  // rule of thumb (in accordance with cam angle)
  init3D (pos, center.x, center.y, center.z);
  init3D (look, center.x, center.y, center.z - 1);  // looking along z (to -z)
  init3D (up, 0, 1, 0);

  DEBUGNL ("setting default camera at pos. " << pos << ", looking to " << look);
  camera_->setposlookup (pos, look, &up);  // default up vector (y axis)
} // viewAll


// extend selection to parent of currently selected node

int VRMLScene::selectParent ()
{
  const IntArray* path = scene_->selectedPath ();
  // never select artificial root-level group node
  if (!path || (path->count () < 2))
    return 0;

  QvGroup* parent = nil;
  QvNode* node = nil;
  int whichnode, size;   
  if (!getSelectionByPath (path, parent, node, whichnode, size))
    return 0;

  IntArray* newpath = new IntArray;
  *newpath = *path;
  newpath->remove (size - 1);

  scene_->selectNode (parent, newpath);  // takes over path
  updateEditContext ();  // select transform node before the first manipulation

  return 1; // redraw

} // selectParent


// select first child (that has an extent) of currently selected node

int VRMLScene::selectChild ()
{
  const IntArray* path = scene_->selectedPath ();
  QvNode* node = scene_->selectedNode ();
  if (!path || !node || !node->isGroupNode ())
    return 0;

  IntArray* newpath = new IntArray;
  *newpath = *path;

  QvGroup* parent = (QvGroup*)node;
  int n = parent->getNumChildren ();
  int i = -1;
  while (++i < n)
    if (parent->getChild (i)->hasextent_)
    { newpath->append (i);
      scene_->selectNode (parent->getChild (i), newpath);  // takes over path
      updateEditContext ();  // select transform node before the first manipulation
      return 1; // redraw
    }

  return 0;

} // selectChild 


// select next sibling node (that has an extent) on current level, cyclical
// step: forward (+1) or backward (-1)

int VRMLScene::selectNext (int step)
{
  const IntArray* path = scene_->selectedPath ();
  if (!path)
    return 0;

  QvGroup* parent = nil;
  QvNode* node = nil;
  int whichnode, size;   
  if (!getSelectionByPath (path, parent, node, whichnode, size))
    return 0;
  int n = parent->getNumChildren ();
  if (n < 2)
    return 0;

  IntArray* newpath = new IntArray;
  *newpath = *path;

  int i = whichnode;
  while ((i += step) != whichnode)
  {
    if (i >= n)
      i = 0;
    else if (i < 0)
      i = n - 1;
    if (i == whichnode)
      break;
    if (parent->getChild (i)->hasextent_)
    {
      newpath->set_value (size - 1, i);
      scene_->selectNode (parent->getChild (i), newpath);  // takes over path
      updateEditContext ();  // select transform node before the first manipulation
      return 1; // redraw
    }
  }    

  return 0;

} // selectNext


/***   Delete, Insert, Copy, Cut, Paste ***/


// deletes a node from the scene graph

int VRMLScene::deleteNode (const IntArray& path)
{
  if (!path.count ())  // root node must not deleted
    return 0;


  // if a node is selected, only the selected node may be deleted;
  // otherwise the new path of the selected must be claculated
  if (scene_->selectedPath ())
    return 0;

  // delete node
  QvGroup* parent = nil;
  QvNode* node = nil;
  int whichnode, 
      size;  
  if (!getSelectionByPath (&path, parent, node, whichnode, size))
    return 0;
  int hasextent = node->hasextent_;

  // delete node
  if (path.count () == 1) // top separator in vrml structure
  { // clear top separator (without some special nodes)
    if (!node->isGroupNode ())
      return 0;
    parent = (QvGroup*)node;
    QvChildList* childlist = parent->getChildren ();
    int i = 0;
    while (i < parent->getNumChildren ())
    { if (parent->getChild (i) == camswitch_)
      { i++;
        continue;
      }
      childlist->remove (i);
    }
  }
  else
    parent->getChildren ()->remove (whichnode);  // unref

  if (hasextent)
    calculateHasExtent ((QvGroup*)root_); // for rebuild necessary
  rebuild (0);

  scene_->sceneManipulated (1);

  return 1; // node deleted

} // delete node


// the parameter path is the path, that the node has after the insertion

int VRMLScene::insertNode (QvNode* node, const IntArray& path)
{
  if (!root_ || !node || (path.count () < 2))  // path must be at least 2
    return 0;

  // check path
  QvGroup* parent = getParentByPath (&path);
  if (!parent || !parent->isGroupNode ())
    return 0;

  // insert node
  int position = path[path.count () - 1];
  if (position < parent->getNumChildren ())
    parent->getChildren ()->insert (position, node); // ref
  else
    parent->getChildren ()->append (node);
  
  // update selected path
  if (scene_->selectedPath ())
  { IntArray *selectedpath = new IntArray;
    *selectedpath = *scene_->selectedPath ();
    int index = path.count () - 1;
    if ((index <= selectedpath->count ()) && (path[index] <= (*selectedpath)[index]))
    { selectedpath->set_value (index, (*selectedpath)[index] + 1);
      scene_->selectNode (scene_->selectedNode (), selectedpath);
    }
    else
      delete selectedpath;
  }

  scene_->sceneManipulated (1);

  if (node->hasextent_ && !parent->hasextent_)  
    calculateHasExtent ((QvGroup*)root_); // for rebuild

  rebuild (0); 

  return 1;

} // insertNode


// for cut, copy, and paste
// use parent of selected node if selected node is the only child of
// its parent that has an extent (this copies node properties as well)

QvNode* VRMLScene::getCutCopyPastePath (IntArray& path)
{
  if (!scene_->selectedPath ())
    return 0;
  path = *scene_->selectedPath ();

  if (path.count () < 2)
  { cerr << "cannot cut/copy top level separator\n";
    return 0; // top separator must not be used for cut, copy, paste
  }

  QvGroup* parent = nil;
  QvNode* node = nil;
  int whichnode, size;   
  if (!getSelectionByPath (&path, parent, node, whichnode, size))
    return 0;

  if (path.count () == 2)
    return node; // parent is top separator

  // take parent, if the selected node is the only with extent
  int extents = 0;
  int i = parent->getNumChildren ();
  while (i--)
    if (parent->getChild (i)->hasextent_)
      extents++;
  if (extents == 1)
    path.remove (size -1); // take the parent

  return parent;

} // getCutCopyPastePath


// delete Node
 
int VRMLScene::deleteSelectedNode ()
{
  IntArray path;
  if (!getCutCopyPastePath (path))
    return 0;

  // delete node from scene graph
  scene_->selectNode (0);
  deleteNode (path);  // ass. returns 1, node is deleted afterwards
  updateEditContext ();
  clearUndoStorage ();

  return 1;

} // deleteSelectedNode


// remove selected node to paste_

int VRMLScene::cutSelectedNode ()
{
  QvNode* node = scene_->selectedNode ();
  if (!node)
    return 0;
  if (!node->isGeometryNode ())
  { cerr << "only geometry nodes can be cut\n";
    return 0;
  }
  IntArray path = *scene_->selectedPath ();

//   IntArray path;
//   QvNode* node = getCutCopyPastePath (path);
//   if (!node)
//     return 0;

  // store selected node
  if (paste_)
    paste_->unref ();
  paste_ = node;
  paste_->ref ();

  scene_->selectNode (0);
  deleteNode (path);  // ass. returns 1, node is deleted afterwards
  updateEditContext ();
  clearUndoStorage ();

  return 1;

} // cutSelectedNode


// makes a new reference of the the selected node 

int VRMLScene::copySelectedNode ()
{
  QvNode* node = scene_->selectedNode ();
  if (!node)
    return 0;
  if (!node->isGeometryNode ())
  { cerr << "only geometry nodes can be copied\n";
    return 0;
  }

//   IntArray path;
//   QvNode* node = getCutCopyPastePath (path);
//   if (!node)
//     return 0;

  // store selected node
  if (paste_)
    paste_->unref ();
  paste_ = node;
  paste_->ref ();

  updateStructureViewerDialog ();  // only paste name must be updated

  return 1;

} // copySelectedNode


// inserts the node

int VRMLScene::pasteSelectedNode ()
{
  if (!paste_ || !root_)
    return 0;

  IntArray path;
  if (!findInsertPath (path))
    return 0;
  int ret = insertNode (paste_, path);
  if (ret)
  {
    if ((paste_->refCount > 2) && !*(paste_->getName ().getString ()))
    { // make name for mulitiple reference (to be saved as DEF/USE)
      char name[16];
      sprintf (name, "node%d", pastenr_++);
      // TODO: should check wheter node name is unique
      QvName newname (name);
      paste_->setName (newname);
    }

    updateStructureViewerDialog ();
    clearUndoStorage ();
  }

  return ret;

} // pasteSelectedNode


// insert a node into the current scene
// child of selected node or additional child of top level separator

int VRMLScene::findInsertPath (IntArray& path)
{
  if (!root_)
    return 0;

  while (path.count ()) // clear path
    path.remove (path.count () - 1);

  // find parent, where to insert node
  QvGroup* parent;

  if (scene_->selectedNode ())  // something is selected
    if (scene_->selectedNode ()->isGroupNode ())  // selection is a group
    { parent = (QvGroup*)scene_->selectedNode ();
      path = *scene_->selectedPath ();
    }
    else  // selection is a geometric node
    {
      return 0;
    }  
  else  // nothing selected
  { 
    parent = (QvGroup*) root_;
    if (!parent->getNumChildren () || !parent->getChild (0)->isGroupNode ())
    { cerr << "no top level Separator (or other grouping node). Insert failed." << endl;
      return 0; // not correct vrml
    }
    parent = (QvGroup*) (parent->getChild (0));
    path.append (0);
  }

  // find position in the groupnode (before first geometry or group node)
/*
  int position = -1;
  while (++position < parent->getNumChildren ())
    if (parent->getChild (position)->hasextent_)
      break;
*/
  path.append (parent->getNumChildren ());

  return 1;

}  // findInsertPath


// read VRML File and insert it into the scene graph

int VRMLScene::readInputInsert (QvInput& in)
{
  if (!root_)
    return 0;

  QvNode* node = 0;
  int ok = QvDB::read (&in, node) && node;

  if (ok)
  { 
    QvGroup* group = (QvGroup*)node;
    if (!checkVRMLScene (group))
      return 0;
    node = group->getChild (0);  // Separator

    IntArray path;
    findInsertPath (path);
    insertNode (node, path);

    group->unref ();
    updateEditContext ();
  }
  else
    HgMessage::message ("Error in file read, data not inserted");

  return !ok;  // error code

} // readInputInsert  


// inserts a VRML primitive node of nodetype into the scene graph

int VRMLScene::insertVRMLPrimitive (int nodetype)
{

  QvNode* node = 0;
  switch (nodetype)
  {
    case QvNodeType::QvCone:
      node = new QvCone;
      break;
    case QvNodeType::QvCube:
      node = new QvCube;
      break;
    case QvNodeType::QvCylinder:
      node = new QvCylinder;
      break;
    case QvNodeType::QvSphere:
      node = new QvSphere;
      break;
  }

  IntArray path;
  findInsertPath (path);
  insertNode (node, path);

  updateEditContext ();

  return 1; // redraw

} // insertVRMLPrimitive


// inserts a VRML primitive node of nodetype into the scene graph

int VRMLScene::insertWWWInline (const char* filename)
{
  if (!filename || !*filename)
    return 0;

  QvWWWInline* wwwinline = new QvWWWInline;
  wwwinline->name.value = filename;
  wwwinline->name.setDefault (0);
  
  IntArray path;
  findInsertPath (path);
  insertNode (wwwinline, path);

  updateEditContext ();
  
  return 1;

} // insertWWWInline


// checks the top separator, if it has only one child
// used for insertion of file data (root of subscene given),
// when starting editing (root == 0, not counted as modification),
// and also used for creating a fow vrml scene (root_ == 0)
// ensures a single Separator child of root (inserted if necessary)
// returns non-null if OK

int VRMLScene::checkVRMLScene (QvGroup* root)
{

  if (!root_)
  { // prepare empty scene to edit
    if (root)
      return 0;  // illegal case

    QvGroup* group = new QvGroup;
    QvSeparator* separator = new QvSeparator;
    group->getChildren ()->append (separator); // top separator
    root_ = group;

    return 1;
  }

  QvGroup* group = root; 
  if (!group)
    group = (QvGroup*) root_;  // check whole scene 

  int n = group->getNumChildren ();
  if (n < 1)
    return 0;

  if (n == 1 && group->getChild (0)->nodeType () == QvNodeType::QvSeparator)
    return 1;  // we already have a Separator as single group child. done.

  QvSeparator* newsep = new QvSeparator;  // add Separator below root group
  while (group->getNumChildren ())
  {
    newsep->getChildren ()->append (group->getChild (0));
    group->getChildren ()->remove (0);
  }

  group->getChildren ()->append (newsep);

  checkVRMLTree ((QvGroup*) group->getChild (0));

  rebuild (0);
  return 1;

} // checkVRMLScene


// convert VRML scene into format suitable for editing
// i.e. convert Translation/Rotation/Scale/MatrixTransform nodes into Transform nodes

void VRMLScene::checkVRMLTree (QvGroup* parent)
{
  QvNode* node;
  int n = parent->getNumChildren ();
  for (int i = 0; i < n; i++)
  {
    node = parent->getChild (i);
    int type = node->nodeType ();

    if ((type == QvNodeType::QvTranslation) 
      || (type == QvNodeType::QvRotation) 
      || (type == QvNodeType::QvScale)
      || (type == QvNodeType::QvMatrixTransform)
    )
    {
      QvTransform* transform = new QvTransform;
      int j = 3;

      switch (type)
      { 
        // translation
        case  QvNodeType::QvTranslation:
          while (j--)
            transform->translation.value[j] = ((QvTranslation*)node)->translation.value[j];

          transform->translation.setDefault (0);
          break;

        // rotation
        case QvNodeType::QvRotation:
          while (j--)
            transform->rotation.axis[j] = ((QvRotation*)node)->rotation.axis[j];
          transform->rotation.angle = ((QvRotation*)node)->rotation.angle;
      
          transform->rotation.setDefault (0);
          break;

        // scale
        case QvNodeType::QvScale:
          while (j--)
            transform->scaleFactor.value[j] = ((QvScale*)node)->scaleFactor.value[j];

          transform->scaleFactor.setDefault (0);
          break;

        // matrix transform
        case QvNodeType::QvMatrixTransform:
          QvMatrixTransform* matrix = (QvMatrixTransform*)node;
          vector3D v = { 0, 0, 0};
          float s = 0;
          matrix2quaternion (matrix->mat_, v, s);
          vector3D axis = { 0, 0, 0};
          float angle = 0;
          quaternion2rotation (v, s, axis, angle);

          vector3D temp;
          float* axis_a = &axis.x;

          while (j--)
          { transform->translation.value[j] = matrix->mat_[3][j];
            transform->rotation.axis[j] = axis_a[j];
            temp = *(vector3D*)(matrix->mat_[j]);
            transform->scaleFactor.value[j] = norm3D (temp);
          }
          transform->rotation.angle = angle;

          transform->translation.setDefault (0);
          transform->rotation.setDefault (0);
          transform->scaleFactor.setDefault (0);
          break;
      }

      parent->getChildren ()->remove (i);
      parent->getChildren ()->insert (i, transform);
    }
    else if (node->isGroupNode ())
      checkVRMLTree ((QvGroup*)node);

  }

} // checkVRMLNode


/********   Undo   *********/

// store node transformation for undo

void VRMLScene::storeForUndo ()
{
  if (!scene_->selectedNode () || !sel_transform_)
    return;

  undotransform_[undonext_].assign (sel_transform_);
  undowhichtransform_[undonext_] = sel_transform_;
  undowhichpath_[undonext_] = *scene_->selectedPath ();

  if ((undonext_ == undostart_) && !undoempty_)
  { if (++undonext_ > (undosteps - 1))
      undonext_ = 0;
    undostart_ = undonext_;
  }
  else
  { if (++undonext_ > (undosteps - 1))
      undonext_ = 0;
  }

  undoempty_ = 0;

}  // storeForUndo


// undo for selection and transformation

int VRMLScene::undoManipulation ()
{
  if (undoempty_)
  { cerr << "no undo possible\n";
    return 0;
  }

  if (--undonext_ < 0)
    undonext_ = (undosteps - 1);

  IntArray* newpath = new IntArray;
  *newpath = undowhichpath_[undonext_];
  QvNode* newnode = getSelectedNodeByPath (newpath);

  scene_->selectNode (newnode, newpath);
  undowhichtransform_[undonext_]->assign (&undotransform_[undonext_]);

  updateEditContext ();

  if (undonext_ == undostart_)
    undoempty_ = 1;

  rebuild (0);

  return 1;

}  // undoManipulation


// clear storage for undo

void VRMLScene::clearUndoStorage ()
{
  undoempty_ = 1;
  undostart_ = undonext_ = 0;

} // clearUndoStorage



/**** Viewpoints ****/

// set values of active viewpoint to scene camera


void VRMLScene::applyViewpoint (
  int which, const point3D& position, const vector3D& axis, float angle, float viewangle, const char* name
)
{
  if (!camswitch_ || (which < 0) || which >= camswitch_->getNumChildren ())
    return;

  // set name in qv
  if (name)
  { 
    QvName newname (name);
    camswitch_->getChild (which)->setName (newname);
  }

  // set position and orientation in qv
  int i;
  const float* pos_a = &position.x;
  const float* axis_a = &axis.x;
  if (camswitch_->getChild (which)->nodeType () == QvNodeType::QvPerspectiveCamera)
  { // perspective camera
    QvPerspectiveCamera* camera = (QvPerspectiveCamera*)camswitch_->getChild (which);
    for (i = 0; i < 3; i++)
    { camera->position.value[i] = pos_a[i];
      camera->orientation.axis[i] = axis_a[i];
    }
    camera->orientation.angle = angle;
    camera->heightAngle.value = viewangle;
  }
  else
  { // ortographic camera
    QvOrthographicCamera* camera = (QvOrthographicCamera*)camswitch_->getChild (which);
    for (i = 0; i < 3; i++)
    { camera->position.value[i] = pos_a[i];
      camera->orientation.axis[i] = axis_a[i];
    }
    camera->orientation.angle = angle;
    camera->height.value = viewangle;
  }

  scene_->sceneManipulated (1);

} // applyViewpoint


void VRMLScene::setViewpoint (int which)
{
  if ((!camswitch_) || (which < 0) || (which >= camswitch_->getNumChildren ()))
    return;

  // get full camera matrix (active camera and (=multiply) viewpointcamera in scene graph)
  matrix4D mat;
  calculateCameraMatrix ();
  ge3dPushIdentity ();
  // cameramat_: transformation of vertices; camerainvmat_: camera position/orientation
  int active = camswitch_->whichChild.value;
  if ((active >= 0 ) && (active < camswitch_->getNumChildren ()))
  {
    if (camswitch_->getChild (which)->nodeType () == QvNodeType::QvPerspectiveCamera)
      ge3dMultMatrix (((QvPerspectiveCamera*)camswitch_->getChild (active))->invmat_);
    else
      ge3dMultMatrix (((QvOrthographicCamera*)camswitch_->getChild (active))->invmat_);
  }
  ge3dMultMatrix (camerainvmat_);
  ge3d_get_and_pop_matrix (mat);

  // transform matrix to rotation as used in VRML
  vector3D v = { 0, 0, 0};
  float s = 0;
  matrix2quaternion (mat, v, s);

  vector3D axis = { 0, 0, 0};
  float angle = 0;
  quaternion2rotation (v, s, axis, angle);

  point3D& position = *(vector3D*)mat[3];
  float viewangle = M_PI / 4;

  applyViewpoint (which, position, axis, angle, viewangle);

  rebuild (0);

} // setViewpoint


void VRMLScene::updateViewpointDialog ()
{
  if (!viewpointdialog_)
    return;

  if (!camswitch_)
  { // make viewpoint node
    camswitch_ = new QvSwitch ();
    QvName name ("Cameras");
    camswitch_->setName (name);
    QvGroup* parent = (QvGroup*)root_;
    if ((parent->getNumChildren () == 1) && (parent->getChild (0)->isGroupNode ()))  // regular case
      parent = (QvGroup*)parent->getChild (0);
    parent->getChildren ()->insert (0, camswitch_);
  }

  // clear file browser
  viewpointdialog_->clear ();

  // update file browser
  QvNode* child;
  const char* name;
  char buf[4];
  for (int i = 0; i < camswitch_->getNumChildren (); i++)
  { child = camswitch_->getChild (i);
    name = child->objName->getString ();
    sprintf (buf, "%2d", i + 1);
    viewpointdialog_->append (buf, name);
  }

  // select active camera
  int selected = (int)camswitch_->whichChild.value;
  viewpointdialog_->select (selected);  // updates its values

} // updateViewpointDialog


void VRMLScene::updateViewpointValues (int which)
{
  if (!viewpointdialog_ || !camswitch_)
    return;
  
  if (which < 0 || which >= camswitch_->getNumChildren ())
  { viewpointdialog_->clearValues ();
    return;
  }
  
  const char* name = 0;
  float* pos = 0;
  float* axis = 0;
  float angle = 0;
  float viewangle = 0;
  int camtype = QvNodeType:: QvPerspectiveCamera;

  if (camswitch_->getChild (which)->nodeType () == QvNodeType::QvPerspectiveCamera)
  {
    QvPerspectiveCamera* camera = (QvPerspectiveCamera*)camswitch_->getChild (which);
    name = camera->objName->getString ();
    pos = camera->position.value;
    axis = camera->orientation.axis;
    angle = camera->orientation.angle;
    viewangle = camera->heightAngle.value;
    camtype = QvNodeType:: QvPerspectiveCamera;
  }
  else
  {
    QvOrthographicCamera* camera = (QvOrthographicCamera*)camswitch_->getChild (which);
    name = camera->objName->getString ();
    pos = camera->position.value;
    axis = camera->orientation.axis;
    angle = camera->orientation.angle;
    viewangle = camera->height.value;
    camtype = QvNodeType:: QvOrthographicCamera;
  }

  //rebuild (0); // TODO why this?
  viewpointdialog_->updateValues (name, pos, axis, angle, viewangle, qvNodeName[camtype]);

} // updateViewpointValues


void VRMLScene::switchtoViewpoint (int which)
{
  if ((!camswitch_) || (which < 0) || (which >= camswitch_->getNumChildren ()))
    return;

  QvNode* camera = camswitch_->getChild (which);
  const char* name = camera->objName->getString ();

  // force rebuild after camera selection
  if (which == camswitch_->whichChild.value)
  { activepcam_ = 0;
    activeocam_ = 0;
  }

  // select camera
  if (camera->nodeType () == QvNodeType::QvPerspectiveCamera)
    activateCamera (name, (QvPerspectiveCamera*)camera, 0);
  else 
    activateCamera (name, 0, (QvOrthographicCamera*)camera);

} // switchtoViewpoint




// adds a viewpoint at the end and sets it to the current camera matrix

void VRMLScene::addViewpoint ()
{
  if (!camswitch_)
    return;

  QvPerspectiveCamera* camera = new QvPerspectiveCamera;
  // camera->heightAngle.value = camera_->getfovy ();  // wrong too on first camera
  camswitch_->getChildren ()->append (camera);
  // rebuild (0);  // TODO build only new camera?  // done in setViewpoint

  int which = camswitch_->getNumChildren () - 1;
  setViewpoint (which);
  switchtoViewpoint (which);

  updateViewpointDialog ();
}

void VRMLScene::deleteViewpoint (int which)
{
  if (!camswitch_ || (which < 0) || (which >= camswitch_->getNumChildren ()))
    return;

  camswitch_->getChildren ()->remove (which);
  rebuild (0);  // TODO remove ?

  camswitch_->whichChild.value = -1;
  updateViewpointDialog ();
  viewAll ();

  scene_->sceneManipulated (1);

} // deleteViewpoint


/****** Constraints ******/


// calculates matrix and inverse matrix of actual camera

void VRMLScene::calculateCameraMatrix ()
{
  ge3dPushIdentity ();
  camera_->setCamera (scene_);
  ge3d_get_and_pop_matrix (cameramat_);

  copymatrix (cameramat_, camerainvmat_);
  invertmatrix (camerainvmat_);

} // computeCameraMatrix



// calculates boundingbox of the selected object and returns also the world matrix of the bounding box
// cameramat_ must be set before calling this function (use calculateCameraMatrix ())
// currently not needed; may be useful later again

void VRMLScene::getBoundingboxObject (point3D& min, point3D& max, matrix4D mat) const
{
  min = sel_omin_;
  max = sel_omax_;

  if (mat)
  { 
    const matrix4D& selectedtransformation = scene_->selectedNode ()->selectedTransformation_;
    ge3dPushIdentity ();
    ge3dMultMatrix (camerainvmat_);
    ge3dMultMatrix (selectedtransformation);  // before the actual transformrotation
    ge3d_get_and_pop_matrix (mat);
  }

/*
  // calculate world coordinates of bounding box
  ge3dPushIdentity ();
  ge3dMultMatrix (mat);
  ge3dTransformMcWc (&sel_omin_, &min);
  ge3dTransformMcWc (&sel_omax_, &max);
  ge3d_pop_matrix ();
*/

} // getBoundingboxObject


// get bounding box without selected transformation applied

void VRMLScene::getBoundingboxLocal (point3D& min, point3D& max, matrix4D mat) const
{
  ge3dPushIdentity ();
  ge3dMultMatrix ((const float (*)[4]) sel_transform_->mat_);
  computeBoundingbox (sel_omin_, sel_omax_, min, max);
  ge3d_pop_matrix ();

  if (mat)
  { // calculate world coordinates for the selected object
    const matrix4D& selectedtransformationT = scene_->selectedNode ()->selectedTransformationT_;
    ge3dPushIdentity ();
    ge3dMultMatrix (camerainvmat_);
    ge3dMultMatrix (selectedtransformationT);  // before the actual transformrotation
    ge3d_get_and_pop_matrix (mat);
  }

/*
  // calculate world coordinates of bounding box
  ge3dPushIdentity ();
  ge3dMultMatrix (mat);
  ge3dTransformMcWc (&lmin, &min);
  ge3dTransformMcWc (&lmax, &max);
  ge3d_pop_matrix ();
*/

} // getBoundingboxLocal


// bounding box in world coordinates
// needs not return a matrix (identity matrix)

void VRMLScene::getBoundingboxWorld (point3D& min, point3D& max) const
{
  const matrix4D& selectedtransformation = scene_->selectedNode ()->selectedTransformation_;

  ge3dPushIdentity ();
  ge3dMultMatrix (camerainvmat_);
  ge3dMultMatrix (selectedtransformation);  // after the actual transformrotation
  computeBoundingbox (sel_omin_, sel_omax_, min, max);
  ge3d_pop_matrix ();

} // getBoundingboxWorld


void VRMLScene::pickTranslation (vector3D& trans)
{
  // get boundingbox
  point3D min, max;
  matrix4D matrix;
//  calculateCameraMatrix ();
  getBoundingboxLocal (min, max, matrix);

  // center and length (center to border) of boundingbox
  point3D center, length;
  add3D (min, max, center);
  scl3D (center, 0.5);
  sub3D (max, min, length);
  scl3D (length, 0.5);

  point3D A, Aw;
  vector3D b;

  float sign_a[3] = { fsgnf (trans.x), fsgnf (trans.y), fsgnf (trans.z) };
  float* trans_a = &trans.x;
  float* length_a = &length.x;
  float* A_a = &A.x;
  float distance;

  ge3dPushIdentity ();
  ge3dMultMatrix (matrix);  // for ge3dTransformMcWc
  ge3d_setlinecolor (0.0, 1.0, 0.0);

  int i = 3;
  while (i--)
    if (sign_a[i])
    {  b = *(vector3D*)matrix[i];
       scl3D (b, sign_a[i]);
       
       A = center;
       A_a[i] += sign_a[i] * length_a[i];
       ge3dTransformMcWc (&A, &Aw);

       if (pickObject (Aw, b, epsilon, 10, 0, 0, 0, /*&hitpoint*/0, 0, 0, 0, &distance))
       { distance -= 2 * epsilon;
         if (fabs (trans_a[i]) > distance)
           trans_a[i] = sign_a[i] * distance;
       }
/*
       // debug
       point3D test = Aw;
       inc3D (test, b);
       ge3dPushIdentity ();
       ge3dMultMatrix (cameramat_);
       ge3dLine (&Aw, &test);
       ge3d_pop_matrix ();
*/
    }

  ge3d_pop_matrix ();


/*// debug
  calculateCameraMatrix ();
  getBoundingboxLocal (min, max, matrix);
  ge3dPushIdentity ();
  ge3dMultMatrix (cameramat_);
  ge3dMultMatrix (matrix);
  ge3d_setlinecolor (0.8, 0.8, 0.8);
  ge3dWireCube (&min, &max);
  ge3d_pop_matrix ();
*/
} // pickTranslation



void VRMLScene::pickScale (vector3D& scale)
{
  // get boundingbox
  point3D min, max;
  matrix4D matrix;
//  calculateCameraMatrix ();
  getBoundingboxObject (min, max, matrix);

  // center and length (center to border) of boundingbox
  point3D center, length;
  add3D (min, max, center);
  scl3D (center, 0.5);
  sub3D (max, min, length);
  scl3D (length, 0.5);

  point3D A, Aw;  // start point, start point in world coordinates
  vector3D b;     // pick direction

  float sign[2] = { 1.0, -1.0 };
  float* scale_a = &scale.x;
  float* length_a = &length.x;
  float* A_a = &A.x;
  float distance = MAXFLOAT;
  float temp = 0;

  ge3dPushIdentity ();
  ge3dMultMatrix (matrix);  // for ge3dTransformMcWc
  ge3d_setlinecolor (0.0, 1.0, 0.0);

  // send picking ray in each direction, which scaling value is positive
  distance = MAXFLOAT;
  int i = 3, j;
  while (i--)
    if (scale_a[i] > 0)
    { j = 2;
      while (j--)
      { b = *(vector3D*)matrix[i];
        scl3D (b, sign[j]);
       
        A = center;
        A_a[i] += sign[j] * length_a[i];
        ge3dTransformMcWc (&A, &Aw);

        if (pickObject (Aw, b, epsilon, 10, 0, 0, 0, /*&hitpoint*/0, 0, 0, 0, &temp))
          if (distance > temp)
            distance = temp;
/*
        // debug
        point3D test = Aw;
        inc3D (test, b);
        ge3dPushIdentity ();
        ge3dMultMatrix (cameramat_);
        ge3dLine (&Aw, &test);
        ge3d_pop_matrix ();
*/
      }
    }
  
  int apply = 0;
  i = 3;
  distance -= 2 * epsilon;
  while (i--)
    if ((scale_a[i] > distance) && (scale_a[i] > 0))
      apply = 1;

  if (apply)
  { i = 3;
    while (i--)
      if (scale_a[i] > 0)
        scale_a[i] = distance;
  }

  ge3d_pop_matrix ();

/*  
  // debug
  ge3dPushIdentity ();
  ge3dMultMatrix (cameramat_);
  ge3dMultMatrix (matrix);

  ge3dLine (&min, &max);
  ge3d_pop_matrix ();
*/
} // pickScale


/*** grid ***/


void VRMLScene::updateGridDialog ()
{
  if (griddialog_)
    griddialog_->update ();
}


void VRMLScene::setGrid (const vector3D& axis, const vector3D& position, float extent, float distance)
{ 
  grid_ = axis;
  grid_position_ = position;
  grid_extent_ = extent; // diagonal of bounding cube is 1
  grid_distance_ = distance;  // relative to extent

} // setGrid


// draws the grid

void VRMLScene::drawGrid ()
{
  if (!(grid_.x || grid_.y || grid_.z))
    return;

   // extent of scene
  vector3D se = root_->omax_;
  dec3D (se, root_->omin_);
  float diagonal = scene_->size ();  // sqrt (se.x * se.x + se.y * se.y + se.z * se.z);
  float halfgridextent = diagonal * grid_extent_ * 0.5;
  float griddistance = grid_distance_ * diagonal;

  // middle of scene
  vector3D middle = root_->omax_;
  inc3D (middle, root_->omin_);
  scl3D (middle, 0.5);

  vector3D min, max, pos;
  min.x = middle.x - halfgridextent;
  min.y = middle.y - halfgridextent;
  min.z = middle.z - halfgridextent;
  max.x = middle.x + halfgridextent;
  max.y = middle.y + halfgridextent;
  max.z = middle.z + halfgridextent;
  pos.x = middle.x + grid_position_.x * se.x;
  pos.y = middle.y + grid_position_.y * se.y;
  pos.z = middle.z + grid_position_.z * se.z;

  ge3d_setlinecolor (0, 0.5, 0.8);
  ge3d_setlinestyle (0x7777);  // dashed

  // x-axis
  float f = min.x; // variable in loop
  while (f <= max.x)
  { 
    if (grid_.y)
      ge3d_line (f, pos.y, min.z, f, pos.y, max.z);
    if (grid_.z)
      ge3d_line (f, min.y, pos.z, f, max.y, pos.z);
    f += griddistance;
  }
  f = max.x;
  if (grid_.y)
    ge3d_line (f, pos.y, min.z, f, pos.y, max.z);
  if (grid_.z)
    ge3d_line (f, min.y, pos.z, f, max.y, pos.z);

  // y-axis
  f = min.y;
  while (f <= max.y)
  { 
    if (grid_.x)
      ge3d_line (pos.x, f, min.z, pos.x, f, max.z);
    if (grid_.z)
      ge3d_line (min.x, f, pos.z, max.x, f, pos.z);
    f += griddistance;
  }
  f = max.y;
  if (grid_.x)
    ge3d_line (pos.x, f, min.z, pos.x, f, max.z);
  if (grid_.z)
    ge3d_line (min.x, f, pos.z, max.x, f, pos.z);

  // z-axis
  f = min.z;
  while (f <= max.z)
  { 
    if (grid_.x)
      ge3d_line (pos.x, min.y, f, pos.x, max.y, f);
    if (grid_.y)
      ge3d_line (min.x, pos.y, f, max.x, pos.y, f);
    f += griddistance;
  }
  f = max.z;
  if (grid_.x)
    ge3d_line (pos.x, min.y, f, pos.x, max.y, f);
  if (grid_.y)
    ge3d_line (min.x, pos.y, f, max.x, pos.y, f);

  ge3d_setlinestyle (-1);  // solid

} // drawGrid


/**** Debug functions *****/

// for debug only


void VRMLScene::debugScenegraph ()
{
  if (!scene_->manipulationAllowed ())
    return;
  cerr << "----------------------------\n";
/*
  // undo
  cerr << "  undoempty = " << undoempty_ 
       << ", undostart = " << undostart_ 
       << ", undonext = " << undonext_ << endl;


  if (scene_->selectedNode ())
    cerr << "selected node = " << qvNodeName[scene_->selectedNode ()->nodeType ()]
         << "  ref = " << scene_->selectedNode ()->refCount << endl;

  if (scene_->selectedPath ())
    cerr << "selected path = " << *(scene_->selectedPath ());
  for (int i = 0; i < undonext_; i++)
  {
    printPath (&undowhichpath_[i]);
    cerr << "    " << qvNodeName[undowhichnode_[i]->nodeType ()] << endl;
  }
*/

/*
  // selections
  cerr << "---\n";
  if (paste_)
    cerr << "paste = " << qvNodeName[paste_->nodeType ()] << "  ref = " << paste_->refCount << endl;

  if (scene_->selectedNode ())
    cerr << "selected node = " << qvNodeName[scene_->selectedNode ()->nodeType ()]
         << "  ref = " << scene_->selectedNode ()->refCount << endl;

  if (scene_->selectedPath ())
    cerr << "selected path = " << *(scene_->selectedPath ());
*/


  // scene graph
  IntArray path;
  debugParse ((QvGroup*)root_, &path);
}


void VRMLScene::debugParse (QvGroup* group, IntArray* path)
{
  if (!group || !path)
    return;

  QvNode* child;
  int numchildren = group->getNumChildren ();
  int i = -1;
  path->append (0);
  while (++i < numchildren)
  { child = group->getChild (i);
    path->set_value (path->count () - 1, i);
    cerr << "(" << child->refCount << ") ";
    if (scene_->selectedPath () && (*path == *scene_->selectedPath ()))
      if (child == scene_->selectedNode ())
        cerr << "** ";
      else
        cerr << "*! ";
    else if (child == sel_transform_)
      cerr << "-- ";
    else if (child == sel_wwwanchor_)
      cerr << "-> ";
    else if (child == sel_texture_)
      cerr << "t2 ";
    else if (child == sel_texturetransform_)
      cerr << "tt ";
    else 
      cerr << "   ";
    printPath (path);
    cerr << "           " 
         << qvNodeName[child->nodeType ()] << "  " 
         << child->getName ().getString () << " ("
         << child->hasextent_ << ")";
    cerr << endl;

    if (child->isGroupNode ())
      debugParse ((QvGroup*)child, path);
  }
  path->remove (path->count () - 1);

} // parseSceneGraph


void VRMLScene::printPath (IntArray* path)
{
  if (!path)
    return;

  int n = path->count ();
  int i = -1;
  while (++i < n)
    cerr << (*path)[i] << ".";
}
