/*
 *  Copyright (C) 1998-99 Luca Deri <deri@unipi.it>
 *                      
 *  			  Centro SERRA, University of Pisa
 *  			  http://www-serra.unipi.it/
 *  					
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Copyright (c) 1994, 1996
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
#define DNS_SNIFF_DEBUG
#define DNS_DEBUG
#define GDBM_DEBUG
#define PURGE_DEBUG
*/

/* #define PRINT_UNKNOWN_PACKETS */

#include "ntop.h"

/*
#define DEBUG
#define DEBUG_THREADS
*/

/* Extern */
extern int newSock, datalink;
extern struct in_addr localHostAddress;
extern u_short numIpProtosToMonitor;
extern void sendString(char *x);
extern void usage(int);
extern void createVendorTable();
extern unsigned int localnet, netmask, webMode;
extern RETSIGTYPE cleanup(int signo);
extern int numericFlag, refreshRate, localAddrFlag, percentMode, idleFlag, logFlag;
extern FILE *logd;
extern short screenNumber, columnSort;
extern void printHeader(int);
extern void printHelp();
extern void handleDNSpacket(const u_char*, u_short, DNSHostInfo *hostPtr, short len);
extern char* getSpecialMacInfo(u_char* ethAddress, short);
extern void printSession(IPSession *theSession, u_short sessionType, 
			 u_short sessionCounter);
extern unsigned char isLocalAddress(struct in_addr *addr);
extern unsigned char isPseudoLocalAddress(struct in_addr *addr);
extern unsigned char isBroadcastAddress(struct in_addr *addr);
extern char* getHostOS(char* osName, int port);
extern int32_t gmt2local(time_t t);
extern int mapGlobalToLocalIdx(int port);
extern short isMulticastAddress(struct in_addr *addr);
extern void updateHostNameInfo(unsigned long numeric, char* symbolic);

extern time_t actTime, initialSniffTime;
extern TrafficCounter lastTotalPkts, 
  lastBroadcastPkts, lastMulticastPkts,
  lastEthernetBytes, lastIpBytes, lastNonIpBytes;
extern FlowFilterList *flowsList;
#ifdef MULTITHREADED
extern pthread_mutex_t packetQueueMutex, hostsHashMutex;
extern pthread_t dequeueThreadId;
PacketInformation packetQueue[PACKET_QUEUE_LENGTH+1];
unsigned int packetQueueLen, maxPacketQueueLen, packetQueueHead, packetQueueTail;
extern TrafficCounter droppedPackets;
#ifdef USE_SEMAPHORES
extern sem_t queueSem;
#else
extern ConditionalVariable queueCondvar;
#ifdef ASYNC_ADDRESS_RESOLUTION
extern ConditionalVariable queueAddressCondvar;
#endif
#endif
extern char* llcsap_string(u_char sap);
extern char* etheraddr_string(const u_char *ep);
extern void extract_fddi_addrs(const struct fddi_header *fddip, 
			       char *fsrc, char *fdst);

/* *** SQL Engine *** */
extern void updateHostTraffic(HostTraffic *el);
extern void notifyHostCreation(HostTraffic *el);
extern void updateDBOSname(HostTraffic *el);
extern void updateDbHostsTraffic();
extern void notifyTCPSession(IPSession *session);
/* ****************** */

#endif

#ifdef HAVE_LSOF /* util.c */
extern ProcessInfo *processes[MAX_NUM_PROCESSES];
extern u_short numProcesses;
extern ProcessInfoList *localPorts[TOP_IP_PORT];
extern u_short updateLsof;
#ifdef MULTITHREADED
extern pthread_mutex_t lsofMutex;
#endif
#endif
#ifdef HAVE_GDBM_H
extern GDBM_FILE gdbm_file;
#ifdef MULTITHREADED
extern pthread_mutex_t gdbmMutex;
#endif
#endif
extern char* intoa(struct in_addr addr);
extern struct enamemem enametable[HASHNAMESIZE];

/* Global */
unsigned long numThptSamples;
float  last60MinutesThpt[60], last24HoursThpt[24], last30daysThpt[30];
unsigned short last60MinutesThptIdx, last24HoursThptIdx, last30daysThptIdx;
u_char dummyEthAddress[ETHERNET_ADDRESS_LEN];
short sortSendMode=0;
int lastNumLines, lastNumCols;
time_t nextSessionTimeoutScan;
struct timeval lastPktTime;
PacketStats rcvdPktStats;
SimpleProtoTrafficInfo tcpGlobalTrafficStats, udpGlobalTrafficStats, icmpGlobalTrafficStats;
u_int broadcastEntryIdx;
unsigned short alternateColor = 0, maxNameLen;

/* Static */
static time_t lastThptUpdate, lastMinThptUpdate, lastFiveMinsThptUpdate;
static TrafficCounter throughput;
static HostTraffic broadcastEntry;
static u_int hlen;

/* Global */
char domainName[MAXHOSTNAMELEN], *shortDomainName;
jmp_buf getname_env;
float peakThroughput, actualThpt, lastMinThpt, lastFiveMinsThpt;
#ifndef HAVE_GDBM_H
struct hnamemem* hnametable[HASHNAMESIZE];
#endif
struct hnamemem eprototable[HASHNAMESIZE];
IpFragment *fragmentTable[HASHNAMESIZE];
HostTraffic* hash_hostTraffic[HASHNAMESIZE];
u_int numHostTrafficEntries;
u_short	extracted_ethertype;
TrafficCounter ethernetPkts, broadcastPkts, multicastPkts;
TrafficCounter ethernetBytes, ipBytes;
TrafficCounter lastMinEthernetBytes, lastFiveMinsEthernetBytes;
IPSession *tcpSession[HASHNAMESIZE]; /* TCP sessions */
IPSession *udpSession[HASHNAMESIZE]; /* UDP sessions */
u_short numTcpSessions=0, numUdpSessions=0;
IpGlobalSession *Gsessions[HASHNAMESIZE]; /* Sessions Global Information  */
char *separator;
ServiceEntry *udpSvc[SERVICE_HASH_SIZE], *tcpSvc[SERVICE_HASH_SIZE];
TrafficEntry ipTrafficMatrix[256][256]; /* Subnet traffic Matrix */
HostTraffic* ipTrafficMatrixHosts[256]; /* Subnet traffic Matrix Hosts */
u_char ipTrafficMatrixPromiscHosts[256];
int32_t thisZone; /* seconds offset from gmt to local time */
SimpleProtoTrafficInfo *ipProtoStats;
TrafficCounter tcpBytes, udpBytes, icmpBytes, dlcBytes, ipxBytes, decnetBytes, netbiosBytes, arpRarpBytes, atalkBytes, otherBytes, ospfBytes, egpBytes, igmpBytes, osiBytes, otherIpBytes;

/* Forward */
char* ipaddr2str(struct in_addr hostIpAddress);
char* savestr(const char *str);
RETSIGTYPE printHostsTraffic(int signumber_ignored, int);
void updateThpt();
void purgeIdleHosts(int);
char* etheraddr_string(const u_char *ep);
void updatePacketCount(u_int srcHost, u_int dstHost, TrafficCounter length);
void updateTrafficMatrix(u_int srcHost, u_int dstHost, TrafficCounter length);
static void freeHostInfo(u_int host);
void purgeOldFragmentEntries();
void resolveAddress(char* symAddr, struct in_addr *hostAddr, short keepAddressNumeric);


/* ************************************ */

u_int checkSessionIdx(u_int idx) {

  if((idx < 0) ||  (idx > HASHNAMESIZE))
    printf("Index error (idx=%u)!!!!\n", idx);
  
  return(idx);
}

/* ************************************ */

#ifdef HAVE_CURSES

void clrscr()
{
  attrset (A_NORMAL);
  erase();

  /* border(0, 0, 0, 0, 0, 0, 0, 0); */
  /* refresh(); */
}

#endif

/* ******************************* */

char* getRowColor() {
#define USE_COLOR

#ifdef USE_COLOR
  if(alternateColor == 0) {
    alternateColor = 1;
    /* return("BGCOLOR=#FFFFCC"); */
    return("BGCOLOR=#EEEEEE");
  } else {
    alternateColor = 0;
    return("");
  }
#else
  return("");
#endif
}

/* ******************************* */

char* getActualRowColor() {
#define USE_COLOR

#ifdef USE_COLOR
  if(alternateColor == 1) {
    /* return("BGCOLOR=#FFFFCC"); */
    return("BGCOLOR=#EEEEEE");
  } else 
    return("");
#else
  return("");
#endif
}

/* ******************************* */

void init_curses ()
{
#ifdef HAVE_CURSES
  if(webMode)
    return;

  initscr();
  cbreak();
  noecho();
  nodelay(stdscr, TRUE);
  nonl();
  intrflush(stdscr, FALSE);
  keypad(stdscr, TRUE);
 
  attrset(A_NORMAL);

  lastNumLines = LINES;
  lastNumCols = COLS;

  printHeader(0);
#endif
}

/* ******************************* */

void cleanup_curses ()
{
#ifdef HAVE_CURSES
  if(webMode) return;
  endwin ();
#endif
}

/* ******************************* */

void resetStats() {
  ethernetPkts = 0;
  ethernetBytes = 0;
  ipBytes = 0;
  broadcastPkts = 0;
  multicastPkts = 0;
  peakThroughput = 0;
  throughput = 0;
  lastThptUpdate = lastMinThptUpdate = lastFiveMinsThptUpdate = time(NULL);
  lastMinEthernetBytes = lastFiveMinsEthernetBytes = 0;

  memset(&hash_hostTraffic, 0, sizeof(hash_hostTraffic));
  numHostTrafficEntries = 0;
  hash_hostTraffic[broadcastEntryIdx] = &broadcastEntry;

  memset(ipTrafficMatrixPromiscHosts, 0, sizeof(ipTrafficMatrixPromiscHosts));
  memset(&tcpGlobalTrafficStats, 0, sizeof(tcpGlobalTrafficStats));
  memset(&udpGlobalTrafficStats, 0, sizeof(udpGlobalTrafficStats));
  memset(&icmpGlobalTrafficStats, 0, sizeof(icmpGlobalTrafficStats));
}

/* ******************************* */

int getPortByName(ServiceEntry **theSvc, char* portName) {
  int idx;

  for(idx=0; idx<SERVICE_HASH_SIZE; idx++) {

#ifdef DEBUG
    if(theSvc[idx] != NULL) 
      printf("%d/%s [%s]\n", 
	     theSvc[idx]->port, 
	     theSvc[idx]->name, portName);
#endif

    if((theSvc[idx] != NULL) 
       && (strcmp(theSvc[idx]->name, portName) == 0))
      return(theSvc[idx]->port);
  }

  return(-1);
}

/* ******************************* */

char* getPortByNumber(ServiceEntry **theSvc, int port) {
  int idx = port % SERVICE_HASH_SIZE;
  ServiceEntry *scan;

  for(;;) {
    scan = theSvc[idx];

    if((scan != NULL) && (scan->port == port))
      return(scan->name);
    else if(scan == NULL)
      return(NULL);
    else
      idx = (idx+1) % SERVICE_HASH_SIZE;
  }
}

/* ******************************* */

char* getPortByNum(int port, int type) {
  char* rsp;

  if(type == IPPROTO_TCP) {
    rsp = getPortByNumber(tcpSvc, port);
  } else {
    rsp = getPortByNumber(udpSvc, port);
  }

  return(rsp);
}

/* ******************************* */

char* getAllPortByNum(int port) {
  char* rsp;
  static char staticBuffer[16];

  rsp = getPortByNumber(tcpSvc, port); /* Try TCP first... */
  if(rsp == NULL)
    rsp = getPortByNumber(udpSvc, port);  /* ...then UDP */

  if(rsp != NULL)
    return(rsp);
  else {
    sprintf(staticBuffer, "%d", port);
    return(staticBuffer);
  }
}

/* ******************************* */

int getAllPortByName(char* portName) {
  int rsp;

  rsp = getPortByName(tcpSvc, portName); /* Try TCP first... */
  if(rsp == -1)
    rsp = getPortByName(udpSvc, portName);  /* ...then UDP */
  
  return(rsp);
}


/* ******************************* */

void addPortHashEntry(ServiceEntry **theSvc, int port, char* name) {
  int idx = port % SERVICE_HASH_SIZE;
  ServiceEntry *scan;

  for(;;) {
    scan = theSvc[idx];

    if(scan == NULL) {
      theSvc[idx] = (ServiceEntry*)malloc(sizeof(ServiceEntry));
      theSvc[idx]->port = (u_short)port;
      theSvc[idx]->name = strdup(name);
      break;
    } else
      idx = (idx+1) % SERVICE_HASH_SIZE;
  }
}

/* ******************************* */

RETSIGTYPE windowSizeChanged(int signumber_ignored) {
  lastNumLines = 0; /* force redraw */
  printHostsTraffic(signumber_ignored, 0);
}

/* ******************************* */

void initIPServices() {
#ifdef WIN32
  FILE* fd;
#else
  struct servent *svnt;
#endif
  memset(udpSvc, 0, sizeof(udpSvc));
  memset(tcpSvc, 0, sizeof(tcpSvc));

#ifndef WIN32
  setservent(1); /* used by getservbyXXX */

  while(((svnt = getservent()) != NULL) 
	&& (svnt->s_port < TOP_IP_PORT))
    /* Courtesy of Bernhard Heibler <bernhard_h@sz-testsys.de> */
    {
      if(strcmp(svnt->s_proto, "tcp") == 0)
	addPortHashEntry(tcpSvc, (int)ntohs((unsigned short int)svnt->s_port), svnt->s_name);
      else
	addPortHashEntry(udpSvc, (int)ntohs((unsigned short int)svnt->s_port), svnt->s_name);
    }

  setservent(0); /* used by getservbyXXX */
#else
  fd = fopen("services", "r");

  if(fd != NULL) {
    char tmpStr[512];

    while(fgets(tmpStr, 512, fd))
      if((tmpStr[0] != '#') && (strlen(tmpStr) > 10)) {
	/* discard		9/tcp		sink null */
	char name[64], port[16], *proto;
	unsigned short int numPort;

	/* printf(" -> '%s'\n", tmpStr); */
	sscanf(tmpStr, "%s %s", &name, &port);
	/* printf("'%s' - '%s'\n", name, port); */
	if(name[0] != '\0') {
	  numPort = atoi(strtok(port, "/"));
	  proto = strtok(NULL, " ");

	  if(strcmp(proto, "tcp") == 0)
	    addPortHashEntry(tcpSvc, (int)ntohs(numPort), proto);
	  else
	    addPortHashEntry(udpSvc, (int)ntohs(numPort), proto);
	}
      }
    fclose(fd);
  } else {
    printf("WARNING: unable to read file /etc/services\n");
  }

#endif

  /* Add some basic services, just in case they
     are not included in /etc/services */
  addPortHashEntry(tcpSvc, 21,  "ftp");
  addPortHashEntry(tcpSvc, 20,  "ftp-data");
  addPortHashEntry(tcpSvc, 80,  "http");
  addPortHashEntry(tcpSvc, 443, "https");
  addPortHashEntry(tcpSvc, 42,  "name");
  addPortHashEntry(tcpSvc, 23,  "telnet");
  addPortHashEntry(udpSvc, 137, "netbios-ns");
  addPortHashEntry(tcpSvc, 137, "netbios-ns");
  addPortHashEntry(udpSvc, 138, "netbios-dgm");
  addPortHashEntry(tcpSvc, 138, "netbios-dgm");
  addPortHashEntry(udpSvc, 139, "netbios-ssn");
  addPortHashEntry(tcpSvc, 139, "netbios-ssn");
  addPortHashEntry(tcpSvc, 109, "pop-2");
  addPortHashEntry(tcpSvc, 110, "pop-3");
  addPortHashEntry(tcpSvc, 1109,"kpop");
  addPortHashEntry(udpSvc, 161, "snmp");
  addPortHashEntry(udpSvc, 162, "snmp-trap");
  addPortHashEntry(udpSvc, 635, "mount");
  addPortHashEntry(udpSvc, 640, "pcnfs");
  addPortHashEntry(udpSvc, 650, "bwnfs");
  addPortHashEntry(udpSvc, 2049,"nfsd");
  addPortHashEntry(udpSvc, 1110,"nfsd-status");
}

/* ******************************* */

void init_counters() {
#ifndef WIN32
  char *p;
#endif
  int len;

  /* (void)setsignal(SIGWINCH, windowSizeChanged);  */

#ifndef WIN32
  /* 
   * The name of the local domain is now calculated properly
   * Kimmo Suominen <kim@tac.nyc.ny.us> 
   */

  if(domainName[0] == '\0') {
    if((getdomainname(domainName, MAXHOSTNAMELEN) != 0) 
       || (domainName[0] == '\0') 
       || (strcmp(domainName, "(none)") == 0))
      {
	if ((gethostname(domainName, MAXHOSTNAMELEN) == 0)
	    && ((p = memchr(domainName, '.', MAXHOSTNAMELEN)) != NULL)) {
	  domainName[MAXHOSTNAMELEN - 1] = '\0';
	  /* 
	   * Replaced memmove with memcpy 
	   * Igor Schein <igor@txc.com>
	   */
	  memcpy(domainName, ++p, (MAXHOSTNAMELEN+domainName-p));

	} else 
	  domainName[0] = '\0';
      }

    if(domainName[0] == '\0') {
      char szLclHost[64];
      struct hostent *lpstHostent;
      struct in_addr stLclAddr;

      gethostname(szLclHost, 64); 
      lpstHostent = gethostbyname(szLclHost);
      if (lpstHostent) {
	struct hostent *hp;
	  
	stLclAddr.s_addr = ntohl(*(lpstHostent->h_addr));
	  
	hp = (struct hostent*)gethostbyaddr((char*)lpstHostent->h_addr, 4, AF_INET);
	  
	if (hp && (hp->h_name)) {
	  char *dotp = (char*)hp->h_name;    
	  int i;
	    
	  for(i=0; (dotp[i] != '\0') && (dotp[i] != '.'); i++)
	    ;
	    
	  if(dotp[i] == '.')
	    strcpy(domainName, &dotp[i+1]);
	}
      }	
    }

    if(domainName[0] == '\0') {
      /* Last chance.... */
      /* strcpy(domainName, "please_set_your_local_domain.org"); */
      ;
    }
  }
#endif

  len = strlen(domainName)-1;

  while((len > 0) && (domainName[len] != '.'))
    len--;
  
  if(len == 0)
    shortDomainName = domainName;
  else
    shortDomainName = &domainName[len+1];

  resetStats();
  createVendorTable();
  columnSort = 0;

  if(webMode) {
    separator = "&nbsp;";
    maxNameLen = MAXHOSTNAMELEN;
  } else {
    separator = " ";
    maxNameLen = MAX_HOST_NAME_LEN;
  }

  memset(dummyEthAddress, 0, ETHERNET_ADDRESS_LEN);
#ifndef HAVE_GDBM_H
  memset(hnametable, 0, sizeof(hnametable));
#endif
  memset(enametable, 0, sizeof(enametable));
  memset(eprototable, 0, sizeof(eprototable));
  memset(hash_hostTraffic, 0, sizeof(hash_hostTraffic));
  numHostTrafficEntries = 0;
  memset(tcpSession, 0, sizeof(tcpSession));
  memset(udpSession, 0, sizeof(udpSession));
  memset(Gsessions, 0, sizeof(Gsessions));

  nextSessionTimeoutScan = time(NULL)+SESSION_SCAN_DELAY;
  thisZone = gmt2local(0);

  lastTotalPkts = lastBroadcastPkts = lastMulticastPkts = 0;
  lastEthernetBytes = lastIpBytes = lastNonIpBytes = 0;
  memset(&rcvdPktStats, 0, sizeof(rcvdPktStats));
  len = (size_t)numIpProtosToMonitor*sizeof(SimpleProtoTrafficInfo);
  ipProtoStats = (SimpleProtoTrafficInfo*)malloc(len);
  memset(ipProtoStats, 0, len);
  tcpBytes = udpBytes = icmpBytes = dlcBytes = ipxBytes = netbiosBytes = 
    decnetBytes = arpRarpBytes = atalkBytes = otherBytes = otherIpBytes = 0;

  memset(&broadcastEntry, 0, sizeof(broadcastEntry));
  broadcastEntry.broadcastHost = 
    broadcastEntry.subnetPseudoLocalHost = 
    broadcastEntry.subnetLocalHost = 1;
  broadcastEntry.hostSymIpAddress = "";

  broadcastEntryIdx = 0;
  hash_hostTraffic[broadcastEntryIdx] = &broadcastEntry;
  numHostTrafficEntries++; /* A new entry has just been added */

  memset(fragmentTable, 0, sizeof(fragmentTable));

  hlen = (datalink == DLT_NULL) ? NULL_HDRLEN : sizeof(struct ether_header);

#ifdef HAVE_LSOF
  memset(localPorts, 0, sizeof(localPorts));
#endif

  memset(last60MinutesThpt, 0, sizeof(last60MinutesThpt));
  memset(last24HoursThpt, 0, sizeof(last24HoursThpt));
  memset(last30daysThpt, 0, sizeof(last30daysThpt));
  last60MinutesThptIdx=0, last24HoursThptIdx=0, last30daysThptIdx=0;
}

/* ******************************* */

int findHostInfo(struct in_addr *hostIpAddress)
{
  int i = 0;

  for(i=0; i<HASHNAMESIZE; i++)
    if(hash_hostTraffic[i] != NULL) 
      if(hash_hostTraffic[i]->hostIpAddress.s_addr == hostIpAddress->s_addr)
	return i;
      
  return(-1);
}

/* ******************************* */

u_int getHostInfo(struct in_addr *hostIpAddress, 
		  u_char *ether_addr) 
{
  u_int idx, i, run=0;
  HostTraffic *el;
  short numCmp;
  char* symEthName = NULL;

  if((hostIpAddress != NULL) && (!isLocalAddress(hostIpAddress))) {
    /* the MAC address is meaningless because isn't local */
    idx = 0;
    memcpy(&idx, &hostIpAddress->s_addr, 4);
    numCmp = 1;
    /* printf("-> %s\n", intoa(*hostIpAddress)); */
  } else if((hostIpAddress != NULL) && isBroadcastAddress(hostIpAddress)) {
    return(broadcastEntryIdx);
  } else {
    memcpy(&idx, &ether_addr[ETHERNET_ADDRESS_LEN-sizeof(u_int)], sizeof(u_int));
    numCmp = 0;
  }

  idx = (u_int)((idx*3) % HASHNAMESIZE);

  for(i=0; i<HASHNAMESIZE; i++) {
    el = hash_hostTraffic[idx]; /* (**) */

    if(el != NULL) { 
      if(numCmp == 0) {
	if (memcmp(el->ethAddress, ether_addr, ETHERNET_ADDRESS_LEN) == 0) {
	  if((el->hostNumIpAddress[0] == '\0') && (hostIpAddress != NULL)) {
	    /* This entry didn't have IP fields set: let's set them now */
	    el->hostIpAddress.s_addr = hostIpAddress->s_addr;
	    strcpy(el->hostNumIpAddress, intoa(*hostIpAddress));

	    if(el->subnetLocalHost & (!el->broadcastHost)) {
	      int a, b, c, d;

	      sscanf(el->hostNumIpAddress, "%d.%d.%d.%d", &a, &b, &c, &d);
	      el->promiscuousMode = ipTrafficMatrixPromiscHosts[d];
	    }

	    if(numericFlag == 0) {
	      el->hostSymIpAddress = ipaddr2str(el->hostIpAddress); 
	    }

	    /* else el->hostSymIpAddress = el->hostNumIpAddress;
	       The line below isn't necessary because (**) has
	       already set the pointer */
	    el->broadcastHost = isBroadcastAddress(&el->hostIpAddress);
	  }	
	  break;
	}
      } else {
	if (el->hostIpAddress.s_addr == hostIpAddress->s_addr)
	  break;
      }
    } else { 
      /* New table entry */
      int len = (size_t)numIpProtosToMonitor*sizeof(ProtoTrafficInfo);

      el = (HostTraffic*)malloc(sizeof(HostTraffic));
      memset(el, 0, sizeof(HostTraffic));
      for(i=0; i<MAX_NUM_CONTACTED_PEERS; i++) el->contactedSentPeersIndexes[i] = NO_PEER;
      for(i=0; i<MAX_NUM_CONTACTED_PEERS; i++) el->contactedRcvdPeersIndexes[i] = NO_PEER;

      /* NOTE: el->nextDBupdate = 0 */

      el->protoIPTrafficInfos = (ProtoTrafficInfo*)malloc(len);
      memset(el->protoIPTrafficInfos, 0, len);

      hash_hostTraffic[idx] = el; /* Insert a new entry */
      numHostTrafficEntries++;

      if((hostIpAddress == NULL)
	 || ((hostIpAddress != NULL) && isLocalAddress(hostIpAddress))) {
	/* This is a local address and then the 
	   ethernet address does make sense */
	memcpy(el->ethAddress, ether_addr, ETHERNET_ADDRESS_LEN);
	symEthName = getSpecialMacInfo(el->ethAddress, webMode);
	el->ethAddressString = strdup(etheraddr_string(ether_addr));
	el->subnetLocalHost = LOCAL_HOST;
	el->subnetPseudoLocalHost = LOCAL_HOST;
      } else if(hostIpAddress != NULL) {
	/* This is packet that's being routed */
	memcpy(el->ethAddress, &hostIpAddress->s_addr, 4); /* Dummy/unique eth address */
	el->subnetLocalHost = REMOTE_HOST;
	if(isPseudoLocalAddress(hostIpAddress))
	  el->subnetPseudoLocalHost = LOCAL_HOST;
	else
	  el->subnetPseudoLocalHost = REMOTE_HOST;
      } else {
	el->subnetLocalHost = REMOTE_HOST;
	el->subnetPseudoLocalHost = REMOTE_HOST;
      }

      if(el->ethAddressString == NULL)
	el->ethAddressString = strdup("");

      if(strncmp(el->ethAddressString, "FF:", 3) == 0) {
	/* 
	   The trick below allows me not to duplicate the 
	   "<broadcast>" string in the code
	*/
	el->hostIpAddress.s_addr = INADDR_BROADCAST;
	el->broadcastHost = 1;
	strcpy(el->hostNumIpAddress, intoa(el->hostIpAddress));
	el->hostSymIpAddress = el->hostNumIpAddress;
      } else if(hostIpAddress != NULL) {
	el->hostIpAddress.s_addr = hostIpAddress->s_addr;
	strcpy(el->hostNumIpAddress, intoa(*hostIpAddress));
	el->broadcastHost = isBroadcastAddress(&el->hostIpAddress);

	if((el->subnetLocalHost == LOCAL_HOST) && (!el->broadcastHost)) {
	  int a, b, c, d;
	  
	  sscanf(el->hostNumIpAddress, "%d.%d.%d.%d", &a, &b, &c, &d);
	  el->promiscuousMode = ipTrafficMatrixPromiscHosts[d];
	}
	
	/* Trick to fill up the address cache */
	if(numericFlag == 0)
	  el->hostSymIpAddress = ipaddr2str(el->hostIpAddress); 
	else
	  el->hostSymIpAddress = el->hostNumIpAddress;
      } else {
	/* el->hostNumIpAddress == "" */
	if(symEthName[0] != '\0') {
	  char buf[255];
	  
	  sprintf(buf, "%s [MAC]", symEthName);
	  el->hostSymIpAddress = strdup(buf);
	} else
	  el->hostSymIpAddress = el->hostNumIpAddress; /* (**) */
      }

#ifdef PURGE_DEBUG
      printf("Added a new hash_hostTraffic entry [%s/%d]\n", 
	     el->hostSymIpAddress, numHostTrafficEntries);
#endif

      el->lastSeen = actTime;
      break;
    } 

    idx = (idx+1) % HASHNAMESIZE;
  }

  if(i == HASHNAMESIZE) {
    /* The hashtable is full */
    if(run == 0) {
      purgeIdleHosts(1);
    } else {
      /* No space yet: let's delete the oldest table entry */ 
      int candidate = 0;
      time_t lastSeenCandidate=0;
      HostTraffic* hostToFree;
      
      for(i=0; i<HASHNAMESIZE; i++)
	if(hash_hostTraffic[i] != NULL) {
	  if((candidate == 0) 
	     || (hash_hostTraffic[idx]->lastSeen < lastSeenCandidate)) {
	    candidate = i;
	    lastSeenCandidate = hash_hostTraffic[idx]->lastSeen;
	  }
	}

      hostToFree = hash_hostTraffic[candidate];
      freeHostInfo(candidate);
      idx = candidate; /* this is a hint for (**) */
    }

    run++;      
  }

  el->lastSeen = actTime;

  return(idx);
}

/* ******************************* */

int checkKeyPressed() {
  int theChar = getch();
  unsigned char in_char;
  int rc = 1, numScreens; 

  if((theChar != EOF) 
     && ((in_char = (unsigned char)theChar)) != 255) {
  
    /*
      char buf[32];

      sprintf(buf, ">%c-%d<", in_char, in_char);
      mvprintw(0, 40, buf); refresh();
      sleep(2);
    */
    switch(in_char) {
    case 'q':
    case 'Q':
      cleanup(-1);
      break;
    case 'l':
    case 'L':
      /* toggle host list */
      localAddrFlag = !localAddrFlag;
      break;
    case 'd':
    case 'D':
      /* toggle idle hosts */
      idleFlag = !idleFlag;
      break;
    case 'p':
    case 'P':
      /* toggle traffic percentage */
      percentMode = (percentMode+1)%3;
      break;
    case 'r':
    case 'R':
      resetStats();
      break;
    case 'n':
    case 'N':
      /* toggle numeric <-> sym ... address format */
      numericFlag = ((numericFlag+1) % 4);
      break;
    case 't':
    case 'T':
      sortSendMode = !sortSendMode;
      break;
    case 'y':
    case 'Y':
      columnSort = (columnSort+1)%4;
      break;
    case ' ': /* space */
      numScreens = ((numIpProtosToMonitor+2 /* avoid round problems */)/3);
      if((numIpProtosToMonitor%3) != 0)
	numScreens++;
      numScreens += 3; /* Protocol Screens */
      screenNumber = (screenNumber+1)%numScreens;
      break;
    case 'h':
    case 'H':
    /* default: */
      printHelp();
    }
  } else
    rc = 0;

  if(rc) 
    printHeader(0);

  return(rc);
}

/* ******************************* */

void updateThptStats() {
  static time_t lastUpdateTime=0;
  int updateTime = 60;

#ifdef DEBUG
  printf("updateThptStats()\n");
#endif

  if(lastUpdateTime == 0)
    lastUpdateTime = actTime + updateTime; /* 1 minute */
  else if(actTime > lastUpdateTime) {
    lastUpdateTime = actTime + updateTime; /* 1 minute */
    last60MinutesThpt[last60MinutesThptIdx] = actualThpt;
    last60MinutesThptIdx = (last60MinutesThptIdx+1) % 60;
    if(last60MinutesThptIdx == 0) { /* It wrapped -> 1 hour is over */
      float average=0;
      int i;

      for(i=0; i<60; i++)
	average += last60MinutesThpt[i];
      
      average /= 60;

      last24HoursThpt[last24HoursThptIdx] = average;
      last24HoursThptIdx = (last24HoursThptIdx + 1) % 24;

      if(last24HoursThptIdx == 0) {
	average=0;

	for(i=0; i<24; i++)
	  average += last24HoursThpt[i];
	
	average /= 24;
	
	last30daysThpt[last30daysThptIdx] = average;
	last30daysThptIdx = (last30daysThptIdx + 1) % 30;
      }
    }

    numThptSamples++;
  }
}

/* ******************************* */

void updateThpt() {
  time_t timeDiff, totalTime;
  int idx;
  HostTraffic *el;

  timeDiff = actTime-lastThptUpdate; 

  /* printf("%u %u %u\n", timeDiff, throughput, ethernetBytes); */

  if(timeDiff > 5 /* secs */) {
    throughput = ethernetBytes-throughput;

    lastThptUpdate = timeDiff;
    actualThpt = (float)throughput/(float)lastThptUpdate;    
    lastThptUpdate = actTime;
    
    if(actualThpt > peakThroughput)
      peakThroughput = actualThpt;

    throughput = ethernetBytes;

    if((timeDiff = actTime-lastMinThptUpdate) > 60 /* 1 minute */) {
      lastMinEthernetBytes = ethernetBytes - lastMinEthernetBytes;
      lastMinThptUpdate = timeDiff;    
      lastMinThpt = (float)lastMinEthernetBytes/(float)lastMinThptUpdate;      
      lastMinEthernetBytes = ethernetBytes;
      lastMinThptUpdate = actTime;
    }
    
    if((timeDiff = actTime-lastFiveMinsThptUpdate) > 300 /* 5 minutes */) {
      lastFiveMinsEthernetBytes = ethernetBytes - lastFiveMinsEthernetBytes;
      lastFiveMinsThptUpdate = timeDiff;
      lastFiveMinsThpt = (float)lastFiveMinsEthernetBytes/(float)lastFiveMinsThptUpdate;      
      lastFiveMinsEthernetBytes = ethernetBytes;
      lastFiveMinsThptUpdate = actTime;
    }    

    /* *********************************************** */        
    totalTime = actTime-initialSniffTime;   
    
    for(idx=0; idx<HASHNAMESIZE; idx++) {
      if((el = hash_hostTraffic[idx]) != NULL) {       
	el->actualRcvdThpt  = (float)(el->bytesReceived-el->lastBytesReceived)/timeDiff;
	if(el->peakRcvdThpt < el->actualRcvdThpt) el->peakRcvdThpt = el->actualRcvdThpt;
	if(el->peakSentThpt < el->actualSentThpt) el->peakSentThpt = el->actualSentThpt;
	el->actualSentThpt  = (float)(el->bytesSent-el->lastBytesSent)/timeDiff;
	el->averageRcvdThpt = ((float)el->bytesReceived)/totalTime;
	el->averageSentThpt = ((float)el->bytesSent)/totalTime;
	hash_hostTraffic[idx]->lastBytesSent     = hash_hostTraffic[idx]->bytesSent;
	hash_hostTraffic[idx]->lastBytesReceived = hash_hostTraffic[idx]->bytesReceived;
      }
    }

    if(webMode) updateThptStats();
  }
}

/* ************************************ */

char* savestr(const char *str)
{
  u_int size;
  char *p;
  static char *strptr = NULL;
  static u_int strsize = 0;

  size = strlen(str) + 1;
  if (size > strsize) {
    strsize = 1024;
    if (strsize < size)
      strsize = size;
    strptr = (char *)malloc(strsize);
    if (strptr == NULL) {
      fprintf(stderr, "savestr: malloc\n");
      exit(1);
    }
  }
  (void)strcpy(strptr, str);
  p = strptr;
  strptr += size;
  strsize -= size;
  return (p);
}


/* ************************************ */

char* getNamedPort(int port) {
  static char outStr[2][8];
  static short portBufIdx=0;
  char* svcName;

  portBufIdx = (portBufIdx+1)%2;

  svcName = getPortByNum(port, IPPROTO_TCP);
  if(svcName == NULL)
    svcName = getPortByNum(port, IPPROTO_UDP);

  if(svcName == NULL)
    sprintf(outStr[portBufIdx], "%d", port);
  else
    strcpy(outStr[portBufIdx], svcName);

  return(outStr[portBufIdx]);
}

/* ************************************ */

void updateHostSessionsList(u_int theHostIdx,
			    u_short port,
			    u_int remotePeerIdx,
			    IPSession *theSession,
			    u_short sessionType,
			    u_char initiator,
			    int role)
{
  /* This is a known port hence we're interested in */
  IpGlobalSession *scanner, *prevScanner;
  HostTraffic* theHost;
  int i;

  if(remotePeerIdx == NO_PEER) {
    printf("WARNING (internal error): wrong index value\n");
    return;
  } else
    theHost = hash_hostTraffic[checkSessionIdx(theHostIdx)];

  if((theHost == NULL) /* Probably the host has been deleted */
     || theHost->broadcastHost /* We could't care less of junk traffic */
     ) 
    return;

  switch(sessionType) {
  case IPPROTO_TCP: /* 6 */
    scanner = hash_hostTraffic[theHostIdx]->tcpSessionList; 
    break;
  case IPPROTO_UDP: /* 17 */
    scanner = hash_hostTraffic[theHostIdx]->udpSessionList; 
    break;
  }
    
  prevScanner = scanner;

  while(scanner != NULL) {
    if(scanner->magic != MAGIC_NUMBER) {
      printf("Magic assertion failed (2)!\n");
      scanner = NULL;
      if(prevScanner != NULL) {
	prevScanner->next = NULL;
      }
      break;
    }

    if((scanner->port == port) && (scanner->initiator == role))
      break;
	
    prevScanner = scanner;
    scanner = (IpGlobalSession*)(scanner->next);
  }
    
  if(scanner == NULL) {
    scanner = (IpGlobalSession*)malloc(sizeof(IpGlobalSession));
    memset(scanner, 0, sizeof(IpGlobalSession));
    scanner->magic = MAGIC_NUMBER;
    scanner->port = port;
    scanner->initiator = role;
    scanner->firstSeen = actTime;

    for(i=0; i<MAX_NUM_SESSION_PEERS; i++) scanner->peersIdx[i] = NO_PEER;

    /* Add the session to the session list */      
    switch(sessionType) {
    case IPPROTO_TCP:
      scanner->next = (IpGlobalSession*)(hash_hostTraffic[theHostIdx]->tcpSessionList);	
      hash_hostTraffic[theHostIdx]->tcpSessionList = scanner; /* list head */    
      break;
    case IPPROTO_UDP:
      scanner->next = (IpGlobalSession*)(hash_hostTraffic[theHostIdx]->udpSessionList);	
      hash_hostTraffic[theHostIdx]->udpSessionList = scanner; /* list head */
      break;
    }
  }

  scanner->lastSeen = actTime;
  scanner->sessionCounter++;

#ifdef DEBUG
  if(logFlag)
    printSession(theSession, sessionType, scanner->sessionCounter);
#endif

  for(i=0; i<MAX_NUM_SESSION_PEERS; i++)
    if((scanner->peersIdx[i] == NO_PEER) || (scanner->peersIdx[i] == remotePeerIdx))
      break;
  
  if((i<MAX_NUM_SESSION_PEERS)
     && ((scanner->peersIdx[i] != NO_PEER)
	 && (scanner->peersIdx[i] != remotePeerIdx)
	 && (strcmp(hash_hostTraffic[checkSessionIdx(scanner->peersIdx[i])]->hostNumIpAddress, 
		    hash_hostTraffic[checkSessionIdx(remotePeerIdx)]->hostNumIpAddress)))
     || (scanner->peersIdx[i] == NO_PEER)) {
    scanner->peersIdx[scanner->lastPeer] = remotePeerIdx;
    scanner->lastPeer = (scanner->lastPeer+1) % MAX_NUM_SESSION_PEERS;
  }

  switch(sessionType) {
  case IPPROTO_TCP:
    if(role == SERVER_ROLE) {
      scanner->bytesSent += theSession->bytesSent;
      scanner->bytesReceived += theSession->bytesReceived;
    } else {
      scanner->bytesSent += theSession->bytesReceived;
      scanner->bytesReceived += theSession->bytesSent;
    }
    break;
  case IPPROTO_UDP:
    scanner->bytesSent += theSession->bytesSent;
    scanner->bytesReceived += theSession->bytesReceived;
    break;
  }
}

/* ************************************ */

static void purgeIdleHostSessions(u_int hostIdx, 
				  IpGlobalSession *sessionScanner) 
{
  int i;
  u_int *scanner;

  while(sessionScanner != NULL) {
    scanner = sessionScanner->peersIdx;
    for(i=0; i<MAX_NUM_SESSION_PEERS; i++)
      if(scanner[i] == hostIdx)
	scanner[i] = NO_PEER;

    sessionScanner = sessionScanner->next;
  }
  
}

/* ************************************ */

/* Delayed free */
static void freeHostInfo(u_int hostIdx) {
  int i, idx, j;
#define FREE_LIST_LEN   32
  static HostTraffic *freeHostList[FREE_LIST_LEN];
  static int nextIdxToFree=0, freeListLen=0;

  HostTraffic *host = hash_hostTraffic[checkSessionIdx(hostIdx)];

  updateHostTraffic(host);

  hash_hostTraffic[hostIdx] = NULL;
  numHostTrafficEntries--;

#ifdef PURGE_DEBUG
  printf("Deleted a hash_hostTraffic entry [%s/%d]\n", 
	 host->hostSymIpAddress, numHostTrafficEntries);
#endif

  free(host->protoIPTrafficInfos);
  free(host->ethAddressString);

  if(host->osName != NULL)
    free(host->osName);

  for(i=0; i<2; i++) {
    IpGlobalSession *nextElement, *element;
    
    if(i == 0)
      element = host->tcpSessionList;
    else
      element = host->udpSessionList;
    
    while(element != NULL) {
      nextElement = element->next;
      /* 
	 The 'peers' field shouldn't be a problem because an idle host
	 isn't supposed to have any session 
      */
      free(element);
      element = nextElement;
    }
  }

#ifdef HAVE_LSOF
  for(i=0; i<MAX_NUM_PROCESSES; i++) {
    if(processes[i] != NULL) {      
      for(j=0; j<MAX_NUM_CONTACTED_PEERS; j++)
	if(processes[i]->contactedIpPeersIndexes[j] == hostIdx) 
	  processes[i]->contactedIpPeersIndexes[j] = NO_PEER;
    }
  }
#endif

  /* Check whether there are hosts that have the host being purged
     as peer */
  for(idx=0; idx<HASHNAMESIZE; idx++)
    if(hash_hostTraffic[idx] != NULL) { 
      if(hash_hostTraffic[idx]->tcpSessionList != NULL)
	purgeIdleHostSessions(hostIdx, hash_hostTraffic[idx]->tcpSessionList);
     
      if(hash_hostTraffic[idx]->udpSessionList != NULL)
	purgeIdleHostSessions(hostIdx, hash_hostTraffic[idx]->udpSessionList);	     

      for(j=0; j<MAX_NUM_CONTACTED_PEERS; j++)
	if(hash_hostTraffic[idx]->contactedSentPeersIndexes[j] == hostIdx)
	  hash_hostTraffic[idx]->contactedSentPeersIndexes[j] = NO_PEER;
   
      for(j=0; j<MAX_NUM_CONTACTED_PEERS; j++)
	if(hash_hostTraffic[idx]->contactedRcvdPeersIndexes[j] == hostIdx)
	  hash_hostTraffic[idx]->contactedRcvdPeersIndexes[j] = NO_PEER;
    }


  if(freeListLen == FREE_LIST_LEN) {
    free(freeHostList[nextIdxToFree]); /* This is the real free */
    freeHostList[nextIdxToFree] = host;
    nextIdxToFree = (nextIdxToFree+1)%FREE_LIST_LEN;
  } else
    freeHostList[freeListLen++] = host;
}

/* ************************************ */
  
void purgeIdleHosts(int ignoreIdleTime) {
  u_int idx;

#ifdef PURGE_DEBUG
  printf("Purging (%d)....\n", ignoreIdleTime);
#endif

  purgeOldFragmentEntries(); /* let's do this too */

  for(idx=0; idx<HASHNAMESIZE; idx++)
    if((idx != broadcastEntryIdx /* This entry isn't purgeable */)
       && (hash_hostTraffic[idx] != NULL) 
       && (hash_hostTraffic[idx]->instanceInUse == 0) 
       && (hash_hostTraffic[idx]->subnetLocalHost != LOCAL_HOST)) {

#ifdef DEBUG
      printf("Found a candidate!!!\n");
#endif

      updateHostTraffic(hash_hostTraffic[idx]);
      
      if((ignoreIdleTime == 0)
	 && ((hash_hostTraffic[idx]->lastSeen+IDLE_HOST_PURGE_TIMEOUT) > actTime)) {
	continue;
      } else {
	freeHostInfo(idx);
      }
    }
}

/* ************************************ */

void scanTimedoutTCPSessions() {
  int idx;

#ifdef DEBUG
  printf("Called scanTimedoutTCPSessions\n");
#endif

  for(idx=0; idx<HASHNAMESIZE; idx++) {
    if(tcpSession[idx] != NULL) {

      if(tcpSession[idx]->magic != MAGIC_NUMBER) {
	tcpSession[idx] = NULL;
	numTcpSessions--;
	printf("Magic assertion failed!\n");
	continue;
      }

#ifdef DEBUG
      printf("Session to scan (%d): %s:%d <--> %s:%d (%lu/%lu)\n", idx,
	     ipaddr2str(hash_hostTraffic[checkSessionIdx(tcpSession[idx]->initiatorIdx)]->hostIpAddress), tcpSession[idx]->sport,
	     ipaddr2str(hash_hostTraffic[checkSessionIdx(tcpSession[idx]->remotePeerIdx)]->hostIpAddress), tcpSession[idx]->dport,
	     tcpSession[idx]->lastSeen+TWO_MSL_TIMEOUT,
	     actTime);
#endif      
      if(
	 ((tcpSession[idx]->sessionState == STATE_TIMEOUT)
	  && ((tcpSession[idx]->lastSeen+TWO_MSL_TIMEOUT) < actTime))
	 || /* The branch below allows to flush sessions which have not been
	       terminated properly (we've received just one FIN (not two). It might be
	       that we've lost some packets (hopefully not). */
	 ((tcpSession[idx]->sessionState >= STATE_FIN1_ACK0)
	  && ((tcpSession[idx]->lastSeen+DOUBLE_TWO_MSL_TIMEOUT) < actTime))
	 /* The line below allows to avoid keeping very old sessions that
	    might be still open, but that are probably closed and we've
	    lost some packets */
	 || ((tcpSession[idx]->lastSeen+IDLE_HOST_PURGE_TIMEOUT) < actTime)
	 )
	{
	  IPSession *sessionToPurge = tcpSession[idx];

 	  tcpSession[idx] = NULL;
	  numTcpSessions--;

	  /* Session to purge */
	 
	  if(sessionToPurge->sport < sessionToPurge->dport) { /* Server -> Client */
	    if(getPortByNum(sessionToPurge->sport, IPPROTO_TCP) != NULL) {
	      updateHostSessionsList(sessionToPurge->initiatorIdx, sessionToPurge->sport, 
				     sessionToPurge->remotePeerIdx, sessionToPurge, 
				     IPPROTO_TCP, SERVER_TO_CLIENT, SERVER_ROLE);
	      updateHostSessionsList(sessionToPurge->remotePeerIdx, sessionToPurge->sport, 
				     sessionToPurge->initiatorIdx, sessionToPurge, 
				     IPPROTO_TCP, CLIENT_FROM_SERVER, CLIENT_ROLE);
	    }
	  } else { /* Client -> Server */
	    if(getPortByNum(sessionToPurge->dport, IPPROTO_TCP) != NULL) {
	      updateHostSessionsList(sessionToPurge->remotePeerIdx, sessionToPurge->dport, 
				     sessionToPurge->initiatorIdx, sessionToPurge, 
				     IPPROTO_TCP, SERVER_FROM_CLIENT, SERVER_ROLE);
	      updateHostSessionsList(sessionToPurge->initiatorIdx, sessionToPurge->dport, 
				     sessionToPurge->remotePeerIdx, sessionToPurge, 
				     IPPROTO_TCP, CLIENT_TO_SERVER, CLIENT_ROLE);
	    }
	  }

	  /* 
	   * Having updated the session information, 'theSession'
	   * can now be purged.
	   */
	  sessionToPurge->magic = 0;

	  notifyTCPSession(sessionToPurge);
	  free(sessionToPurge); /* No inner pointers to free */
	}
    }
  } /* end for */

}
  
/* ************************************ */

void updateUsedPorts(HostTraffic *srcHost, HostTraffic *dstHost,
		     u_short sport, u_short dport) {

  if(sport < TOP_ASSIGNED_IP_PORTS) {
    srcHost->usedServerPorts[sport]++;
    dstHost->usedClientPorts[sport]++; 
  }
  
  if(dport < TOP_ASSIGNED_IP_PORTS) {
    srcHost->usedClientPorts[dport]++; 
    dstHost->usedServerPorts[dport]++;
  }

  /*
    if(sport < TOP_ASSIGNED_IP_PORTS)
    srcHost->usedClientPorts[sport]++;      
  
    if(dport < TOP_ASSIGNED_IP_PORTS)
    dstHost->usedServerPorts[dport]++;
  */
}

/* ************************************ */

void handleSession(IPSession *sessions[],
		   u_short *numSessions,
		   u_int srcHostIdx,
		   u_short sport,
		   u_int dstHostIdx,
		   u_short dport,
		   u_int length,                   
		   struct tcphdr *tp,
		   u_int tcpDataLength)
{
  u_int idx, initialIdx;
  IPSession *theSession = NULL;
  short flowDirection;
  u_short sessionType, check;
  u_short sessSport, sessDport;
  HostTraffic *srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
  HostTraffic *dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];


  if(srcHost->broadcastHost || dstHost->broadcastHost)
    return;

#ifdef DEBUG
  printf("%8x/%8x %8x\n", 
	 srcHost->hostIpAddress.s_addr,
	 dstHost->hostIpAddress.s_addr,
	 localHostAddress.s_addr);
#endif

  if(tp == NULL)
    sessionType = IPPROTO_UDP;
  else {
    sessSport = ntohs(tp->th_sport);
    sessDport = ntohs(tp->th_dport);   
    sessionType = IPPROTO_TCP;
  }
  
  /* 
   * The hash key has to be calculated in a specular
   * way: its value has to be the same regardless 
   * of the flow direction.
   *
   * Patch on the line below courtesy of  
   * Paul Chapman <pchapman@fury.bio.dfo.ca>
   */
  initialIdx = idx = (u_int)(((u_long)srcHost+(u_long)dstHost+
			      (u_long)sport+(u_long)dport) % HASHNAMESIZE);

  if(sessionType == IPPROTO_TCP) {
    for(;;) {
      theSession = sessions[idx];
      if(theSession != NULL) {

	if((theSession->initiatorIdx == srcHostIdx) && (theSession->remotePeerIdx == dstHostIdx)
	   && (theSession->sport == sport) && (theSession->dport == dport)) {
	  flowDirection = CLIENT_TO_SERVER;
	  break;
	} else if((theSession->initiatorIdx == dstHostIdx) && (theSession->remotePeerIdx == srcHostIdx)
		  && (theSession->sport == dport) && (theSession->dport == sport)) {
	  flowDirection = SERVER_TO_CLIENT;
	  break;
	}
      } else if(theSession == NULL) { 
	/* New Session */

	if(tp->th_flags & TH_FIN) {
	  /* We've received a FIN for a session that
	     we've not seen (ntop started when the 
	     session completed).
	  */
#ifdef DEBUG
	  printf("Received FIN %u for unknown session\n",  ntohl(tp->th_seq)+tcpDataLength);
#endif
	  return; /* Nothing else to do */
	}

	if((*numSessions) > (HASHNAMESIZE/2)) {
	  /* The hash table is getting large: let's replace the oldest session
	     with this one we're allocating */
	  u_int usedIdx;
	
	  for(idx=0; idx<HASHNAMESIZE; idx++) {
	    if(sessions[idx] != NULL) {
	      if(theSession == NULL) {
		theSession = sessions[idx];
		usedIdx = idx;
	      }
	      else if(theSession->lastSeen > sessions[idx]->lastSeen) {
		theSession = sessions[idx];
		usedIdx = idx;
	      }
	    }
	  }

	  sessions[usedIdx] = NULL;
	} else { 
	  /* There's enough space left in the hashtable */
	  theSession = (IPSession*)malloc(sizeof(IPSession));
	  memset(theSession, 0, sizeof(IPSession));
	  theSession->magic = MAGIC_NUMBER;
	  (*numSessions)++;
	}

	updateUsedPorts(srcHost, dstHost, sport, dport);

	while(sessions[initialIdx] != NULL)
	  initialIdx = ((initialIdx+1) % HASHNAMESIZE);

	sessions[initialIdx] = theSession;

	theSession->initiatorIdx = checkSessionIdx(srcHostIdx);
	theSession->remotePeerIdx = checkSessionIdx(dstHostIdx);
	theSession->sport = sport;
	theSession->dport = dport;
	theSession->firstSeen = actTime;
	flowDirection = CLIENT_TO_SERVER;

#ifdef DEBUG
	if(logFlag) printSession(theSession, sessionType, 0);
#endif
	break;
      }

      idx = ((idx+1) % HASHNAMESIZE);
    }
  
    theSession->lastSeen = actTime;

    if(flowDirection == CLIENT_TO_SERVER)
      theSession->bytesSent += length;
    else
      theSession->bytesReceived += length;

    /*
     *
     * In this case the session is over hence the list of
     * sessions initiated/received by the hosts can be updated
     *
     */

    if(tp->th_flags & TH_FIN) {
      u_int32_t fin = ntohl(tp->th_seq)+tcpDataLength;

      if(sport < dport) /* Server -> Client */
	check = (fin != theSession->lastSCFin);
      else /* Client -> Server */
	check = (fin != theSession->lastCSFin);

      if(check) {
	/* This is not a duplicated (retransmitted) FIN */
	theSession->finId[theSession->numFin] = fin;
	theSession->numFin = (theSession->numFin+1)%MAX_NUM_FIN;;

	if(sport < dport) /* Server -> Client */
	  theSession->lastSCFin = fin;
	else /* Client -> Server */
	  theSession->lastCSFin = fin;
	switch(theSession->sessionState) {
	case STATE_BEGIN:
	  theSession->sessionState = STATE_FIN1_ACK0;
	  break;
	case STATE_FIN1_ACK0:
	  theSession->sessionState = STATE_FIN1_ACK1;
	  break;
	case STATE_FIN1_ACK1:
	  theSession->sessionState = STATE_FIN2_ACK1;
	  break;
#ifdef DEBUG
	default:
	  printf("ERROR: unable to handle received FIN (%u) !\n", fin);
#endif
	}
      } else {
#ifdef DEBUG
	printf("Rcvd Duplicated FIN %u\n", fin);
#endif
      }
    } else if((tp->th_flags & TH_ACK) && (theSession->numFin > 0)) { 
      u_int32_t ack = ntohl(tp->th_ack);
      int i;

      if(sport < dport) /* Server -> Client */
	check = (ack != theSession->lastSCAck);
      else /* Client -> Server */
	check = (ack != theSession->lastCSAck);

      if(check) {
	/* This is not a duplicated ACK */

	if(sport < dport) /* Server -> Client */
	  theSession->lastSCAck = ack;
	else /* Client -> Server */
	  theSession->lastCSAck = ack;
	
	for(i=0; i<theSession->numFin; i++) {
	  if((theSession->finId[i]+1) == ack) {
	    theSession->numFinAcked++;
	    theSession->finId[i] = 0;

	    switch(theSession->sessionState) {
	    case STATE_FIN1_ACK0:
	      theSession->sessionState = STATE_FIN1_ACK1;
	      break;
	    case STATE_FIN2_ACK0:
	      theSession->sessionState = STATE_FIN2_ACK1;
	      break;
	    case STATE_FIN2_ACK1:
	      theSession->sessionState = STATE_FIN2_ACK2;
	      break;
#ifdef DEBUG
	    default:
	      printf("ERROR: unable to handle received ACK (%u) !\n", ack);
#endif
	    }	    
	    break;
	  }
	}
      } else {
      }
    }

    theSession->lastFlags = tp->th_flags;

    if((theSession->sessionState == STATE_FIN2_ACK2)
       || ((tp->th_flags & TH_RST) != 0)) /* abortive release */ {
      theSession->sessionState = STATE_TIMEOUT;      
    }
  } else if(sessionType == IPPROTO_UDP) { 
    IPSession tmpSession;

    memset(&tmpSession, 0, sizeof(IPSession));

    updateUsedPorts(srcHost, dstHost, sport, dport);
    
    tmpSession.lastSeen = actTime;
    tmpSession.initiatorIdx = checkSessionIdx(srcHostIdx), 
      tmpSession.remotePeerIdx = checkSessionIdx(dstHostIdx);
    tmpSession.bytesSent = length, tmpSession.bytesReceived = 0;
    tmpSession.sport = sport, tmpSession.dport = dport;

#ifdef DEBUG
    if(logFlag) printSession(&tmpSession, sessionType, 0);
#endif

    if(getPortByNum(sport, sessionType) != NULL) {
      updateHostSessionsList(srcHostIdx, sport, dstHostIdx, &tmpSession, 
			     sessionType, SERVER_TO_CLIENT, SERVER_ROLE);
      tmpSession.bytesSent = 0, tmpSession.bytesReceived = length;
      updateHostSessionsList(dstHostIdx, sport, srcHostIdx, &tmpSession, 
			     sessionType, CLIENT_FROM_SERVER, CLIENT_ROLE);
    } else {
#if defined(MULTITHREADED) && defined(HAVE_LSOF)
      accessMutex(&lsofMutex);
#endif
#ifdef HAVE_LSOF
      updateLsof = 1; /* Force lsof update */
#endif
#if defined(MULTITHREADED) && defined(HAVE_LSOF)
      releaseMutex(&lsofMutex);
#endif
    }  

    if(getPortByNum(dport, sessionType) != NULL) {
      updateHostSessionsList(srcHostIdx, dport, dstHostIdx, &tmpSession, 
			     sessionType, CLIENT_TO_SERVER, CLIENT_ROLE);
      tmpSession.bytesSent = 0, tmpSession.bytesReceived = length;
      updateHostSessionsList(dstHostIdx, dport, srcHostIdx, &tmpSession, 
			     sessionType, SERVER_FROM_CLIENT, SERVER_ROLE);
    } else {
#if defined(MULTITHREADED) && defined(HAVE_LSOF)
      accessMutex(&lsofMutex);
#endif
#ifdef HAVE_LSOF
      updateLsof = 1; /* Force lsof update */
#endif
#if defined(MULTITHREADED) && defined(HAVE_LSOF)
      releaseMutex(&lsofMutex);
#endif
    }
  }  
}

/* ************************************ */

#ifdef HAVE_LSOF

void addLsofContactedPeers(ProcessInfo *process, u_int peerHostIdx) {
  int i;

  if((process == NULL)
     || (peerHostIdx == NO_PEER)
     || hash_hostTraffic[checkSessionIdx(peerHostIdx)]->broadcastHost)
    return;

  for(i=0; i<MAX_NUM_CONTACTED_PEERS; i++)
    if(process->contactedIpPeersIndexes[i] == peerHostIdx)
      return;

  process->contactedIpPeersIndexes[process->contactedIpPeersIdx] = peerHostIdx;
  process->contactedIpPeersIdx = (process->contactedIpPeersIdx+1) % MAX_NUM_CONTACTED_PEERS;
}


/* ************************************ */

void handleLsof(u_int srcHostIdx,
		u_short sport,
		u_int dstHostIdx,
		u_short dport,
		u_int length) {
  HostTraffic *srcHost, *dstHost;

  srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
  dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];

  if(srcHost->hostIpAddress.s_addr == localHostAddress.s_addr)
    if((sport < TOP_IP_PORT) && (localPorts[sport] != NULL)) {
      ProcessInfoList *scanner = localPorts[sport];
      
      while(scanner != NULL) {
	scanner->element->bytesSent += length;
	scanner->element->lastSeen   = actTime;
	addLsofContactedPeers(scanner->element, dstHostIdx);
	scanner = scanner->next;
      }
    }
  
  if(dstHost->hostIpAddress.s_addr == localHostAddress.s_addr)
    if((dport < TOP_IP_PORT) && (localPorts[dport] != NULL)) {
      ProcessInfoList *scanner = localPorts[dport];
      
      while(scanner != NULL) {
	  scanner->element->bytesReceived += length;
	  scanner->element->lastSeen   = actTime;
	  addLsofContactedPeers(scanner->element, srcHostIdx);
	  scanner = scanner->next;
	}
    }
}  

#endif

/* *********************************** */

void handleTCPSession(u_int srcHostIdx,
		      u_short sport,
		      u_int dstHostIdx,
		      u_short dport,
		      u_int length, 
		      struct tcphdr *tp,
		      u_int tcpDataLength) {

  if((tp->th_flags & TH_SYN) == 0) {
    /* When we receive SYN it means that the client is trying to
       open a session with the server hence the session is NOT YET
       open. That's why we don't count this as a session in this
       case.
    */
    handleSession(tcpSession, &numTcpSessions, 
		  srcHostIdx, sport, 
		  dstHostIdx, dport, 
		  length, tp, tcpDataLength);
  }

#ifdef HAVE_LSOF
  handleLsof(srcHostIdx, sport, dstHostIdx, dport, length);
#endif
}

/* ************************************ */

void handleUDPSession(u_int srcHostIdx,
		      u_short sport,
		      u_int dstHostIdx,
		      u_short dport,
		      u_int length) {

  handleSession(udpSession, &numUdpSessions, srcHostIdx, sport,
		dstHostIdx, dport, length, NULL, 0);

#ifdef HAVE_LSOF
  handleLsof(srcHostIdx, sport, dstHostIdx, dport, length);
#endif
}

/* ************************************ */

int handleIP(u_short port,   
	     u_int srcHostIdx,
	     u_int dstHostIdx,
	     u_int length) {
  
  int idx = mapGlobalToLocalIdx(port);
  HostTraffic *srcHost, *dstHost;

  if(idx == -1)
    return(-1); /* Unable to locate requested index */

  srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
  dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];

  if(idx != -1) {
    if(srcHost->subnetPseudoLocalHost == LOCAL_HOST) {
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
	if((srcHostIdx != broadcastEntryIdx) && (!srcHost->broadcastHost)) 
	  srcHost->protoIPTrafficInfos[idx].sentLocally += length;
	if((dstHostIdx != broadcastEntryIdx) && (!dstHost->broadcastHost))
	  dstHost->protoIPTrafficInfos[idx].receivedLocally += length;
	ipProtoStats[idx].local += length;
      } else {
	if((srcHostIdx != broadcastEntryIdx) && (!srcHost->broadcastHost))
	  srcHost->protoIPTrafficInfos[idx].sentRemotely += length;
	if((dstHostIdx != broadcastEntryIdx) && (!dstHost->broadcastHost)) 
	  dstHost->protoIPTrafficInfos[idx].receivedLocally += length;
	ipProtoStats[idx].local2remote += length;
      }
    } else {
      /* srcHost is remote */
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
	if((srcHostIdx != broadcastEntryIdx) && (!srcHost->broadcastHost))
	  srcHost->protoIPTrafficInfos[idx].sentLocally += length;
	if((dstHostIdx != broadcastEntryIdx) && (!dstHost->broadcastHost)) 
	  dstHost->protoIPTrafficInfos[idx].receivedFromRemote += length;
	ipProtoStats[idx].remote2local += length;
      } else {
	if((srcHostIdx != broadcastEntryIdx) && (!srcHost->broadcastHost))
	  srcHost->protoIPTrafficInfos[idx].sentRemotely += length;
	if((dstHostIdx != broadcastEntryIdx) && (!dstHost->broadcastHost))
	  dstHost->protoIPTrafficInfos[idx].receivedFromRemote += length;
	ipProtoStats[idx].remote += length;
      }
    }    
  }

  return(idx);
}

/* ************************************ */

void addContactedPeers(u_int senderIdx, u_int receiverIdx) {
  short i, found;
  HostTraffic *sender, *receiver;
  
  sender = hash_hostTraffic[checkSessionIdx(senderIdx)];
  receiver = hash_hostTraffic[checkSessionIdx(receiverIdx)];

  /* ******************************* */
  if(sender != NULL) {
    for(found=0, i=0; i<MAX_NUM_CONTACTED_PEERS; i++) 
      if(sender->contactedSentPeersIndexes[i] != NO_PEER) {
	if((sender->contactedSentPeersIndexes[i] == receiverIdx) 
	   || (receiver->broadcastHost 
	       && hash_hostTraffic[checkSessionIdx(sender->
						   contactedSentPeersIndexes[i])]->broadcastHost)) {
	  found = 1;
	  break;
	}
      }
    
    if(found == 0) {
      sender->contactedSentPeersIndexes[sender->contactedSentPeersIdx] = receiverIdx;
      sender->contactedSentPeersIdx = (sender->contactedSentPeersIdx+1) % MAX_NUM_CONTACTED_PEERS;
    }
  }


  /* ******************************* */
  if(receiver != NULL) {
    for(found=0, i=0; i<MAX_NUM_CONTACTED_PEERS; i++)
      if(receiver->contactedRcvdPeersIndexes[i] != NO_PEER) {
	if((receiver->contactedRcvdPeersIndexes[i] == senderIdx) 
	   || (sender->broadcastHost 
	       && hash_hostTraffic[checkSessionIdx(receiver->
						   contactedRcvdPeersIndexes[i])]->broadcastHost)) {
	  found = 1;
	  break;
	}
      }
    
    if(found == 0) {
      receiver->contactedRcvdPeersIndexes[receiver->contactedRcvdPeersIdx] = senderIdx;
      receiver->contactedRcvdPeersIdx = (receiver->contactedRcvdPeersIdx+1) % MAX_NUM_CONTACTED_PEERS;
    }
  }
}

/* ************************************ */

void addFragmentInfo(HostTraffic *srcHost, 
		     HostTraffic *dstHost,
		     u_short sport, 
		     u_short dport, 
		     u_int fragmentId) {
  IpFragment *fragment;
  u_int idx, startIdx, retry=0;

ADD_FRAGMENT:
  startIdx = idx = ((u_int)((unsigned long)srcHost+(unsigned long)dstHost)*fragmentId) % HASHNAMESIZE;

#ifdef DEBUG
  printf("Added fragmentId %u\n", fragmentId);
#endif

  while(1) {
    fragment = fragmentTable[idx];

    if(fragment == NULL) {
      fragment = (IpFragment*)malloc(sizeof(IpFragment));
      memset(fragment, 0, sizeof(IpFragment));
      fragment->src = srcHost;
      fragment->dest = dstHost;
      fragment->fragmentId = fragmentId;
      fragment->sport = sport;
      fragment->dport = dport;
      fragment->firstSeen = actTime;
      fragmentTable[idx] = fragment;
      return;
    }

    idx = (idx+1) % HASHNAMESIZE;

    if(startIdx == idx) {
      if(retry == 0) {
	purgeOldFragmentEntries();
	retry++;
	goto ADD_FRAGMENT;
      } else {
#ifdef DEBUG
	printf("ERROR: the fragment table is full!\n");
#endif
	break;
      }
    }
  }
}


/* ************************************ */

void getFragmentInfo(HostTraffic *srcHost, 
		     HostTraffic *dstHost,
		     u_short *sport, 
		     u_short *dport, 
		     u_int fragmentId,
		     u_short fragmentAction) {

  IpFragment *fragment;
  u_int idx, startIdx;

  startIdx = idx = ((u_int)((unsigned long)srcHost+(unsigned long)dstHost)*fragmentId) % HASHNAMESIZE;

  while(1) {
    fragment = fragmentTable[idx];

    if((fragment != NULL) 
       && (fragment->src == srcHost)
       && (fragment->dest == dstHost)
       && (fragment->fragmentId == fragmentId)) {
      (*sport) = fragment->sport;
      (*dport) = fragment->dport;

#ifdef DEBUG
      printf("Found fragmentId %u\n", fragmentId);
#endif
      if(fragmentAction == DELETE_FRAGMENT) {
	free(fragment);
	fragmentTable[idx] = NULL;
#ifdef DEBUG
	printf("Removed fragmentId %u entry\n", fragmentId);
#endif
      }

      break;
    }

    idx = (idx+1) % HASHNAMESIZE;

    if(startIdx == idx) {
#ifdef DEBUG
      printf("ERROR: unable to find the expected fragment %u!\n", 
	     fragmentId);
#endif
      break;
    }
  }

}

/* ************************************ */

void purgeOldFragmentEntries() {
  int i;

  for(i=0; i<HASHNAMESIZE; i++) {
    if((fragmentTable[i] != NULL)
       && ((fragmentTable[i]->firstSeen+DOUBLE_TWO_MSL_TIMEOUT) > actTime)) {
      free(fragmentTable[i]);
      fragmentTable[i] = NULL;
    }
  }
}

/* ************************************ */

void processDNSPacket(const u_char *bp, u_int length, u_int hlen) {
  DNSHostInfo hostPtr;
  struct in_addr hostIpAddress;
#ifdef HAVE_GDBM_H
  datum key_data, data_data;
#endif

  memset(&hostPtr, 0, sizeof(DNSHostInfo));

  handleDNSpacket(bp, hlen+sizeof(struct udphdr),
		  &hostPtr, (short)(length-(hlen+sizeof(struct udphdr))));
  
  if((hostPtr.name[0] != '\0') && (hostPtr.addrList[0] != 0)) {
    int i;
	
    hostIpAddress.s_addr = ntohl(hostPtr.addrList[0]);

#ifdef MULTITHREADED
    accessMutex(&gdbmMutex);
#endif 
  
    for(i=0; i<MAXALIASES; i++)
      if(hostPtr.aliases[i][0] != '\0') {
#ifdef DNS_SNIFF_DEBUG 
	printf("%s alias of %s\n",
	       hostPtr.aliases[i], hostPtr.name);
#endif
	key_data.dptr = intoa(hostIpAddress);
	key_data.dsize = strlen(key_data.dptr)+1;
	data_data.dptr = hostPtr.aliases[0]; /* Let's take the first one */
	data_data.dsize = strlen(data_data.dptr)+1;
	gdbm_store(gdbm_file, key_data, data_data, GDBM_REPLACE);
      }

    data_data.dptr = hostPtr.name;
    data_data.dsize = strlen(data_data.dptr)+1;

    for(i=0; i<MAXADDRS; i++) 
      if(hostPtr.addrList[i] != 0){	  
	hostIpAddress.s_addr = ntohl(hostPtr.addrList[i]);
#ifdef DNS_SNIFF_DEBUG 
	printf("%s <-> %s\n", 
	       hostPtr.name, intoa(hostIpAddress));
#endif
	key_data.dptr = intoa(hostIpAddress);
	key_data.dsize = strlen(key_data.dptr)+1;
	gdbm_store(gdbm_file, key_data, data_data, GDBM_REPLACE);
      }

#ifdef MULTITHREADED
    releaseMutex(&gdbmMutex);
#endif 
  }
}

/* ************************************ */

void processIpPkt(const u_char *bp, u_int length,
		  u_char *ether_src, u_char *ether_dst)
{
  u_short sport, dport;
  struct ip ip;
  struct tcphdr tp;
  struct udphdr up;
  u_int hlen, tcpDataLength, off;
  char *proto;
  u_int srcHostIdx, dstHostIdx;
  u_char saveflags;
  u_char flags = '.';
  HostTraffic *srcHost=NULL, *dstHost=NULL;
  u_char etherAddrSrc[ETHERNET_ADDRESS_LEN+1], etherAddrDst[ETHERNET_ADDRESS_LEN+1];


  /* Need to copy this over in case bp isn't properly aligned.
   * This occurs on SunOS 4.x at least.                
   * Paul D. Smith <psmith@baynetworks.com>
   */
  memcpy(&ip, bp, sizeof(struct ip));
  hlen = (u_int)ip.ip_hl * 4;

  if((ether_src == NULL) && (ether_dst == NULL)) {
    /* Ethernet-less protocols (e.g. PPP/RAW IP) */    

    memcpy(etherAddrSrc, &(ip.ip_src.s_addr), sizeof(ip.ip_src.s_addr));
    etherAddrSrc[ETHERNET_ADDRESS_LEN] = '\0';
    ether_src = etherAddrSrc;

    memcpy(etherAddrDst, &(ip.ip_dst.s_addr), sizeof(ip.ip_dst.s_addr));
    etherAddrDst[ETHERNET_ADDRESS_LEN] = '\0';
    ether_dst = etherAddrDst;
  }

  NTOHL(ip.ip_src.s_addr); srcHostIdx = getHostInfo(&ip.ip_src, ether_src);
  NTOHL(ip.ip_dst.s_addr); dstHostIdx = getHostInfo(&ip.ip_dst, ether_dst);  

  srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
  dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];

  updatePacketCount(srcHostIdx, dstHostIdx, length);
  updateTrafficMatrix(srcHostIdx, dstHostIdx, length);

  if(srcHost->subnetPseudoLocalHost == LOCAL_HOST) {
    if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
      srcHost->bytesSentLocally += length;
      dstHost->bytesReceivedLocally += length;
    } else {
      srcHost->bytesSentRemotely += length;
      dstHost->bytesReceivedLocally += length;
    }
  } else {
    /* srcHost is remote */
    if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
      srcHost->bytesSentLocally += length;
      dstHost->bytesReceivedFromRemote += length;
    } else {
      srcHost->bytesSentRemotely += length;
      dstHost->bytesReceivedFromRemote += length;
    }
  }    

  ipBytes += length;
  off = ntohs(ip.ip_off);

  switch(ip.ip_p) {
  case IPPROTO_TCP:
    proto = "TCP";
    tcpBytes += length;
    memcpy(&tp, bp+hlen, sizeof(struct tcphdr));

    tcpDataLength = ntohs(ip.ip_len) - hlen - (tp.th_off * 4);

    sport = ntohs(tp.th_sport);
    dport = ntohs(tp.th_dport);
       
    if (off & 0x3fff) { /* Handle fragmented packets */
      u_int fragmentOffset = (off & 0x1fff) * 8;

      if(fragmentOffset == 0) /* First fragment */
	addFragmentInfo(srcHost, dstHost, sport, dport, ntohs(ip.ip_id));
      else {
	if(!(off & IP_MF)) {
	  /* This is the last fragment */
	  getFragmentInfo(srcHost, dstHost, &sport, 
			  &dport, ntohs(ip.ip_id), DELETE_FRAGMENT);
	} else {
	  /* Intermediate fragment */
	  getFragmentInfo(srcHost, dstHost, &sport, 
			  &dport, ntohs(ip.ip_id), KEEP_FRAGMENT);
	}
      }

      tcpDataLength = ntohs(ip.ip_len)-hlen;
    }

    sport = ntohs(tp.th_sport);
    dport = ntohs(tp.th_dport);
    if ((saveflags = tp.th_flags) & (TH_SYN|TH_FIN|TH_RST|TH_PUSH)) {
      if (saveflags & TH_SYN)
	flags = 'S';
      if (saveflags & TH_FIN)
	flags = 'F';
      if (saveflags & TH_RST)
	flags = 'R';
      if (saveflags & TH_PUSH)
	flags = 'P';
    } 

    if(srcHost->subnetPseudoLocalHost == LOCAL_HOST) {
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
	srcHost->tcpSentLocally += length;
	dstHost->tcpReceivedLocally += length;
	tcpGlobalTrafficStats.local += length;
      } else {
	srcHost->tcpSentRemotely += length;
	dstHost->tcpReceivedLocally += length;
	tcpGlobalTrafficStats.local2remote += length;
      }
    } else {
      /* srcHost is remote */
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
	srcHost->tcpSentLocally += length;
	dstHost->tcpReceivedFromRemote += length;
	tcpGlobalTrafficStats.remote2local += length;
      } else {
	srcHost->tcpSentRemotely += length;
	dstHost->tcpReceivedFromRemote += length;
	tcpGlobalTrafficStats.remote += length;
      }
    }    
    
    if(handleIP(dport, srcHostIdx, dstHostIdx, length) == -1)
      handleIP(sport, srcHostIdx, dstHostIdx, length);

    handleTCPSession(srcHostIdx, sport, dstHostIdx, 
		     dport, length, &tp, tcpDataLength);
    break;

  case IPPROTO_UDP:
    proto = "UDP";
    udpBytes += length;
    memcpy(&up, bp+hlen, sizeof(struct udphdr));
    /* print TCP packet useful for debugging */

#ifdef SLACKWARE
    sport = ntohs(up.source);
    dport = ntohs(up.dest);
#else
    sport = ntohs(up.uh_sport);
    dport = ntohs(up.uh_dport);
#endif

    if(sport == 53 /* domain */) {
#ifdef DNS_SNIFF_DEBUG 
      /*
      printf("(%s -> %s) ", 
	     srcHost->hostSymIpAddress, 
	     dstHost->hostSymIpAddress);
*/
#endif

      processDNSPacket(bp, length, hlen);
    }


    if (off & 0x3fff) { /* Handle fragmented packets */
      u_int fragmentOffset = (off & 0x1fff) * 8;

      if(fragmentOffset == 0) /* First fragment */
	addFragmentInfo(srcHost, dstHost, sport, dport, ntohs(ip.ip_id));
      else {
	if(!(off & IP_MF)) {
	  /* This is the last fragment */
	  getFragmentInfo(srcHost, dstHost, &sport, 
			  &dport, ntohs(ip.ip_id), DELETE_FRAGMENT);
	} else {
	  /* Intermediate fragment */
	  getFragmentInfo(srcHost, dstHost, &sport, 
			  &dport, ntohs(ip.ip_id), KEEP_FRAGMENT);
	}
      }

      tcpDataLength = ntohs(ip.ip_len)-hlen;
    }

    if(srcHost->subnetPseudoLocalHost == LOCAL_HOST) {
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
	srcHost->udpSentLocally += length;
	dstHost->udpReceivedLocally += length;
	udpGlobalTrafficStats.local += length;
      } else {
	srcHost->udpSentRemotely += length;
	dstHost->udpReceivedLocally += length;
	udpGlobalTrafficStats.local2remote += length;
      }
    } else {
      /* srcHost is remote */
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) {
	srcHost->udpSentLocally += length;
	dstHost->udpReceivedFromRemote += length;
	udpGlobalTrafficStats.remote2local += length;
      } else {
	srcHost->udpSentRemotely += length;
	dstHost->udpReceivedFromRemote += length;
	udpGlobalTrafficStats.remote += length;
      }
    }        

    if(handleIP(dport, srcHostIdx, dstHostIdx, length) == -1)
      handleIP(sport, srcHostIdx, dstHostIdx, length);

    handleUDPSession(srcHostIdx, sport, dstHostIdx, dport, length);
    break;

  case IPPROTO_ICMP:
    proto = "ICMP";
    icmpBytes += length;
    srcHost->icmpSent += length;
    dstHost->icmpReceived += length;

    if(srcHost->subnetPseudoLocalHost == LOCAL_HOST) 
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) 
	icmpGlobalTrafficStats.local += length;
      else 
	icmpGlobalTrafficStats.local2remote += length;    
    else /* srcHost is remote */
      if(dstHost->subnetPseudoLocalHost == LOCAL_HOST) 
	icmpGlobalTrafficStats.remote2local += length;
       else 
	icmpGlobalTrafficStats.remote += length;
     
    break;

  case IPPROTO_OSPF:
    proto = "OSPF";
    ospfBytes += length;
    srcHost->ospfSent += length;
    dstHost->ospfReceived += length;
    break;

  case IPPROTO_IGMP:
    proto = "IGMP";
    igmpBytes += length;
    srcHost->igmpSent += length;
    dstHost->igmpReceived += length;
    break;

  default:
    proto = "IP (Other)";
    otherIpBytes += length;
    sport = dport = -1;
    srcHost->otherSent += length;
    dstHost->otherReceived += length;
    break;
  }
}

/* ************************************ */

void updatePacketCount(u_int srcHostIdx, u_int dstHostIdx, 
		       TrafficCounter length) {
  
  HostTraffic *srcHost, *dstHost;
  
  if(length > 1514) { /* Sanity check */
    printf("Wrong packet length (%u) !\n", (unsigned long)length);
    length = 1514;
  }

  srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
  dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];

  if((srcHost == NULL) || (dstHost == NULL))
    return;
  
  srcHost->pktSent++;

  if((dstHostIdx == broadcastEntryIdx) || dstHost->broadcastHost) {
    srcHost->pktBroadcastSent++;
    srcHost->bytesBroadcastSent += length;
    broadcastPkts++;
  } else if(isMulticastAddress(&(dstHost->hostIpAddress))) {
#ifdef DEBUG
    printf("%s -> %s\n", srcHost->hostSymIpAddress, dstHost->hostSymIpAddress);
#endif
    srcHost->pktMulticastSent++;
    srcHost->bytesMulticastSent += length;
    multicastPkts++;
  }

  srcHost->bytesSent += length;
  if(dstHost != NULL) dstHost->bytesReceived += length;

  dstHost->pktReceived++;

  if(length < 64) rcvdPktStats.upTo64++;
  else if(length < 128) rcvdPktStats.upTo128++;
  else if(length < 256) rcvdPktStats.upTo256++;
  else if(length < 512) rcvdPktStats.upTo512++;
  else if(length < 1024) rcvdPktStats.upTo1024++;
  else if(length < 1518) rcvdPktStats.upTo1518++;
  else rcvdPktStats.above1518++;

  if((rcvdPktStats.shortest == 0)
     || (rcvdPktStats.shortest > length))
    rcvdPktStats.shortest = length;

  if(rcvdPktStats.longest < length)
    rcvdPktStats.longest = length;
  
  if((dstHost != NULL) /*&& (!dstHost->broadcastHost)*/)
    addContactedPeers(srcHostIdx, dstHostIdx);
}

/* ************************************ */

void updateTrafficMatrix(u_int srcHostIdx, u_int dstHostIdx,
			 TrafficCounter length) {
  unsigned long a, b;
  HostTraffic *srcHost, *dstHost;

  srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
  dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];

  a = (unsigned long)(srcHost->hostIpAddress.s_addr);
  b = (unsigned long)(dstHost->hostIpAddress.s_addr);

  if(((a & netmask) == localnet) 
     && ((b & netmask) == localnet)) {    
    a &= 0x000000FF, b &= 0x000000FF;
    
    ipTrafficMatrixHosts[a] = srcHost, ipTrafficMatrixHosts[b] = dstHost;
    ipTrafficMatrixHosts[a]->promiscuousMode = ipTrafficMatrixPromiscHosts[a],
    ipTrafficMatrixHosts[b]->promiscuousMode = ipTrafficMatrixPromiscHosts[b];
    ipTrafficMatrix[a][b].bytesSent += length, ipTrafficMatrix[b][a].bytesReceived += length;
  }
}

/* ************************************ */

#ifdef MULTITHREADED

void queuePacket(u_char *unused, 
		 const struct pcap_pkthdr *h,
		 const u_char *p) {

  /****************************
    - If the queue is full then wait until a slot is freed
    - If the queue is getting full then periodically wait
      until a slot is freed
  *****************************/

#ifdef WIN32_DEMO
	static u_short numQueuedPackets=0;

	if(numQueuedPackets++ >= MAX_NUM_PACKETS)
		return;
#endif

  if(packetQueueLen >= PACKET_QUEUE_LENGTH) {
#ifdef DEBUG
    printf("Dropping packet!!! [packet queue=%d/max=%d]\n", 
	   packetQueueLen, maxPacketQueueLen);
#endif
    droppedPackets++;

#ifdef HAVE_SCHED_H
        sched_yield(); /* Allow other threads (dequeue) to run */
#endif
		sleep(1);  
  } else {  
#ifdef DEBUG
    printf("About to queue packet... \n");
#endif
    accessMutex(&packetQueueMutex);
    memcpy(&packetQueue[packetQueueHead].h, h, sizeof(struct pcap_pkthdr));
    memset(packetQueue[packetQueueHead].p, 0, DEFAULT_SNAPLEN);
    memcpy(packetQueue[packetQueueHead].p, p, h->caplen);
    packetQueueHead = (packetQueueHead+1) % PACKET_QUEUE_LENGTH;
    packetQueueLen++;
    if(packetQueueLen > maxPacketQueueLen)
      maxPacketQueueLen = packetQueueLen;
    releaseMutex(&packetQueueMutex);
#ifdef DEBUG
    printf("Queued packet... [packet queue=%d/max=%d]\n", 
	   packetQueueLen, maxPacketQueueLen);
#endif

#ifdef DEBUG_THREADS
    printf("+ [packet queue=%d/max=%d]\n", packetQueueLen, maxPacketQueueLen);
#endif
  }

#ifdef USE_SEMAPHORES
    incrementSem(&queueSem);
#else
    signalCondvar(&queueCondvar);
#endif
#ifdef HAVE_SCHED_H
    sched_yield(); /* Allow other threads (dequeue) to run */
#endif
}

/* ************************************ */

void* dequeuePacket(void* notUsed) {
  PacketInformation pktInfo;

  while(1) {
#ifdef DEBUG
    printf("Waiting for packet...\n");
#endif

/* LUCA !!!!!!!!!!! */
/* #ifdef WIN32 */
	while(packetQueueLen == 0) 
/*  #endif */
	{
#ifdef USE_SEMAPHORES
	   waitSem(&queueSem);
#else
	   waitCondvar(&queueCondvar);
#endif
	}

#ifdef DEBUG
    printf("Got packet...\n");
#endif
    accessMutex(&packetQueueMutex);
    memcpy(&pktInfo.h, &packetQueue[packetQueueTail].h, 
	   sizeof(struct pcap_pkthdr));
    memcpy(pktInfo.p, packetQueue[packetQueueTail].p, DEFAULT_SNAPLEN);
    packetQueueTail = (packetQueueTail+1) % PACKET_QUEUE_LENGTH;
    packetQueueLen--;
    releaseMutex(&packetQueueMutex);
#ifdef DEBUG_THREADS
    printf("- [packet queue=%d/max=%d]\n", packetQueueLen, maxPacketQueueLen);
#endif

#ifdef DEBUG
    printf("Processing packet... [packet queue=%d/max=%d]\n", 
	   packetQueueLen, maxPacketQueueLen);
#endif

   /* Sanity check */
   if(pktInfo.h.caplen > 1514) pktInfo.h.caplen = 1514;
   if(pktInfo.h.len > 1514) pktInfo.h.len = 1514;
 
    pbuf_process(NULL, &pktInfo.h, pktInfo.p);
  }

  return(NULL); /* NOTREACHED */
}

#endif /* MULTITHREADED */


/* ************************************ */


void flowsProcess(const struct pcap_pkthdr *h, const u_char *p) {
  FlowFilterList *list = flowsList;
  
  while(list != NULL) {
    if((list->filterCode != NULL)
       && (bpf_filter(list->filterCode, (u_char*)p, h->len, h->caplen))) {
      list->bytes += h->len;
      list->packets++;
      if(list->pluginPtr != NULL) {
	void(*pluginFunc)(const struct pcap_pkthdr *h, const u_char *p);
	
	pluginFunc = (void(*)(const struct pcap_pkthdr*, 
			      const u_char*))list->pluginPtr->pluginFunc;
	pluginFunc(h, p);
      }	
    }
    
    list = list->next;
  }
}

/* ************************************ */

/*
 * This is the top level routine of the printer.  'p' is the points
 * to the ether header of the packet, 'tvp' is the timestamp,
 * 'length' is the length of the packet off the wire, and 'caplen'
 * is the number of bytes actually captured.
 */
void pbuf_process(u_char *unused, 
		  const struct pcap_pkthdr *h, 
		  const u_char *p)
{
  struct ether_header *ep, ehdr;
  struct tokenRing_header *trp;
  struct fddi_header *fddip;
  u_int caplen = h->caplen;
  u_int headerDisplacement = 0, length = h->len;
  const u_char *orig_p = p;
  u_char *ether_src, *ether_dst;
  unsigned short eth_type=0;
  /* Token-Ring Strings */
  struct tokenRing_llc *trllc;

  /* static long numPkt=0; printf("%ld\n", numPkt++); */


#ifdef MULTITHREADED
  accessMutex(&hostsHashMutex);
#endif

  /*
    Let's check whether it's time to free up
    some space before to continue....
  */
  if(numHostTrafficEntries > HASHNAMETHRESHOLDSIZE)
    purgeIdleHosts(0 /* Delete only idle hosts */);

  memcpy(&lastPktTime, &h->ts, sizeof(lastPktTime));

  /* ethernet assumed */    
  if (caplen >= hlen) {
    HostTraffic *srcHost=NULL, *dstHost=NULL;
    u_int srcHostIdx, dstHostIdx;
    
    ep = (struct ether_header *)p;
 
    switch(datalink) {
    case DLT_FDDI: 
      fddip = (struct fddi_header *)p;
      length -= FDDI_HDRLEN;
      p += FDDI_HDRLEN;
      caplen -= FDDI_HDRLEN;

      extract_fddi_addrs(fddip, (char *)ESRC(&ehdr), (char *)EDST(&ehdr));
      ether_src = (u_char*)ESRC(&ehdr), ether_dst = (u_char*)EDST(&ehdr);

      if ((fddip->fc & FDDIFC_CLFF) == FDDIFC_LLC_ASYNC) {
	struct llc llc;
	
	memcpy((char *)&llc, (char *)p, min(caplen, sizeof(llc)));
	if (llc.ssap == LLCSAP_SNAP && llc.dsap == LLCSAP_SNAP
	    && llc.llcui == LLC_UI) {
	  if (caplen >= sizeof(llc)) {
	    caplen -= sizeof(llc);
	    length -= sizeof(llc);
	    p += sizeof(llc);
	    
	    if(EXTRACT_16BITS(&llc.ethertype[0]) == ETHERTYPE_IP) {
	      /* encapsulated IP packet */
	      processIpPkt(p, length, ether_src, ether_dst); 
	      return;
	    }
	  }
	}
      }
      break;

    case DLT_NULL: /* loopaback interface */
      /* 
	 Support for ethernet headerless interfaces (e.g. lo0)
	 Courtesy of Martin Kammerhofer <dada@sbox.tu-graz.ac.at> 
      */

      length -= NULL_HDRLEN; /* don't count nullhdr */

      /* All this crap is due to the old little/big endian story... */
      if((p[0] == 0) && (p[1] == 0) && (p[2] == 8) && (p[3] == 0)) {
	eth_type = ETHERTYPE_IP;
	ether_src = ether_dst = dummyEthAddress;
      }
      break;
    case DLT_PPP:
      headerDisplacement = PPP_HDRLEN;
      /*
	PPP is like RAW IP. The only difference is that PPP
	has a header that's not present in RAW IP.
	
	IMPORTANT: DO NOT PUT A break BELOW this comment
      */
    case DLT_RAW: /* RAW IP (no ethernet header) */
      length -= headerDisplacement; /* don't count PPP header */
      processIpPkt(p+headerDisplacement, length, NULL, NULL);
      break;

    case DLT_IEEE802: /* Token Ring */ 
      trp = (struct tokenRing_header*)p;
      ether_src = (u_char*)trp->trn_shost, ether_dst = (u_char*)trp->trn_dhost;

      hlen = sizeof(struct tokenRing_header) - 18;
      
      if (trp->trn_shost[0] & TR_RII) /* Source Routed Packet */
	hlen += ((ntohs(trp->trn_rcf) & TR_RCF_LEN_MASK) >> 8);
      
      length -= hlen, caplen -= hlen;
      
      p += hlen;
      trllc = (struct tokenRing_llc *)p;

      if (trllc->dsap == 0xAA && trllc->ssap == 0xAA)
	hlen = sizeof(struct tokenRing_llc);
      else
	hlen = sizeof(struct tokenRing_llc) - 5;
            
      length -= hlen, caplen -= hlen;
      
      p += hlen;	
	
      if (hlen == sizeof(struct tokenRing_llc))
	eth_type = ntohs(trllc->ethType);
      else
	eth_type = 0;
      break;
    default:
      eth_type = ntohs(ep->ether_type);      
      ether_src = ESRC(ep), ether_dst = EDST(ep);
    } /* switch(datalink) */

    if((datalink != DLT_PPP) && (datalink != DLT_RAW)) {
      if((datalink == DLT_IEEE802) && (eth_type < ETHERMTU)) {
	trp = (struct tokenRing_header*)orig_p;
	ether_src = (u_char*)trp->trn_shost, ether_dst = (u_char*)trp->trn_dhost;
	srcHostIdx = getHostInfo(NULL, ether_src);
	dstHostIdx = getHostInfo(NULL, ether_dst);
	srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
	dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];
	srcHost->otherSent += length;
	dstHost->otherReceived += length;	
	updatePacketCount(srcHostIdx, dstHostIdx, length);
      } else
	if((datalink != DLT_IEEE802) && (eth_type <= ETHERMTU) && (length > 3)) {
	  /* The code below has been taken from tcpdump */
	  u_char *p1, sap_type;
	  struct llc llcHeader;
      
	  srcHostIdx = getHostInfo(NULL, ether_src);
	  dstHostIdx = getHostInfo(NULL, ether_dst);
	  srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
	  dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];

	  p1 = (u_char*)(p+hlen);

	  /* Watch out for possible alignment problems */
	  memcpy(&llcHeader, (char*)p1, min(length, sizeof(llcHeader)));

	  sap_type = llcHeader.ssap & ~LLC_GSAP;
	  llcsap_string(sap_type);
	  
	  if ((llcHeader.ssap == LLCSAP_GLOBAL && llcHeader.dsap == LLCSAP_GLOBAL)
	      || (sap_type == 0xe0)) {
	    /* IPX */
	    srcHost->ipxSent += length;
	    dstHost->ipxReceived += length;		
	    ipxBytes += length;
	  } else if (llcHeader.ssap == LLCSAP_NETBIOS 
		     && llcHeader.dsap == LLCSAP_NETBIOS) {
	    /* Netbios */
	    srcHost->netbiosSent += length;
	    dstHost->netbiosReceived += length;	
	    netbiosBytes += length;
	  } else if ((sap_type == 0xf0) || (sap_type == 0xb4)
		     || (sap_type == 0xc4) || (sap_type == 0xf8)) {
	    /* DLC (protocol used for printers) */
	    srcHost->dlcSent += length;
	    dstHost->dlcReceived += length;	
	    dlcBytes += length;	
	  } else if (sap_type == 0xAA) {  /* Appletalk */
	    srcHost->appletalkSent += length;
	    dstHost->appletalkReceived += length;
	    atalkBytes += length;
	  } else if ((sap_type == 0x06) 
		     || (sap_type == 0xFE) 
		     || (sap_type == 0xFC)) {  /* OSI */
	    srcHost->osiSent += length;
	    dstHost->osiReceived += length;
	    osiBytes += length;
	  } else {
	    /* Unknown */

#ifdef PRINT_UNKNOWN_PACKETS
	    printf("[%u] [%x] %s %s > %s\n", (u_short)sap_type,(u_short)sap_type,
		   etheraddr_string(ether_src),
		   llcsap_string(llcHeader.ssap & ~LLC_GSAP),
		   etheraddr_string(ether_dst));
#endif

	    srcHost->otherSent += length;
	    dstHost->otherReceived += length;	
	    otherBytes += length;
	  }

	  updatePacketCount(srcHostIdx, dstHostIdx, length);
	} else if(eth_type == ETHERTYPE_IP) {
	  if((datalink == DLT_IEEE802) && (eth_type > ETHERMTU))
	    processIpPkt(p, length, ether_src, ether_dst);
	  else
	    processIpPkt(p+hlen, length, ether_src, ether_dst);
	} else {
	  /* Non IP */
	  srcHostIdx = getHostInfo(NULL, ether_src);
	  dstHostIdx = getHostInfo(NULL, ether_dst);
    
	  srcHost = hash_hostTraffic[checkSessionIdx(srcHostIdx)];
	  dstHost = hash_hostTraffic[checkSessionIdx(dstHostIdx)];

	  if(length > hlen)
	    length -= hlen;
	  else
	    length = 0;
      
	  switch(eth_type) 
	    {
	    case ETHERTYPE_ARP: /* ARP - Address resolution Protocol */
	    case ETHERTYPE_REVARP: /* Reverse ARP */
	      srcHost->arp_rarpSent += length;
	      dstHost->arp_rarpReceived += length;
	      arpRarpBytes += length;
	      break;	  
	    case ETHERTYPE_DN: /* Decnet */
	      srcHost->decnetSent += length;
	      dstHost->decnetReceived += length;
	      decnetBytes += length;
	      break;	  
	    case ETHERTYPE_ATALK: /* AppleTalk */
	    case ETHERTYPE_AARP:
	      srcHost->appletalkSent += length;
	      dstHost->appletalkReceived += length;
	      atalkBytes += length;
	      break;
	    default:
	      srcHost->otherSent += length;
	      dstHost->otherReceived += length;
	      otherBytes += length;
	      break;
	    }

	  updatePacketCount(srcHostIdx, dstHostIdx, length);
	}
    }
  }

  ethernetPkts++;
  ethernetBytes += h->len;

#ifdef MULTITHREADED
  if((actTime-lastThptUpdate) > (refreshRate*CLOCKS_PER_SEC)) {
    if(!webMode) 
      printHostsTraffic(0, 0);
  }
#endif

  if(flowsList != NULL) /* Handle flows last */
    flowsProcess(h, p);

#ifdef MULTITHREADED
  releaseMutex(&hostsHashMutex);
#endif

}

/* ************************************ */

void updateOSName(HostTraffic *el) {
#ifdef HAVE_GDBM_H
  datum key_data, data_data;
#endif

  if(el->osName == NULL) {
    char *theName = NULL, tmpBuf[64];

    if(el->hostNumIpAddress[0] == '\0') {
      el->osName = strdup("");
      return;
    }

#ifdef DEBUG
    printf("updateOSName(%s)\n", el->hostNumIpAddress);
#endif

#ifdef HAVE_GDBM_H
    sprintf(tmpBuf, "@%s", el->hostNumIpAddress);
    key_data.dptr = tmpBuf;
    key_data.dsize = strlen(tmpBuf)+1;
    
#ifdef MULTITHREADED
    accessMutex(&gdbmMutex);
#endif 

    data_data = gdbm_fetch(gdbm_file, key_data);

#ifdef MULTITHREADED
    releaseMutex(&gdbmMutex);
#endif 

    if(data_data.dptr != NULL) {
      strcpy(tmpBuf, data_data.dptr);
      free(data_data.dptr);
      theName = tmpBuf;      
    }
#endif /* HAVE_GDBM_H */

    if(theName == NULL)
      theName = getHostOS(el->hostNumIpAddress, -1);
    
    if(theName == NULL)
      el->osName = strdup("");
    else {
      el->osName = strdup(theName);

      updateDBOSname(el);

#ifdef HAVE_GDBM_H
      sprintf(tmpBuf, "@%s", el->hostNumIpAddress);
      key_data.dptr = tmpBuf;
      key_data.dsize = strlen(tmpBuf)+1;
      data_data.dptr = el->osName;
      data_data.dsize = strlen(el->osName)+1;
      
      if(gdbm_store(gdbm_file, key_data, data_data, GDBM_REPLACE) != 0)
	printf("Error while adding osName for '%s'\n.\n", el->hostNumIpAddress);
      else {
#ifdef GDBM_DEBUG
	printf("Added data: %s [%s]\n", tmpBuf, el->osName);
#endif	
      }
#endif /* HAVE_GDBM_H */
    }
  }
}

/* *********************************** */

void updateDbHostsTraffic() {
  u_int i;
  HostTraffic *el;

#ifdef DEBUG
  printf("updateDbHostsTraffic()\n");
#endif

  for(i=0; i<HASHNAMESIZE; i++) {
    el = hash_hostTraffic[i]; /* (**) */

    if((el != NULL) 
       && (!el->broadcastHost)
       && (el->nextDBupdate < actTime)) {

      el->instanceInUse++;

      if(el->nextDBupdate == 0)
	notifyHostCreation(el);
      else if(el->nextDBupdate < actTime) {	
	updateHostTraffic(el);
	if(el->osName == NULL) {
	  updateOSName(el);
	}
      }

      el->nextDBupdate = actTime + DB_TIMEOUT_REFRESH_TIME;
      el->instanceInUse--;
    }
  }
}




