# arch-tag: 7eae5a80-d62f-49b4-b61e-b20f460f5fe8
# Copyright (C) 2003--2005 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Internal module providing log-message handling.

This module implements some of public interface for the
pybaz_ package. But for convenience reasons the author prefers
to store this code in a file separate from ``__init__.py``.

.. _pybaz: pybaz-module.html

This module is strictly internal and should never be used.
"""

import email
import email.Parser
import email.Generator

__all__ = [
    'LogMessage',
    ]

class LogMessage(object):

    """Log message for use with commit, import or tag operations.

    This is the write-enabled counterpart of Patchlog. When creating a new
    revision with import, commit or tag, a log message file can be used to
    specify a long description and custom headers.

    Commit and import can use the default log file of the source tree, with a
    special name. You can create the LogMessage object associated to the
    default log file with the WorkingTree.log_message method.

    For integration with external tools, it is useful to be able to parse an
    existing log file and write the parsed object back idempotently. We are
    lucky since this idempotence is provided by the standard email.Parser and
    email.Generator.
    """

    def __init__(self, name):
        self.__name = name
        self.__email = None
        self.__dirty = False

    def get_name(self): return self.__name
    name = property(get_name)

    def load(self):
        """Read the log message from disk."""
        self.__email = email.Parser.Parser().parse(file(self.__name))
        self.__dirty = False

    def save(self):
        """Write the log message to disk."""
        if self.__dirty:
            f = file(self.__name, 'w')
            email.Generator.Generator(f).flatten(self.__email)
            self.__dirty = False

    def clear(self):
        """Clear the in-memory log message.

        When creating a new log message file, this method must be used
        first before setting the message parts. That should help early
        detection of erroneous log file names.
        """
        self.__email = email.Message.Message()
        self.__dirty = True

    def __getitem__(self, header):
        """Text of a patchlog header by name."""
        if self.__email is None:
            self.load()
        return self.__email[header]

    def __setitem__(self, header, text):
        """Set a patchlog header."""
        if self.__email is None:
            self.load()
        try:
            self.__email.replace_header(header, text)
        except KeyError:
            self.__email.add_header(header, text)
        self.__dirty = True

    def get_description(self):
        """Body of the log message."""
        if self.__email is None:
            self.load()
        assert not self.__email.is_multipart()
        return self.__email.get_payload()

    def set_description(self, s):
        """Set the body of the log message."""
        if self.__email is None:
            self.load()
        self.__email.set_payload(s)
        self.__dirty = True

    description = property(get_description, set_description)
