#include "config.h"
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "types.h"
#include "dirdb.h"
#include "boot/psetting.h"

struct dirdbEntry
{
	uint32_t parent;
	char *name; /* we pollute malloc a lot with this */
	int refcount;
};

struct __attribute__((packed)) dirdbheader
{
	char sig[60];
	uint32_t entries;
};
const char dirdbsigv1[60] = "Cubic Player Directory Data Base\x1B\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00";


static struct dirdbEntry *dirdbData=0;
static uint32_t dirdbNum=0;
static int dirdbDirty=0;

int dirdbInit(void)
{
	char path[PATH_MAX+1];
	struct dirdbheader header;
	int f;
	int i;
	int retval;

	if ((strlen(cfConfigDir)+11)>PATH_MAX)
	{
		fprintf(stderr, "dirdb: CPDIRDB.DAT path is too long\n");
		return 1;
	}
	strcpy(path, cfConfigDir);
	strcat(path, "CPDIRDB.DAT");

	if ((f=open(path, O_RDONLY))<0)
	{
		perror("open(cfConfigDir/CPDIRDB.DAT)");
		return 1;
	}

	fprintf(stderr, "Loading %s .. ", path);

	if (read(f, &header, sizeof(header))!=sizeof(header))
	{
		fprintf(stderr, "No header\n");
		close(f);
		return 1;
	}
	if (memcmp(header.sig, dirdbsigv1, 60))
	{
		fprintf(stderr, "Invalid header\n");
		close(f);
		return 1;
	}
	dirdbNum=uint32_little(header.entries);
	if (!dirdbNum)
		goto endoffile;
	dirdbData = calloc (dirdbNum, sizeof(struct dirdbEntry));
	if (!dirdbData)
	{
		dirdbNum=0;
		goto outofmemory;
	}

	for (i=0; i<dirdbNum; i++)
	{
		uint16_t len;
		if (read(f, &len, sizeof(uint16_t))!=sizeof(uint16_t))
		{
			fprintf(stderr, "EOF\n");
			close(f);
			return 1;
		}
		if (len)
		{
			len = uint16_little(len);

			if (read(f, &dirdbData[i].parent, sizeof(uint32_t))!=sizeof(uint32_t))
				goto endoffile;
			dirdbData[i].parent = uint32_little(dirdbData[i].parent);

			dirdbData[i].name=malloc(len+1);
			if (!dirdbData[i].name)
				goto outofmemory;
			if (read(f, dirdbData[i].name, len)!=len)
			{
				free(dirdbData[i].name);
				goto endoffile;
			}
			dirdbData[i].name[len]=0; /* terminate the string */
		}
	}
	close(f);
	for (i=0; i<dirdbNum; i++)
	{
		if (dirdbData[i].parent!=DIRDB_NOPARENT)
		{
			if (dirdbData[i].parent>=dirdbNum)
			{
				fprintf(stderr, "Invalid parent in a node ..");
				dirdbData[i].parent=0;
			} else
				dirdbData[dirdbData[i].parent].refcount++;
		}
	}
	fprintf(stderr, "Done\n");
	return 1;
endoffile:
	fprintf(stderr, "EOF\n");
	close(f);
	retval=1;
	goto unload;
outofmemory:
	fprintf(stderr, "out of memory\n");
	close(f);
	retval=0;
unload:
	for (i=0; i<dirdbNum; i++)
	{
		if (dirdbData[i].name)
		{
			free(dirdbData[i].name);
			dirdbData[i].name=0;
		}
		dirdbData[i].parent=0;
	}
	return retval;
}

void dirdbClose(void)
{
	int i;
	if (!dirdbNum)
		return;
	for (i=0; i<dirdbNum; i++)
		if (dirdbData[i].name)
			free(dirdbData[i].name);
	free(dirdbData);
	dirdbData = 0;
	dirdbNum = 0;
}


uint32_t dirdbFindAndRef(uint32_t parent, char const *name /* NAME_MAX + 1 */)
{
	uint32_t i;
	struct dirdbEntry *new;

	if (strlen(name)>NAME_MAX)
	{
		fprintf(stderr, "dirdbFindAndRef: name too long\n");
		return DIRDB_NOPARENT;
	}

	if ((parent!=DIRDB_NOPARENT)&&(parent>=dirdbNum))
	{
		fprintf(stderr, "dirdbFindAndRef: invalid parent\n");
		return DIRDB_NOPARENT;
	}
	for (i=0;i<dirdbNum;i++)
		if (dirdbData[i].name)
			if ((dirdbData[i].parent==DIRDB_NOPARENT)&&(!strcmp(name, dirdbData[i].name)))
			{
				dirdbData[i].refcount++;
				return i;
			}
	dirdbDirty=1;
	for (i=0;i<dirdbNum;i++)
		if (!dirdbData[i].name)
		{
reentry:
			dirdbData[i].name=strdup(name);
			dirdbData[i].parent=parent;
			dirdbData[i].refcount++;
			if (parent!=DIRDB_NOPARENT)
				dirdbData[parent].refcount++;
			return i;
		}
	
	new=realloc(dirdbData, (dirdbNum+16)*sizeof(struct dirdbEntry));
	if (!new)
	{
		fprintf(stderr, "dirdbFindAndRef: out of memory\n");
		_exit(1); /* we could exit nice here to the caller */
	}
	dirdbData=new;
	memset(dirdbData+dirdbNum, 0, 16*sizeof(struct dirdbEntry));
	i=dirdbNum;
	dirdbNum+=16;
	goto reentry;
}

void dirdbRef(uint32_t node)
{
	if (node>=dirdbNum)
	{
		fprintf(stderr, "dirdbFindAndRef: invalid node\n");
		return;
	}
	dirdbData[node].refcount++;
}

uint32_t dirdbResolvPathWithBaseAndRef(uint32_t base, char *name /* PATH_MAX + 1 */)
{
	char segment[PATH_MAX+1];
	char *next, *split;
	uint32_t retval=base, newretval;

	if (strlen(name)>PATH_MAX)
	{
		fprintf(stderr, "dirdbResolvPathWithBase: name too long\n");
		return DIRDB_NOPARENT;
	}
	next=name;
	if (retval!=DIRDB_NOPARENT)
		dirdbRef(retval);
	while (next)
	{
		if ((split=strchr(next, '/')))
		{
			strncpy(segment, next, split-next);
			segment[split-next]=0;
			next=split+1;
			if (!next)
				next=0;
		} else {
			strcpy(segment, next);
			next=0;
		}
		if (!strlen(segment))
			continue;
		newretval=dirdbFindAndRef(retval, segment);
		dirdbUnref(retval);
		retval=newretval;
	}
	return retval;
}

void dirdbUnref(uint32_t node)
{
	uint32_t parent;
	if (node>=dirdbNum)
	{
		fprintf(stderr, "dirdbUnref: invalid node\n");
		return;
	}
	dirdbData[node].refcount--;
	if (dirdbData[node].refcount)
		return;
	dirdbDirty=1;
	parent = dirdbData[node].parent;
	dirdbData[node].parent=0;
	free(dirdbData[node].name);
	dirdbData[node].name=0;
	dirdbUnref(parent);
}

void dirdbGetname(uint32_t node, char *name /*NAME_MAX+1*/)
{
	name[0]=0;
	if (node>=dirdbNum)
	{
		fprintf(stderr, "dirdbGetname: invalid node\n");
		return;
	}
	if (!dirdbData[node].name)
	{
		fprintf(stderr, "dirdbGetname: invalid node #2\n");
		return;
	}
	strcpy(name, dirdbData[node].name);

}

static void dirdbGetFullnameR(uint32_t node, char *name, int *left, int withbase)
{
	if (dirdbData[node].parent!=DIRDB_NOPARENT)
	{
		dirdbGetFullnameR(dirdbData[node].parent, name, left, withbase);
		if (!*left)
			goto errorout;
		strcat(name, "/");
		(*left)--;
	} else
		if (!withbase)
			return;

	if ((*left)<=strlen(dirdbData[node].name))
		goto errorout;
	strcat(name, dirdbData[node].name);
	(*left)-=strlen(dirdbData[node].name);
	return;
errorout:
	fprintf(stderr, "dirdbGetFullname: string grows to long\n");
	return;
}

void dirdbGetFullname(uint32_t node, char *name /* PATH_MAX+1, ends not with a / */)
{
	int i = PATH_MAX;
	*name=0;
	if (node>=dirdbNum)
	{
		fprintf(stderr, "dirdbGetFullname: invalid node\n");
		return;
	}
	dirdbGetFullnameR(node, name, &i, 1);
}

extern void dirdbGetFullnameNoBase(uint32_t node, char *name /* PATH_MAX+1, ends not with a / */)
{
	int i = PATH_MAX;
	*name=0;
	if (node>=dirdbNum)
	{
		fprintf(stderr, "dirdbGetFullname: invalid node\n");
		return;
	}
	dirdbGetFullnameR(node, name, &i, 0);
}

void dirdbFlush(void)
{
	char path[PATH_MAX+1];
	int f;
	uint32_t i;
	uint32_t max;
	uint16_t buf16;
	uint32_t buf32;
	struct dirdbheader header;

	if (!dirdbDirty)
		return;

	for (i=0;i<dirdbNum;i++)
	{
		if (dirdbData[i].name)
			if (!dirdbData[i].refcount)
			{
				dirdbData[i].refcount++;
				dirdbUnref(i);
			}
	}

	if ((strlen(cfConfigDir)+11)>PATH_MAX)
	{
		fprintf(stderr, "dirdb: CPDIRDB.DAT path is too long\n");
		return;
	}
	strcpy(path, cfConfigDir);
	strcat(path, "CPDIRDB.DAT");

	if ((f=open(path, O_WRONLY|O_CREAT|O_TRUNC, S_IREAD|S_IWRITE))<0)
	{
		perror("open(cfConfigDir/CPDIRDB.DAT)");
		return;
	}

	max=0;
	for (i=0;i<dirdbNum;i++)
		if (dirdbData[i].name)
			max=i+1;

	memcpy(header.sig, dirdbsigv1, sizeof(dirdbsigv1));
	header.entries=uint32_little(max);

	if (write(f, &header, sizeof(header))!=sizeof(header))
		goto writeerror;

	for (i=0;i<max;i++)
	{
		if (dirdbData[i].name)
		{
			int len = strlen(dirdbData[i].name);
			buf16=uint16_little(len);
			if (write(f, &buf16, sizeof(uint16_t))!=sizeof(uint16_t))
				goto writeerror;
			buf32=uint32_little(dirdbData[i].parent);
			if (write(f, &buf32, sizeof(uint32_t))!=sizeof(uint32_t))
				goto writeerror;
			if (write(f, dirdbData[i].name, len)!=len)
				goto writeerror;
		}
	}
	close(f);
	dirdbDirty=0;
	return;
writeerror:
	perror("dirdb write()");
	close(f);
}
