#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "edv_types.h"
#include "cfg.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_status_bar.h"
#include "recbin.h"
#include "recbin_cb.h"
#include "recbin_op_cb.h"
#include "recbin_dnd.h"
#include "recbin_contents_list.h"
#include "endeavour2.h"
#include "edv_op.h"
#include "edv_cb.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_list_seek.h"
#include "edv_cfg_list.h"
#include "config.h"


/* GTK+ Signal Callbacks */
void EDVRecBinContentsItemDestroyCB(gpointer data);

static gint EDVRecBinCListColumnSortDateNexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
);
static gint EDVRecBinCListColumnSortDateAccessCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateModifyCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateChangeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortDateDeleteCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);
static gint EDVRecBinCListColumnSortCapacityUsedCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
);

gint EDVRecBinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
gint EDVRecBinKeyEventCB(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
);
gint EDVRecBinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);

void EDVRecBinHandleChildAttachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);
void EDVRecBinHandleChildDetachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
);

void EDVRecBinResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
);
void EDVRecBinClickColumnCB(
	GtkCList *clist, gint column, gpointer data
);
void EDVRecBinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
void EDVRecBinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

/* Menu Item Callbacks */
void EDVRecBinMenuItemCB(GtkWidget *widget, gpointer data);
gint EDVRecBinMenuItemEnterCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);
gint EDVRecBinMenuItemLeaveCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

/* Find Bar Callbacks */
void EDVRecBinFindBarStartCB(edv_findbar_struct *fb, gpointer data);
void EDVRecBinFindBarEndCB(
	edv_findbar_struct *fb, gint total_matches, gpointer data
);
void EDVRecBinFindBarMatchCB(
	const gchar *path, const struct stat *lstat_buf,
	const gchar *excerpt, gint line_index,
	gpointer data
);

/* Status Bar Callbacks */
void EDVRecBinStatusMessageCB(const gchar *message, gpointer data);
void EDVRecBinStatusProgressCB(gfloat progress, gpointer data);

/* Write Protect Changed Callback */
void EDVRecBinWriteProtectChangedCB(
	edv_recbin_struct *recbin, gboolean state
);

/* Recycled Object Callbacks */
void EDVRecBinRecycledObjectAddedNotifyCB(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinRecycledObjectModifiedNotifyCB(
	edv_recbin_struct *recbin, guint index
);
void EDVRecBinRecycledObjectRemovedNotifyCB(
	edv_recbin_struct *recbin, guint index
);

/* Reconfigured Callback */
void EDVRecBinReconfiguredNotifyCB(edv_recbin_struct *recbin);

/* MIME Type Callbacks */
void EDVRecBinMimeTypeAddedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mime_type_struct *mt
);
void EDVRecBinMimeTypeModifiedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mime_type_struct *mt
);
void EDVRecBinMimeTypeRemovedCB(
	edv_recbin_struct *recbin, gint mt_num
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Recycle Bin contents GtkCList item "destroy" signal callback.
 */
void EDVRecBinContentsItemDestroyCB(gpointer data)
{
	EDVRecycledObjectDelete(EDV_RECYCLED_OBJECT(data));
}


/*
 *	Returns the sort code for the rows sorted by date.
 *
 *	The recycled object structures are obtained from each row and the
 *	dates are compared.
 */
static gint EDVRecBinCListColumnSortDateNexus(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2,
	gint sort_code
)
{
	gint sort_column;
	const GtkCListRow *row1 = (const GtkCListRow *)ptr1;
	const GtkCListRow *row2 = (const GtkCListRow *)ptr2;
	edv_recycled_object_struct *obj1, *obj2;

	if((clist == NULL) || (row1 == NULL) || (row2 == NULL))
	    return(0);

	sort_column = clist->sort_column;
	if((sort_column < 0) || (sort_column >= clist->columns))
	    return(0);

	obj1 = EDV_RECYCLED_OBJECT(row1->data);
	obj2 = EDV_RECYCLED_OBJECT(row2->data);
	if((obj1 == NULL) || (obj2 == NULL))
	    return(0);

	switch(sort_code)
	{
	  case 0:       /* Access time */
	    if(obj1->access_time <= obj2->access_time)
		    return((gint)(obj1->access_time < obj2->access_time));
		else
		    return(-1);
	    break;

	  case 1:       /* Modify time */
	    if(obj1->modify_time <= obj2->modify_time)
		    return((gint)(obj1->modify_time < obj2->modify_time));
		else
		    return(-1);
	    break;

	  case 2:       /* Change time */
	    if(obj1->change_time <= obj2->change_time)
		    return((gint)(obj1->change_time < obj2->change_time));
		else
		    return(-1);
	    break;

	  case 3:	/* Delete time */
	    if(obj1->deleted_time <= obj2->deleted_time)
		    return((gint)(obj1->deleted_time < obj2->deleted_time));
		else
		    return(-1);
	    break;
	}

	return(0);
}

static gint EDVRecBinCListColumnSortDateAccessCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 0
	));
}

static gint EDVRecBinCListColumnSortDateModifyCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 1
	));
}

static gint EDVRecBinCListColumnSortDateChangeCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 2
	));
}

static gint EDVRecBinCListColumnSortDateDeleteCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	return(EDVRecBinCListColumnSortDateNexus(
	    clist, ptr1, ptr2, 3
	));
}

static gint EDVRecBinCListColumnSortCapacityUsedCB(
	GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2
)
{
	const GtkCListRow       *row1 = (const GtkCListRow *)ptr1,
				*row2 = (const GtkCListRow *)ptr2;
	edv_recycled_object_struct	*o1 = EDV_RECYCLED_OBJECT(row1->data),
					*o2 = EDV_RECYCLED_OBJECT(row2->data);
	if((o1 != NULL) && (o2 != NULL))
	{
	    if(o1->size <= o2->size)
		return((gint)(o1->size < o2->size));
	    else
		return(-1);
	}
	else
	    return(0);
}


/*
 *	Recycle Bin toplevel GtkWidget "delete_event" signal callback.
 */
gint EDVRecBinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
	    return(TRUE);

	if(recbin->processing)
	    return(TRUE);

	EDVRecBinOPClose(recbin);

	return(TRUE);
}

/*
 *	Recycle Bin "key_press_event" or "key_release_event" signal
 *	callback.
 */
gint EDVRecBinKeyEventCB(
	 GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gboolean is_press;
	gint etype;
	guint keyval, state;
	GtkWidget *w, *focus_widget;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((widget == NULL) || (key == NULL) || (recbin == NULL))
	    return(status);

	if(recbin->processing)
	    return(status);

	core = recbin->core;
	if(core == NULL)
	    return(status);

	w = recbin->toplevel;
	focus_widget = (w != NULL) ? GTK_WINDOW(w)->focus_widget : NULL;
	cfg_list = core->cfg_list;
	etype = key->type;
	is_press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

/* Stop emit of signal */
#define DO_STOP_KEY_SIGNAL_EMIT	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  is_press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* If the focus_widget is not a GtkEditable then check if the
	 * keyval is an accelerator key before all subsequence checks
	 */
	if((focus_widget != NULL) ?
	    !GTK_IS_EDITABLE(focus_widget) : TRUE
	)
	{
	    edv_recbin_op op = (edv_recbin_op)EDVMatchAccelKeyOPID(
		cfg_list, EDV_CFG_PARM_RECBIN_ACCELERATOR_KEYS,
		keyval, state
	    );
	    if(op > 0)
	    {
		if(is_press)
		{
		    edv_recbin_opid_struct *opid = EDVRecBinMatchOPID(
			recbin, op
		    );
		    if(opid != NULL)
			EDVRecBinOPCB(NULL, -1, opid);
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		return(status);
	    }
	}

	/* Check which widget this signal is for */

	/* Contents List */
	if(widget == recbin->contents_clist)
	{
	    GtkCList *clist = GTK_CLIST(widget);
	    gint row = EDVCListGetSelectedLast(clist, NULL);

	    switch(keyval)
	    {
	      case GDK_space:
	      case GDK_KP_Space:
		row = clist->focus_row;
		if((row >= 0) && (row < clist->rows) && is_press)
		{
		    gboolean already_selected = FALSE;

		    /* Check if this row is already selected */
		    GList *glist = clist->selection;
		    while(glist != NULL)
		    {
			if(row == (gint)glist->data)
			{
			    already_selected = TRUE;
			    break;
			}
			glist = g_list_next(glist);
		    }

		    gtk_clist_freeze(clist);
		    if(already_selected)
			gtk_clist_unselect_row(clist, row, 0);
		    else
			gtk_clist_select_row(clist, row, 0);
		    gtk_clist_thaw(clist);
		}
		DO_STOP_KEY_SIGNAL_EMIT
		status = TRUE;
		break;

	      default:
		/* For all other alphanumeric character keys and while
		 * no modifier keys are held, attempt to seek to the
		 * item who's name starts with the letter of the key
		 * that was pressed
		 */
		if(isalnum((int)keyval) && is_press &&
		   !(state & GDK_CONTROL_MASK) &&
		   !(state & GDK_SHIFT_MASK)    
		)
		{
		    /* Get the column that is displaying the object
		     * name as column_type_name
		     */
		    gint column_type_name = -1;
		    const cfg_item_struct *cfg_list = core->cfg_list;
		    const cfg_intlist_struct *intlist = EDV_GET_INTLIST(
			EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
		    );
		    if(intlist != NULL)
		    {
			/* Iterate through column type intlist */
			gint i = 0;
			GList *glist = intlist->list;
			while(glist != NULL)
			{
			    if((edv_recbin_column_type)glist->data ==
				EDV_RECBIN_COLUMN_TYPE_NAME
			    )
			    {
				column_type_name = i;
				break;
			    }
			    i++;
			    glist = g_list_next(glist);
			}
		    }

		    gtk_clist_freeze(clist);
		    EDVCListSeekCharacter(
			clist, column_type_name, 0, keyval
		    );
		    gtk_clist_thaw(clist);

		    DO_STOP_KEY_SIGNAL_EMIT
		    status = TRUE;
		}
		break;
	    }
	}

	return(status);
#undef DO_STOP_KEY_SIGNAL_EMIT
}

/*
 *	Recycle Bin GtkWidget "button_press_event" signal callback.
 */
gint EDVRecBinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	edv_core_struct *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((widget == NULL) || (recbin == NULL))
	    return(status);

	if(recbin->processing)
	    return(status);

	core = recbin->core;
	if(core == NULL)
	    return(status);

	/* Get event type */
	etype = button->type;

	/* Check which widget this signal is for */
	if(widget == recbin->contents_clist)
	{
	    gint row, column;
	    gint rows_selected = 0, selected_row = -1;
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(widget);


	    /* Find row and column based on given coordinates */
	    if(!gtk_clist_get_selection_info(
		clist, button->x, button->y, &row, &column
	    ))
	    {
		row = -1;
		column = 0;
	    }

	    /* Get number of selected rows and highest selected row */
	    glist = clist->selection;
	    while(glist != NULL)
	    {
		rows_selected++;
		selected_row = (gint)glist->data;
		glist = g_list_next(glist);
	    }

	    /* Handle by button number */
	    switch(button->button)
	    {
	      case 3:
		if(etype == GDK_BUTTON_PRESS)
		{
		    GtkMenu *menu;

		    /* Select item before mapping menu? */
		    if(CFGItemListGetValueI(
			core->cfg_list, EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS
		    ) && (row >= 0) && (row < clist->rows))
		    {
			/* Select the row that the button was pressed over.
			 * if no key modifiers are held then this will also
			 * unselect all previously selected rows.
			 */
			gtk_clist_freeze(clist);
			if(!(button->state & GDK_CONTROL_MASK) &&
			   !(button->state & GDK_SHIFT_MASK)
			)
			    gtk_clist_unselect_all(clist);
			clist->focus_row = row;
			gtk_clist_select_row(clist, row, 0);
			gtk_clist_thaw(clist);
		    }

		    /* Update all menus and map right click menu */
		    EDVRecBinUpdateMenus(recbin);
		    menu = (GtkMenu *)recbin->contents_clist_menu;
		    if(menu != NULL)
			gtk_menu_popup(
			    menu, NULL, NULL,
			    NULL, NULL,
			    button->button, button->time
			);
		}
		status = TRUE;
		break;

	      case 2:
		if(etype == GDK_BUTTON_PRESS)
		{
		    if((row >= 0) && (row < clist->rows))
		    {
/* Renaming not allowed */
		    }
		}
		break;

	      case 1:
		/* Double click? */
		if(etype == GDK_2BUTTON_PRESS)
		{
		    if((row >= 0) && (row < clist->rows))
		    {
			/* Do recover recycled object */
			EDVRecBinOPRecover(recbin);
			status = TRUE;
		    }
		}
		break;
	    }

	    if(etype == GDK_BUTTON_PRESS)
		gtk_widget_grab_focus(widget);
	}

	return(status);
}


/*
 *	GtkHandleBox "child_attached" signal callback.
 */
void EDVRecBinHandleChildAttachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((handle_box == NULL) || (recbin == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}

/*
 *	GtkHandleBox "child_detached" signal callback.
 */
void EDVRecBinHandleChildDetachedCB(
	GtkHandleBox *handle_box, GtkWidget *child, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((handle_box == NULL) || (recbin == NULL))
	    return;

	gtk_widget_queue_resize(
	    gtk_widget_get_toplevel(GTK_WIDGET(handle_box))
	);
}


/*
 *	Recycle Bin GtkCList "resize_column" signal callback.
 */
void EDVRecBinResizeColumnCB(
	GtkCList *clist, gint column, gint width, gpointer data
)
{
	edv_core_struct *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	core = recbin->core;
	if(core == NULL)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    const gint m = EDV_RECBIN_CONTENTS_CLIST_COLUMNS_MAX;
	    edv_recbin_column_type column_type = EDV_RECBIN_COLUMN_TYPE_NAME;
	    cfg_intlist_struct *column_types_intlist, *column_width_intlist;

	    /* Get column_type from the given column index */
	    column_types_intlist = CFGItemListGetValueIntList(
		core->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	    );
	    if(column_types_intlist != NULL)
	    {
		column_type = (edv_recbin_column_type)g_list_nth_data(
		    column_types_intlist->list, column
		);
	    }

	    /* Get column widths intlist */
	    column_width_intlist = CFGItemListGetValueIntList(
		core->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
	    );
	    if(column_width_intlist != NULL)
	    {
		GList *glist = g_list_nth(
		    column_width_intlist->list, (guint)column_type
		);
		if(glist != NULL)
		{
		    glist->data = (gpointer)width;
		}
		else
		{
		    /* The column type is not in the column widths
		     * list so create a new column widths list from
		     * the current column widths list and include
		     * the missing column type
		     */
		    gint i;
		    GList *glist_new = NULL;

		    glist = column_width_intlist->list;

		    for(i = 0; i < m; i++)
		    {
			if(glist != NULL)
			{
			    glist_new = g_list_append(
				glist_new, glist->data
			    );
			    glist = g_list_next(glist);
			}
			else
			    glist_new = g_list_append(glist_new, NULL);
		    }

		    g_list_free(column_width_intlist->list);
		    column_width_intlist->list = glist_new;

		    glist = g_list_nth(glist_new, (guint)column_type);
		    if(glist != NULL)
			glist->data = (gpointer)width;
		}
	    }
	}
}

/*
 *	Recycle Bin GtkCList "click_column" signal callback.
 */
void EDVRecBinClickColumnCB(
	GtkCList *clist, gint column, gpointer data
)
{
	edv_core_struct *core;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	core = recbin->core;
	if(core == NULL)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    cfg_intlist_struct *column_types_intlist;
	    GtkCListCompareFunc cmp_func = NULL;
	    GtkCListCompareFunc cmp_func_str =
		 (GtkCListCompareFunc)EDVCListColumnSortStringCB;
	    GtkCListCompareFunc cmp_func_num =
		(GtkCListCompareFunc)EDVCListColumnSortNumberCB;


	    EDVRecBinSetBusy(recbin, TRUE);
	    GUIBlockInput(recbin->toplevel, TRUE);
	    recbin->processing = TRUE;


	    /* Get column types mapping */
	    column_types_intlist = CFGItemListGetValueIntList(
		core->cfg_list, EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	    );
	    if(column_types_intlist != NULL)
	    {
		edv_recbin_column_type column_type = (edv_recbin_column_type)g_list_nth_data(
		    column_types_intlist->list,
		    column
		);
		switch(column_type)
		{
		  case EDV_RECBIN_COLUMN_TYPE_NAME:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_SIZE:
		    cmp_func = cmp_func_num;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_TYPE:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_PERMISSIONS:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_OWNER:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_GROUP:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_ACCESS:
		    cmp_func = EDVRecBinCListColumnSortDateAccessCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_MODIFIED:
		    cmp_func = EDVRecBinCListColumnSortDateModifyCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_CHANGED:
		    cmp_func = EDVRecBinCListColumnSortDateChangeCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_DATE_DELETED:
		    cmp_func = EDVRecBinCListColumnSortDateDeleteCB;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_LINKED_TO:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_ORIGINAL_LOCATION:
		    cmp_func = cmp_func_str;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_INDEX:
		    cmp_func = cmp_func_num;
		    break;
		  case EDV_RECBIN_COLUMN_TYPE_CAPACITY_USED:
		    cmp_func = EDVRecBinCListColumnSortCapacityUsedCB;
		    break;
		}
	    }


	    gtk_clist_freeze(clist);

	    /* Set sort column settings on the clist */
	    if(column != clist->sort_column)
		gtk_clist_set_sort_column(clist, column);
	    else
		gtk_clist_set_sort_type(
		    clist,
		    (clist->sort_type == GTK_SORT_ASCENDING) ?
			GTK_SORT_DESCENDING : GTK_SORT_ASCENDING
		);
	    if(cmp_func != NULL)
		gtk_clist_set_compare_func(clist, cmp_func);

	    /* Sort rows */
	    gtk_clist_sort(clist);

	    gtk_clist_thaw(clist);

	    recbin->processing = FALSE;
	    GUIBlockInput(recbin->toplevel, FALSE);
	    EDVRecBinSetBusy(recbin, FALSE);

	    EDVRecBinUpdateMenus(recbin);
	}
}

/*
 *	Recycle Bin GtkCList "select_row" signal callback.
 */
void EDVRecBinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    edv_core_struct *core = recbin->core;

	    /* Get total number of objects selected */
	    const gint nselected = g_list_length(clist->selection);

	    /* Get object */
	    edv_recycled_object_struct *obj = EDV_RECYCLED_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );

	    /* Update selected row */
	    recbin->contents_clist_selected_row = row;

	    /* Update DND icon */
	    EDVRecBinContentsDNDSetIcon(recbin, row, column);

	    if(obj != NULL)
	    {
		/* Update status bar message */
		if(!STRISEMPTY(obj->name))
		{
		    gchar *s, *size_str = NULL;
		    const gchar *type_str = NULL;

		    /* Get object type string and size string */
		    switch(obj->type)
		    {
		      case EDV_OBJECT_TYPE_UNKNOWN:
			type_str = "Object";
			size_str = NULL;
			break;
			break;
		      case EDV_OBJECT_TYPE_FILE:
			type_str = "File";
			size_str = g_strdup_printf(
			    " (%s byte%s)",
			    EDVGetObjectSizeStr(
				recbin->core,
				obj->size
			    ),
			    (obj->size == 1) ? "" : "s"
			);
			break;
		      case EDV_OBJECT_TYPE_DIRECTORY:
			type_str = "Directory";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_LINK:
			type_str = "Link";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
			type_str = "Block device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
			type_str = "Character device";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_FIFO:
			type_str = "FIFO pipe";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_SOCKET:
			type_str = "Socket";
			size_str = NULL;
			break;
		      case EDV_OBJECT_TYPE_ERROR:
			type_str = "Error";
			size_str = NULL;
			break;
		    }

		    /* Set status bar message */
		    if(nselected > 1)
			s = g_strdup_printf(
			    "%s objects selected",
			    EDVGetObjectSizeStr(core, nselected)
			);
		    else if(!strcmp(obj->name, ".."))
			s = g_strdup_printf(
			    "Parent directory selected"
			);
		    else
			s = g_strdup_printf(
			    "%s \"%s\" selected%s",
			    type_str, obj->name,
			    (size_str != NULL) ? size_str : ""
			);
		    EDVStatusBarMessage(
			recbin->status_bar, s, FALSE
		    );
		    g_free(s);
		    g_free(size_str);
		}
		else
		{
		    EDVStatusBarMessage(
			recbin->status_bar,
			"Object with NULL name selected",
			FALSE
		    );
		}
	    }

	    /* Check if selected row is fully visible, if not then adjust
	     * scroll position to try and make it visible
	     */
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    row, -1,	/* Row, column */
		    0.5f, 0.0f	/* Row, column */
		);

/*	    EDVRecBinSetTitle(recbin); */
	    EDVRecBinUpdateMenus(recbin);
	}
}

/*
 *	Recycle Bin GtkCList "unselect_row" signal callback.
 */
void EDVRecBinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((clist == NULL) || (recbin == NULL))
	    return;

	if(recbin->processing)
	    return;

	/* Check which clist this signal is for */
	if(GTK_WIDGET(clist) == recbin->contents_clist)
	{
	    gchar *s;
	    edv_core_struct *core = recbin->core;

	    /* Get total number of objects selected */
	    const gint nselected = g_list_length(clist->selection);

	    /* Update status bar message */
	    if(nselected > 0)
		s = g_strdup_printf(
		    "%s object%s selected",
		    EDVGetObjectSizeStr(core, nselected),
		    (nselected == 1) ? "" : "s"  
		);
	    else
		s = STRDUP("No objects selected");
	    EDVStatusBarMessage(
		recbin->status_bar, s, FALSE
	    );
	    g_free(s);

	    EDVRecBinUpdateMenus(recbin);
	}
}


/*
 *	Menu item activate callback.
 *
 *	The given client data must be a edv_recbin_opid_struct *.
 */
void EDVRecBinMenuItemCB(GtkWidget *widget, gpointer data)
{
	EDVRecBinOPCB(NULL, -1, data);
}

/*
 *	Menu item "enter_notify_event" signal callback.
 *
 *	The given client data must be a edv_recbin_opid_struct *.
 */
gint EDVRecBinMenuItemEnterCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVRecBinOPEnterCB(NULL, -1, data);
	return(TRUE);
}

/*
 *	Menu item "leave_notify_event" signal callback.
 *
 *	The given client data must be a edv_recbin_opid_struct *.
 */
gint EDVRecBinMenuItemLeaveCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	EDVRecBinOPLeaveCB(NULL, -1, data);
	return(TRUE);
}


/*
 *	Find bar start find callback.
 */
void EDVRecBinFindBarStartCB(edv_findbar_struct *fb, gpointer data)
{
	GtkCList *clist;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
	    return;

	clist = (GtkCList *)recbin->contents_clist;
	if(clist == NULL)
	    return;

	EDVRecBinSetBusy(recbin, TRUE);
/*	GUIBlockInput(recbin->toplevel, TRUE); */

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	EDVRecBinUpdateMenus(recbin);
}

/*
 *	Find bar end find callback.
 */
void EDVRecBinFindBarEndCB(
	edv_findbar_struct *fb, gint total_matches, gpointer data
)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
	    return;

/*	GUIBlockInput(recbin->toplevel, FALSE); */
	EDVRecBinSetBusy(recbin, FALSE);
}


/*
 *	Find bar match callback.
 */
void EDVRecBinFindBarMatchCB(
	const gchar *path, const struct stat *lstat_buf,
	const gchar *excerpt, gint line_index,
	gpointer data
)
{
	gint row;
	GtkCList *clist;
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if((path == NULL) || (lstat_buf == NULL) || (recbin == NULL))
	    return;

	clist = (GtkCList *)recbin->contents_clist;
	if(clist == NULL)
	    return;

	row = EDVRecBinContentsFindRowByIndex(recbin, (guint)atoi(path));
	if((row >= 0) && (row < clist->rows))
	    gtk_clist_select_row(clist, row, 0);
}


/*
 *	Status message callback.
 */
void EDVRecBinStatusMessageCB(const gchar *message, gpointer data)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
	    return;

	EDVStatusBarMessage(recbin->status_bar, message, FALSE);
}

/*
 *	Status progress callback.
 */
void EDVRecBinStatusProgressCB(gfloat progress, gpointer data)
{
	edv_recbin_struct *recbin = EDV_RECBIN(data);
	if(recbin == NULL)
	    return;

	EDVStatusBarProgress(recbin->status_bar, progress, FALSE);
}





/*
 *	Write protect changed callback.
 */
void EDVRecBinWriteProtectChangedCB(
	edv_recbin_struct *recbin, gboolean state
)
{
	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	EDVRecBinUpdateMenus(recbin);
}

/*
 *	Recycled object added callback.
 */
void EDVRecBinRecycledObjectAddedNotifyCB(
	edv_recbin_struct *recbin, guint index
)
{
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core = recbin->core;
	if(core == NULL)
	    return;

	/* Contents List */
	EDVRecBinContentsObjectAddedNotify(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a change
	 * in the number of recycled objects from 0.
	 */
	if(core->last_recbin_items != recbin->last_recbin_items)
	    EDVRecBinUpdateMenus(recbin);
}

/*
 *	Recycled object modified callback.
 */
void EDVRecBinRecycledObjectModifiedNotifyCB(
	edv_recbin_struct *recbin, guint index
)
{
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core = recbin->core;
	if(core == NULL)
	    return;

	/* Contents List */
	EDVRecBinContentsObjectModifiedNotify(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a
	 * change in the number of recycled objects from 0
	 */
	if(core->last_recbin_items != recbin->last_recbin_items)
	    EDVRecBinUpdateMenus(recbin);
}

/*
 *	Recycled object removed callback.
 */
void EDVRecBinRecycledObjectRemovedNotifyCB(
	edv_recbin_struct *recbin, guint index
)
{
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core = recbin->core;
	if(core == NULL)
	    return;

	/* Contents List */
	EDVRecBinContentsObjectRemovedNotify(recbin, index);

	/* Update menus for the purpose of updating the Recycle Bin
	 * toplevel window's WM icon. Do this only if there is a change
	 * in the number of recycled objects from 0
	 */
	if(core->last_recbin_items != recbin->last_recbin_items)
	    EDVRecBinUpdateMenus(recbin);
}


/*
 *	Reconfigured callback.
 */
void EDVRecBinReconfiguredNotifyCB(edv_recbin_struct *recbin)
{
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *status_bar;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	core = recbin->core;
	if(core == NULL)
	    return;

	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;


	/* Reset last state markers so that resources get explicitly
	 * checked due to reconfiguring
	 */
	recbin->last_recbin_items = -1;
	recbin->last_write_protect_state = -1;


	/* Update title */
	EDVRecBinSetTitle(recbin);

	/* Update Accelkey Labels */
	EDVRecBinAccelkeysRegenerate(recbin);

	/* Regenerate Tool Bar */
	EDVRecBinToolBarRegenerate(recbin);

	/* Show tool bar? */
	w = recbin->tool_bar_handle;
	if(w != NULL)
	{
	    recbin->tool_bar_map_state = CFGItemListGetValueI(
		cfg_list, EDV_CFG_PARM_RECBIN_SHOW_TOOL_BAR
	    );
	    if(recbin->tool_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show find bar? */
	w = recbin->find_bar_handle;
	if(w != NULL)
	{
	    recbin->find_bar_map_state = CFGItemListGetValueI(
		cfg_list, EDV_CFG_PARM_RECBIN_SHOW_FIND_BAR
	    );
	    if(recbin->find_bar_map_state)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Show status bar? */
	status_bar = recbin->status_bar;
	if(status_bar != NULL)
	{
	    recbin->status_bar_map_state = CFGItemListGetValueI(
		cfg_list, EDV_CFG_PARM_RECBIN_SHOW_STATUS_BAR
	    );
	    if(recbin->status_bar_map_state)
		EDVStatusBarMap(status_bar);
	    else
		EDVStatusBarUnmap(status_bar);
	}


	/* Update RC styles */
	w = recbin->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = recbin->contents_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, lists_rcstyle);
	w = recbin->contents_clist_menu;
	if((w != NULL) && (standard_rcstyle != NULL))
	    gtk_widget_modify_style_recursive(w, standard_rcstyle);

	/* Realize listings */
	EDVRecBinContentsRealizeListing(recbin);

	/* Update menus */
	EDVRecBinUpdateMenus(recbin);

	/* Notify toplevel to resize */
	w = recbin->toplevel;
	if(w != NULL)
	    gtk_widget_queue_resize(w);
}


/*
 *	MIME Type added callback.
 */
void EDVRecBinMimeTypeAddedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mime_type_struct *mt
)
{
	/* Treat a MIME Type added the same as it would be for a MIME
	 * Type modified, forward signal to the MIME Type modified
	 * callback
	 */
	EDVRecBinMimeTypeModifiedCB(recbin, mt_num, mt);
}

/*
 *	MIME Type modified callback.
 */
void EDVRecBinMimeTypeModifiedCB(
	edv_recbin_struct *recbin,
	gint mt_num, edv_mime_type_struct *mt
)
{
	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	/* Realize listings */
	EDVRecBinContentsRealizeListing(recbin);

/*	EDVRecBinUpdateMenus(recbin); */
}

/*
 *	MIME Type removed callback.
 */
void EDVRecBinMimeTypeRemovedCB(
	edv_recbin_struct *recbin, gint mt_num
)
{
	if(recbin == NULL)
	    return;

	if(recbin->processing)
	    return;

	/* Realize listings */
	EDVRecBinContentsRealizeListing(recbin);

/*	EDVRecBinUpdateMenus(recbin); */
}
