(in-package :contextl)

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun check-op/layer (op layer-name)
    (unless (member op '(+ -) :test #'eq)
      (error "Incorrect activation operator ~S. Must be + or -." op))
    (unless (symbolp layer-name)
      (error "Not a layer name: ~S." layer-name))
    (when (eq layer-name 't)
      (case op
        (+ (error "The root layer cannot be placed in front of other layers."))
        (- (error "The root layer must always be active."))))))

(defvar *layer-class-definers*
  (make-enclosing-package "LAYER-CLASS-DEFINERS"))

(defun defining-layer (name)
  (if (eq name 't) 't
    (enclose-symbol name *layer-class-definers*)))

(defclass layer-class (standard-class)
  ((layer-name :initarg original-name
               :reader %layer-name)))

(defmethod validate-superclass
           ((class layer-class)
            (superclass (eql (find-class 'standard-object))))
  t)

(defmacro deflayer (name &optional superlayers &body options)
  `(defclass ,(defining-layer name) ,(mapcar #'defining-layer superlayers)
     ,@(if options options '(()))
     (:metaclass layer-class)
     (original-name ,name)))

(defgeneric layer-name (layer)
  (:method ((layer symbol)) layer)
  (:method ((layer layer-class)) (car (%layer-name layer))))

(defgeneric find-layer (layer &optional errorp)
  (:method ((layer symbol) &optional (errorp t))
   (or (find-class (defining-layer layer) nil)
       (when errorp
         (error "There is no layer named ~S." layer))))
  (:method ((layer layer-class) &optional (errorp t))
   (declare (ignore errorp))
   layer))

(defun layer-prototype (layer)
  (class-prototype (find-layer layer)))

(defclass root-specializer () ()
  (:metaclass layer-class))
(finalize-inheritance (find-class 'root-specializer))

(defstruct layer-context
  (prototype (class-prototype (find-class 'root-specializer))
             :type standard-object
             :read-only t)
  (specializer (find-class 'root-specializer)
               :type layer-class
               :read-only t)
  (children/ensure-active () :type list)
  (children/ensure-inactive () :type list))

#-allegro
(declaim (type layer-context *root-context* *active-context*))
#+allegro
(eval-when (:load-toplevel :execute)
  (proclaim '(type layer-context *root-context* *active-context*)))

(defparameter *root-context* (make-layer-context))

(defparameter *active-context* *root-context*)

(defun layer-active-p (layer &optional (context *active-context*))
  (subtypep (layer-context-specializer context)
            (find-layer layer)))

(defun maybe-create-active-layer-context (layer)
  (let ((layer-class (find-layer layer))
        (active-context-specializer (layer-context-specializer *active-context*)))
    (when (subtypep active-context-specializer layer-class)
      (let ((first-layer (first (class-direct-superclasses active-context-specializer))))
        (cond ((eq first-layer layer-class)
               (return-from maybe-create-active-layer-context *active-context*))
              ((subtypep first-layer layer-class)
               (error "Layer ~S cannot be placed in front of ~S because the latter inherits from the former." (layer-name layer-class) (layer-name first-layer)))
              (t (let ((active-context (ensure-inactive-layer-context layer)))
                   (setq active-context-specializer (layer-context-specializer active-context)))))))
    (let ((new-specializer
           (make-instance 'layer-class
                          :direct-superclasses
                          (list layer-class active-context-specializer))))
      (finalize-inheritance new-specializer)
      (make-layer-context
       :prototype (class-prototype new-specializer)
       :specializer new-specializer))))

(declaim (inline ensure-active-layer-context))

(defun ensure-active-layer-context (layer)
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (or (getf (layer-context-children/ensure-active *active-context*) layer)
      (setf (getf (layer-context-children/ensure-active *active-context*) layer)
            (maybe-create-active-layer-context layer))))

(defun ensure-active-layer (layer-name)
  (check-op/layer '+ layer-name)
  (setf *active-context*
        (locally
          (declare (optimize (speed 3) (debug 0) (safety 0)
                             (compilation-speed 0)))
          (ensure-active-layer-context layer-name))))

(defun maybe-create-inactive-layer-context (layer)
  (let ((layer-class (find-layer layer))
        (active-context-specializer (layer-context-specializer *active-context*)))
    (if (not (subtypep active-context-specializer layer-class))
        *active-context*
      (loop for context-specializer = active-context-specializer
            then (second (class-direct-superclasses context-specializer))
            for active-layers = (list (first (class-direct-superclasses context-specializer)))
            then (cons (first (class-direct-superclasses context-specializer)) active-layers)
            until (eq context-specializer (find-class 'root-specializer))
            finally
            (return (loop with new-layer-context = *root-context*
                          for active-layer in (cdr active-layers)
                          if (subtypep active-layer layer-class) do
                          (unless (eq active-layer layer-class)
                            (error "The layer ~S cannot be deactivated or rearranged because the active layer ~S inherits from it." (layer-name layer-class) (layer-name active-layer)))
                          else do
                          (setq new-layer-context
                                (let ((*active-context* new-layer-context))
                                  (ensure-active-layer-context (layer-name active-layer))))
                          finally (return new-layer-context)))))))

(declaim (inline ensure-inactive-layer-context))

(defun ensure-inactive-layer-context (layer)
  (declare (optimize (speed 3) (debug 0) (safety 0)
                     (compilation-speed 0)))
  (or (getf (layer-context-children/ensure-inactive *active-context*) layer)
      (setf (getf (layer-context-children/ensure-inactive *active-context*) layer)
            (maybe-create-inactive-layer-context layer))))

(defun ensure-inactive-layer (layer-name)
  (check-op/layer '- layer-name)
  (setf *active-context*
        (locally
          (declare (optimize (speed 3) (debug 0) (safety 0)
                             (compilation-speed 0)))
          (ensure-inactive-layer-context layer-name))))

(defmacro with-active-layer (layer-name &body body)
  (check-op/layer '+ layer-name)
  `(let ((*active-context*
          (locally
            (declare (optimize (speed 3) (debug 0) (safety 0)
                               (compilation-speed 0)))
            (ensure-active-layer-context ',layer-name))))
     ,@body))

(defmacro with-active-layers ((&rest layer-names) &body body)
  (if layer-names
      `(with-active-layer ,(car layer-names)
         (with-active-layers ,(cdr layer-names)
           ,@body))
    `(progn ,@body)))

(defmacro with-inactive-layer (layer-name &body body)
  (check-op/layer '- layer-name)
  `(let ((*active-context*
          (locally
            (declare (optimize (speed 3) (debug 0) (safety 0)
                               (compilation-speed 0)))
            (ensure-inactive-layer-context ',layer-name))))
     ,@body))

(defmacro with-inactive-layers ((&rest layer-names) &body body)
  (if layer-names
      `(with-inactive-layer ,(car layer-names)
         (with-inactive-layers ,(cdr layer-names)
           ,@body))
    `(progn ,@body)))

(defun funcall-with-layer (op layer function &rest args)
  (declare (dynamic-extent args))
  (check-op/layer op layer)
  (let ((*active-context*
         (locally
           (declare (optimize (speed 3) (debug 0) (safety 0)
                              (compilation-speed 0)))
           (ecase op
             (+ (ensure-active-layer-context layer))
             (- (ensure-inactive-layer-context layer))))))
    (apply function args)))

(define-compiler-macro funcall-with-layer (&whole form op layer function &rest args)
  (if (and (consp op) (eq (car op) 'quote)
           (consp layer) (eq (car layer) 'quote))
      (let ((evop (eval op))
            (evlayer (eval layer)))
        (check-op/layer evop evlayer)
        (ecase evop
          (+ `(with-active-layer ,evlayer
                (funcall ,function ,@args)))
          (- `(with-inactive-layer ,evlayer
                (funcall ,function ,@args)))))
    form))

(defun funcall-with-layers (layers function &rest args)
  (declare (dynamic-extent args))
  (if layers
      (funcall-with-layer
       (car layers)
       (cadr layers)
       (lambda ()
         (apply #'funcall-with-layers (cddr layers) function args)))
    (apply function args)))

(define-compiler-macro funcall-with-layers (&whole form layers function &rest args)
  (cond ((null layers) `(funcall ,function ,@args))
        ((and (consp layers) (eq (car layers) 'quote))
         (let ((evlayers (eval layers)))
           (unless (listp evlayers)
             (error "Incorrect layers argument to funcall-with-layers: ~S." layers))
           (if (null evlayers)
               `(funcall ,function ,@args)
             (progn
               (check-op/layer (car evlayers) (cadr evlayers))
               (ecase (car evlayers)
                 (+ `(with-active-layer ,(cadr evlayers)
                       (funcall-with-layers ',(cddr evlayers) ,function ,@args)))
                 (- `(with-inactive-layer ,(cadr evlayers)
                       (funcall-with-layers ',(cddr evlayers) ,function ,@args))))))))
        (t form)))

(defun apply-with-layer (op layer function &rest args)
  (declare (dynamic-extent args))
  (check-op/layer op layer)
  (let ((*active-context*
         (locally
           (declare (optimize (speed 3) (debug 0) (safety 0)
                              (compilation-speed 0)))
           (ecase op
             (+ (ensure-active-layer-context layer))
             (- (ensure-inactive-layer-context layer))))))
    (apply #'apply function args)))

(define-compiler-macro apply-with-layer (&whole form op layer function &rest args)
  (if (and (consp op) (eq (car op) 'quote)
           (consp layer) (eq (car layer) 'quote))
      (let ((evop (eval op))
            (evlayer (eval layer)))
        (check-op/layer evop evlayer)
        (ecase evop
          (+ `(with-active-layer ,evlayer
                (apply ,function ,@args)))
          (- `(with-inactive-layer ,evlayer
                (apply ,function ,@args)))))
    form))

(defun apply-with-layers (layers function &rest args)
  (declare (dynamic-extent args))
  (if layers
      (funcall-with-layer
       (car layers)
       (cadr layers)
       (lambda ()
         (apply #'apply-with-layers (cddr layers) function args)))
    (apply #'apply function args)))

(define-compiler-macro apply-with-layers (&whole form layers function &rest args)
  (cond ((null layers) `(apply ,function ,@args))
        ((and (consp layers) (eq (car layers) 'quote))
         (let ((evlayers (eval layers)))
           (unless (listp evlayers)
             (error "Incorrect layers argument to apply-with-layers: ~S." layers))
           (if (null evlayers)
               `(apply ,function ,@args)
             (progn
               (check-op/layer (car evlayers) (cadr evlayers))
               (ecase (car evlayers)
                 (+ `(with-active-layer ,(cadr evlayers)
                       (apply-with-layers ',(cddr evlayers) ,function ,@args)))
                 (- `(with-inactive-layer ,(cadr evlayers)
                       (apply-with-layers ',(cddr evlayers) ,function ,@args))))))))
        (t form)))
