#!/usr/bin/env python
# arch-tag: 3522791e-8f35-4a09-88ab-7a56aea73aa1
# Copyright (C) 2004 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for archives without any content.
"""

import os

import pybaz as arch
from pybaz import errors
from pybaz.pathname import DirName, FileName, PathName

import framework
import fixtures


class MakeArchive(framework.NewTestCase):
    fixture = fixtures.ArchiveDirFixture()

    def test_no_registered_archives(self):
        """archives is empty when no archive is registered."""
        archives = list(arch.iter_archives())
        self.failUnlessEqual(len(archives), 0)

    def test_archive_name(self):
        """Archive.name is present and read-only."""
        name = self.fixture.arch_name
        archive = arch.Archive(name)
        self.failUnless(hasattr(archive, 'name'))
        self.assertRaises(
            AttributeError, setattr, archive, 'name', 'error')
        self.assertEqual(archive.name, name)

    def test_archive_location_readonly(self):
        """archive.location is present and read-only."""
        name = self.fixture.arch_name
        archive = arch.Archive(name)
        self.failUnless(hasattr(archive, 'location'))
        self.failUnlessRaises(
            AttributeError, setattr, archive, 'location', 'error')

    def test_archive_location_not_registered(self):
        """Archive.location is None if not registered."""
        name = self.fixture.arch_name
        archive = arch.Archive(name)
        self.assertEqual(archive.location, None)

    def test_make_archive(self):
        """make_archive does not fail."""
        name = self.fixture.arch_name
        location = self.fixture.location(name)
        archive = arch.make_archive(name, location)
        self.failUnlessEqual(archive.name, name)
        return archive

    def test_make_already_made_archive(self):
        """Making the same archive twice fails."""
        name = self.fixture.arch_name
        location = self.fixture.location(name)
        arch.make_archive(name, location)
        self.failUnlessRaises(
            arch.ExecProblem, arch.make_archive, name, location)

    def test_archive_location(self):
        """Archive.location is corrrect when archive is registered."""
        name = self.fixture.arch_name
        location = self.fixture.location(name)
        archive = self.test_make_archive()
        self.assertEqual(archive.location, location)

    def test_iter_archives_one(self):
        """iter_archive is correct when one archive is registered."""
        name = self.fixture.arch_name
        location = self.fixture.location(name)
        self.test_make_archive()
        archives = list(arch.iter_archives())
        self.assertEqual(len(archives), 1)
        archive = archives[0]
        self.assertEqual(archive.name, name)
        self.failUnlessEqual(archive.location, location)


class MakeArchiveVersion(framework.NewTestCase):
    fixture = fixtures.ArchiveDirFixture()

    def _help_create_archive(self, **kwargs):
        name = self.fixture.arch_name
        location = self.fixture.location(name)
        archive = arch.make_archive(name, location, **kwargs)
        return archive

    def read_version(self, archive):
        location = self.fixture.location(archive.name)
        return open(DirName(location)/'.archive-version').read()

    def test_version_string(self):
        """Archive.version_string works."""
        archive = self._help_create_archive()
        version_string = self.read_version(archive).rstrip('\n')
        self.assertEqual(version_string, archive.version_string)

    def test_make_archive_version_baz(self):
        """make_archive's default version string says baz."""
        archive = self._help_create_archive()
        expected = 'Bazaar archive format 1 0'
        self.assertEqual(expected, archive.version_string)

    def test_make_archive_version_tla(self):
        """make_archive's option tla works."""
        archive = self._help_create_archive(tla=True)
        expected = 'Hackerlab arch archive directory, format version 2.'
        self.assertEqual(expected, archive.version_string)


class IsNotRegistered(framework.NewTestCase):
    fixture = fixtures.UnregisteredArchiveFixture()

    def test_archive_not_registered(self):
        """Archive.is_registered is False if not registered."""
        archive = self.fixture.archive
        self.assertEqual(archive.is_registered(), False)

    def test_archive_not_exists(self):
        """Archive.exists raises ArchiveNotRegistered if not registered."""
        archive = self.fixture.archive
        self.assertRaises(errors.ArchiveNotRegistered, archive.exists)


class ArchiveSubclass(arch.Archive):
    """Dummy Archive subclass."""
    pass


class IsRegistered(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture()

    def test_archive_registered(self):
        """Archive.is_registered is True if registered."""
        archive = self.fixture.archive
        self.assertEqual(archive.is_registered(), True)
        archive = ArchiveSubclass(archive)
        self.assertEqual(archive.is_registered(), True)

    def test_archive_exists(self):
        """Archive.exists() is True if registered."""
        archive = self.fixture.archive
        self.assertEqual(archive.exists(), True)
        archive = ArchiveSubclass(archive)
        self.assertEqual(archive.exists(), True)


class LocationUtils(framework.NewTestCase):

    def test_is_url(self):
        """_Archive_Baz_1_3._is_url works correctly."""
        from pybaz import _builtin
        _is_url = _builtin._Archive_Baz_1_3._is_url
        self.failUnless(_is_url('http://example.com'))
        self.failUnless(_is_url('sftp://user@example.com/archive'))
        self.failIf(_is_url('foo'))
        self.failIf(_is_url('/dev/null'))
        self.failIf(_is_url('dir/base'))


class UnregisterArchive(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture()

    def test(self):
        archive = self.fixture.archive
        self.assertEqual(archive.is_registered(), True)
        archive.unregister()
        self.assertEqual(archive.is_registered(), False)


class RegisterableArchiveFixture(fixtures.ArchiveFixture):

    """A fixture that provides an archive that can be registered."""

    def setUp(self):
        super(RegisterableArchiveFixture, self).setUp()
        self.archive.unregister()


class RegisterArchive(framework.NewTestCase):
    fixture = RegisterableArchiveFixture()

    def test_with_name(self):
        name = self.fixture.archive.name
        location = self.fixture.location(name)
        archive = arch.register_archive(name, location)
        self.assertEqual(archive.is_registered(), True)
        self.assertEqual(archive.name, name)
        self.assertEqual(archive.location, location)

    def test_registered_name(self):
        name = self.fixture.archive.name
        other_name = name + '-OTHER'
        location = self.fixture.location(name)
        archive = arch.register_archive(other_name, location)
        self.assertEqual(archive.is_registered(), True)
        self.assertEqual(archive.name, other_name)
        self.assertEqual(archive.location, location)

    def test_auto_name(self):
        name = self.fixture.archive.name
        location = self.fixture.location(name)
        archive = arch.register_archive(None, location)
        self.assertEqual(archive.is_registered(), True)
        self.assertEqual(archive.name, name)
        self.assertEqual(archive.location, location)


class RegisteredNameFixture(RegisterableArchiveFixture):

    """A fixture with an archive registered under a non-official name."""

    def setUp(self):
        super(RegisterableArchiveFixture, self).setUp()
        name = self.archive.name
        self.official_name = name
        self.archive.unregister()
        other_name = name + '-OTHER'
        location = self.location(name)
        self.archive = arch.register_archive(other_name, location)


class UnregisterRegisteredName(framework.NewTestCase):
    fixture = RegisteredNameFixture()

    def test_unregister_registered_name(self):
        archive = self.fixture.archive
        official = arch.Archive(self.fixture.official_name)
        self.assertEqual(archive.is_registered(), True)
        self.assertEqual(official.is_registered(), False)
        archive.unregister()
        self.assertEqual(archive.is_registered(), False)
        self.assertEqual(official.is_registered(), False)


class MakeMirror(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture(tla=False)

    def _help_make_mirror(self, kwargs, meta_name, meta_expected):
        master = self.fixture.archive
        mirror_name = master.name + '-MIRROR'
        location = self.fixture.location(mirror_name)
        mirror = master.make_mirror(mirror_name, location, **kwargs)
        meta_file = DirName(mirror.location)/'=meta-info'/meta_name
        self.failUnless(os.path.isfile(meta_file))
        meta_data = open(meta_file).read()
        self.failUnlessEqual(meta_expected, meta_data)

    def test_make_mirror(self):
        """Archive.make_mirror works for creating a -MIRROR."""
        self._help_make_mirror({}, 'mirror', self.fixture.arch_name + '\n')

    def test_make_mirror_listing(self):
        """Archive.make_mirror can set up http-blows."""
        self._help_make_mirror({'listing': True}, 'http-blows',
                               'it sure does\n')

    def test_make_mirror_signed(self):
        """Archive.make_mirror can set up signed-archive."""
        self._help_make_mirror({'signed': True}, 'signed-archive',
                               'system cracking is (nearly always) lame\n')

    def test_make_mirror_relative(self):
        """Archive.make_mirror works with relative location."""
        meta_name, meta_expected = 'mirror', self.fixture.arch_name + '\n'
        master = self.fixture.archive
        mirror_name = master.name + '-MIRROR'
        location = self.fixture.location(mirror_name)
        assert location.basename() == mirror_name
        dirname = location.dirname()
        os.chdir(dirname)
        mirror = master.make_mirror(mirror_name, mirror_name)
        meta_file = DirName(mirror.location)/'=meta-info'/meta_name
        self.failUnless(os.path.isfile(meta_file))
        meta_data = open(meta_file).read()
        self.failUnlessEqual(meta_expected, meta_data)


class ArchiveMetaInfo(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture(tla=False)

    def open_meta_info(self, name, flags):
        archive = self.fixture.archive
        return open(DirName(archive.location)/'=meta-info'/name, flags)

    def test_meta_info(self):
        """Archive._meta_info works."""
        archive = self.fixture.archive
        self.assertEqual(archive.name, archive._meta_info('name'))
        self.assertRaises(KeyError, archive._meta_info, 'not_here')
        self.open_meta_info('empty', 'w').close()
        self.assertEqual(None, archive._meta_info('empty'))
        print >> self.open_meta_info('empty-line', 'w')
        self.assertEqual(str(), archive._meta_info('empty-line'))

    def test_has_meta_info(self):
        """Archive._has_meta_info works."""
        archive = self.fixture.archive
        self.failIf(archive._has_meta_info('not_here'))
        self.open_meta_info('empty', 'w')
        self.assert_(archive._has_meta_info('empty'))
        print >> self.open_meta_info('empty-line', 'w')
        self.assert_(archive._has_meta_info('empty-line'))
        print >> self.open_meta_info('something', 'w'), 'hello'
        self.assert_(archive._has_meta_info('something'))

    def test_is_signed(self):
        """Archive.is_signed works."""
        archive = self.fixture.archive
        mirror_name = archive.name + '-MIRROR'
        location = self.fixture.location(mirror_name)
        mirror = archive.make_mirror(mirror_name, location, signed=True)
        self.assertEqual(False, archive.is_signed)
        self.assertEqual(True, mirror.is_signed)

    def test_has_listings(self):
        """Archive.has_listings works."""
        archive = self.fixture.archive
        mirror_name = archive.name + '-MIRROR'
        location = self.fixture.location(mirror_name)
        mirror = archive.make_mirror(mirror_name, location, listing=True)
        self.assertEqual(False, archive.has_listings)
        self.assertEqual(True, mirror.has_listings)


class MakeMirrorVersion(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture(tla=False)

    def _help_create_mirror(self, **kwargs):
        master = self.fixture.archive
        mirror_name = master.name + '-MIRROR'
        location = self.fixture.location(mirror_name)
        mirror = master.make_mirror(mirror_name, location, **kwargs)
        return mirror

    def test_mirror_version_baz(self):
        """The default version string says baz."""
        mirror = self._help_create_mirror()
        expected = 'Bazaar archive format 1 0'
        self.assertEqual(expected, mirror.version_string)

    def test_mirror_version_tla(self):
        """The tla option to make_mirror works."""
        mirror = self._help_create_mirror(tla=True)
        expected = 'Hackerlab arch archive directory, format version 2.'
        self.assertEqual(expected, mirror.version_string)


class MirrorArchive(framework.NewTestCase):
    fixture = fixtures.MirrorFixture()

    def test_is_mirror(self):
        """Archive.is_mirror works."""
        self.assert_(not self.fixture.archive.is_mirror)
        self.assert_(self.fixture.mirror.is_mirror)

    def test_official_name(self):
        """Archive.official_name works."""
        archive = self.fixture.archive
        self.assertEqual(archive.official_name, archive.name)
        mirror = self.fixture.mirror
        self.assertEqual(mirror.official_name, archive.name)

    def test_mirror_official_name(self):
        """Archive.make_mirror uses official name."""
        mirror2_name = self.fixture.mirror.name + '-2'
        location2 = self.fixture.location(mirror2_name)
        mirror2 = self.fixture.mirror.make_mirror(mirror2_name, location2)
        self.assertEqual(mirror2.official_name, self.fixture.archive.name)


framework.register(__name__)
