%{
/* mswatch configuration lexical analyzer */
%}

%{
#include <stdlib.h> /* atoi() */
#include <string.h> /* strdup() */
#include "parser.h"
int yyerror(char msg[]);

/* strcpy() with character escaping; eg source "\\." yields target "." */
void strcpy_escaped(char* target, const char* source)
{
	int target_i = 0, source_i = 0;
	if (!target || !source)
		return;
	for (; source[source_i] != '\0'; target_i++, source_i++)
	{
		if (source[source_i] == '\\')
			source_i++;
		target[target_i] = source[source_i];
	}
	target[target_i] = '\0';
}

//#define DEBUG_LEXER
#ifdef DEBUG_LEXER
#include <stdio.h> /* printf() */
#define DPRINTF(x...) printf(x)
#else
#define DPRINTF(x...) do { } while(0)
#endif

static int sent_eof_eol = 0;
%}

WHITESPACE    [ \t]
BASEDELAY     "base_delay"
INTERDELAY    "inter_delay"
MAXDELAY      "max_delay"
SYNC          "sync"
MAILBOXPREFIX "mailbox_prefix"
STORE         "store"
WATCH         "watch"
CBOPEN        "{"
CBCLOSE       "}"
COMMENT       "#"[^\n]*"\n"
NUM           [0-9]+
VAL           [^\\\n#\" \t]+
VALE          ([^\\\n#\" \t]|\\.)+
VALQ          \"[^\"\n#]+\"
VALQE         \"([^\"\n#]|\\\")+\"

%%
{BASEDELAY}  { DPRINTF("->base_delay\n"); return BASEDELAY; }
{INTERDELAY} { DPRINTF("->inter_delay\n"); return INTERDELAY; }
{MAXDELAY}   { DPRINTF("->max_delay\n");  return MAXDELAY; }
{SYNC}       { DPRINTF("->sync\n");       return SYNC; }
{MAILBOXPREFIX} { DPRINTF("->mailbox_prefix\n"); return MAILBOXPREFIX; }
{STORE}      { DPRINTF("->store\n");      return STORE; }
{WATCH}      { DPRINTF("->watch\n");      return WATCH; }
{CBOPEN}     { DPRINTF("->{\n");          return CBOPEN; }
{CBCLOSE}    { DPRINTF("->}\n");          return CBCLOSE; }

\n           { DPRINTF("->EOL\n");        return EOL; }
{COMMENT}    { DPRINTF("->comment \"%s\", EOL\n", yytext); return EOL; }
{WHITESPACE} { DPRINTF("->whitespace\n"); }
<<EOF>>      {
               // lex defines yyunput but may not use it.
               // "Use" it here to avoid -Werror death.
               (void) yyunput;
	
               /* return EOL at EOF */
               if (!sent_eof_eol)
			   {
			     sent_eof_eol = 1;
				 return EOL;
			   }
   			   else
			     yyterminate();
			 }

{NUM}        {
               yylval.num = atoi(yytext);
               DPRINTF("-> number %d\n", yylval);
               return NUM;
             }
{VAL}        {
               DPRINTF("-> \"%s\"\n", yytext);
               yylval.str = strdup(yytext);
               return TEXT;
             }
{VALE}       {
               strcpy_escaped(yytext, yytext);
               DPRINTF("-> \"%s\"\n", yytext);
               yylval.str = strdup(yytext);
               return TEXT;
             }
{VALQ}       {
               char* yycopy;
               yytext[yyleng - 1] = '\0'; /* change \" to \0 */
               yycopy = strdup(&yytext[1]);
               DPRINTF("-> \"%s\"\n", yycopy);
               yylval.str = yycopy;
               return TEXT;
             }
{VALQE}      {
               char* yycopy;
               yytext[yyleng - 1] = '\0'; /* change \" to \0 */
               yycopy = strdup(&yytext[1]);
               strcpy_escaped(yycopy, yycopy);
               DPRINTF("-> \"%s\"\n", yycopy);
               yylval.str = yycopy;
               return TEXT;
             }

.            {
               char* c;
               fprintf(stderr, "Unrecognized characters: ");
               c = yytext;
               while (*c)
               {
                 fprintf(stderr, "'%c' (%d)", *c, *c);
				 if (*c == '\t') fprintf(stderr, " [is tab]");
                 c++;
               }
               fprintf(stderr, ". Ignoring.\n");
             }
%%
