/***************************************************************************
 *   Weather module for Kadu communicator                                  *
 *   2004 Micha Maek (bysk)                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "autodownloader.h"
#include "weather_global.h"

#include "config_file.h"
#include "modules/hints/hint_manager.h"
#include "debug.h"

#include <qtimer.h>

AutoDownloader::AutoDownloader( QObject* parent )
:
	QObject( parent )
{
	autoDownloadTimer_ = new QTimer(this);
	
	connect(autoDownloadTimer_, SIGNAL(timeout()), this, SLOT(autoDownload()));
	connect(&downloader_, SIGNAL(finished()), this, SLOT(autoDownloadingFinished()));
	
	lastAuto_ = config_file.readBoolEntry("Weather", "bAuto");
	lastHints_ = config_file.readBoolEntry("Weather", "bHint");
	lastDesc_ = config_file.readBoolEntry("Weather", "bDescription");
	
	if( WeatherGlobal::KEEP_FORECAST > 0 && lastAuto_ )
	{
		autoDownloadTimer_->start( WeatherGlobal::KEEP_FORECAST*60*60*1000, false );
		autoDownload();
	}
}

void AutoDownloader::parametersChanged()
{
	kdebugf();
	
	bool bAuto = config_file.readBoolEntry("Weather", "bAuto");
	
	if( !bAuto && autoDownloadTimer_->isActive())
		autoDownloadTimer_->stop();
	else if( bAuto && ((bAuto  && !lastAuto_) ||
	         (config_file.readBoolEntry("Weather", "bHint") &&  !lastHints_) ||
	         (config_file.readBoolEntry("Weather", "bDescription") && !lastDesc_)) )
	{
		autoDownloadTimer_->start(WeatherGlobal::KEEP_FORECAST*60*60*1000, false);
		autoDownload();
	}
	
	lastAuto_ = bAuto;
	lastHints_ = config_file.readBoolEntry("Weather", "bHint");
	lastDesc_ = config_file.readBoolEntry("Weather", "bDescription");
	
	kdebugf2();
}

void AutoDownloader::autoDownload()
{
	kdebugf();
	
	QString server = config_file.readEntry( "Weather", "MyServer" );
	QString cityId = config_file.readEntry( "Weather", "MyCityId" );
	
	if( server.isEmpty() || cityId.isEmpty() )
		return;
	
	downloader_.downloadForecast( server, cityId );
	
	kdebugf2();
}

void AutoDownloader::autoDownloadingFinished()
{
	kdebugf();
	
	if( WeatherGlobal::KEEP_FORECAST > 0 && config_file.readBoolEntry("Weather", "bAuto"))
	{
		if(config_file.readBoolEntry("Weather", "bHint"))
		{
			const ForecastDay& day = downloader_.getForecast().Days[config_file.readNumEntry("Weather", "HintDay")];
			hint_manager->addHint(parse(day, config_file.readEntry("Weather", "HintText")),
				QPixmap(day["Icon"]),
				config_file.readFontEntry("Weather", "HintFont"),
				config_file.readColorEntry("Weather", "FontColor"),
				config_file.readColorEntry("Weather", "BackgroundColor"),
				config_file.readNumEntry("Weather", "HintTimeout"));
		}
		
		if(config_file.readBoolEntry("Weather", "bDescription") && !gadu->status().isOffline())
		{
			const ForecastDay& day = downloader_.getForecast().Days[config_file.readNumEntry("Weather", "DescriptionDay")];
			QString newStatusText = parse(day, config_file.readEntry("Weather", "DescriptionText"));
			if(newStatusText != gadu->status().description() || !gadu->status().hasDescription())
				gadu->status().setDescription(newStatusText);
		}
	}
	
	kdebugf2();
}

QString AutoDownloader::parse(const ForecastDay& day, QString str)
{
	str.replace("%l", downloader_.getForecast().LocationName);
	str.replace("%T", downloader_.getForecast().loadTime.toString( "h:mm" ));
	str.replace("%d", day["Name"]);
	str.replace("%t", day["Temperature"]);
	str.replace("%o", day["Description"]);
	str.replace("%w", day["Wind speed"]);
	str.replace("%p", day["Pressure"]);
	str.replace("&deg;","");
	str.replace("&nbsp;"," ");
	return str;
}
