/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <xmms/xmmsctrl.h>

#include "debug.h"
#include "kadu.h"
#include "message_box.h"
#include "userlist.h"
#include "config_dialog.h"
#include "status.h"
#include "action.h"
#include "../notify/notify.h"
#include "custom_input.h"
#include "icons_manager.h"
#include "chat_manager.h"

#include "xmms.h"

#include <qtimer.h>
#include <qtooltip.h>
#include <qpopupmenu.h>

#define MODULE_XMMS_VERSION 1.33

// For CP1250->ISO8859-2 converter
const char CODE_CP1250[]	= {0xb9,0x9c,0x9f,0xa5,0x8c,0x8f};
const char CODE_ISO8859_2[] = {0xb1,0xb6,0xbc,0xa1,0xa6,0xac};

// For ID3 tags signatures cutter
const char DEFAULT_SIGNATURES[] = "! WWW.POLSKIE-MP3.TK ! \n! www.polskie-mp3.tk ! ";

// The main XMMS module object
Xmms *xmms_session;

// Kadu initializing functions
extern "C" int xmms_init()
{
	xmms_session = new Xmms();

	return 0;
}

extern "C" void xmms_close()
{
	delete xmms_session;
}

// Implementation of Xmms class

Xmms::Xmms()
{
	kdebugf();

	// XMMS menus in chats
	menu = new QPopupMenu();
	popups[0] = menu->insertItem(tr("Put formated string"), this, SLOT(putSongTitle(int)));
	popups[1] = menu->insertItem(tr("Put song title"), this, SLOT(putSongTitle(int)));
	popups[2] = menu->insertItem(tr("Put song file name"), this, SLOT(putSongTitle(int)));
	popups[3] = menu->insertItem(tr("Put all playlist titles"), this, SLOT(putPlayList(int)));
	popups[4] = menu->insertItem(tr("Put all playlist files"), this, SLOT(putPlayList(int)));

	// Config file
	config = new ConfigFile(ggPath(QString("/kadu-xmms.conf")));

	// Title checking timer
	timer = new QTimer();
	connect(timer, SIGNAL(timeout()), this, SLOT(checkTitle()));
	timer->start(1000);

	// Monitor of creating chats
	connect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), this, SLOT(chatCreated(const UserGroup*)));
	connect(chat_manager, SIGNAL(chatDestroying(const UserGroup*)), this, SLOT(chatDestroying(const UserGroup*)));
	ChatList::ConstIterator it;
	for ( it = chat_manager->chats().begin(); it != chat_manager->chats().end(); it++ )
		handleCreatedChat(*it);

	// XMMS statuses menu item
	bool menuPos = config->readBoolEntry("XMMS", "dockMenu", false);
	if (menuPos)
		popups[5] = dockMenu->insertItem(tr("Enable XMMS statuses"), this, SLOT(toggleStatuses(int)), 0, -1, 10);
	else
		popups[5] = kadu->mainMenu()->insertItem(tr("Enable XMMS statuses"), this, SLOT(toggleStatuses(int)), 0, -1, 0);

	// Checking for %0 in description
	connect(kadu, SIGNAL(changingStatus(UserStatus&, bool&)), this, SLOT(checkStatusDesc(UserStatus&, bool&)));

	// Configuration
	ConfigDialog::addTab("XMMS", dataPath("kadu/modules/data/xmms/xmms.png"));

	ConfigDialog::registerSlotOnCreateTab("XMMS" ,this, SLOT(onCreateConfig()));
	ConfigDialog::registerSlotOnApplyTab("XMMS" ,this, SLOT(onUpdateConfig()));
	ConfigDialog::registerSlotOnCloseTab("XMMS" ,this, SLOT(onDestroyConfig()));

		ConfigDialog::addLineEdit(config, "XMMS", "XMMS", QT_TRANSLATE_NOOP("@default", "XMMS message format for chat windows"), "chatString", "XMMS: %t [%c / %l]",
			QT_TRANSLATE_NOOP("@default", "%t - song title, %f - file name, %l - song length (MM:SS),<br>%c - current song position (MM:SS), %p - percents of played song"));
		ConfigDialog::addLineEdit(config, "XMMS", "XMMS", QT_TRANSLATE_NOOP("@default", "XMMS status format"), "statusString", "-=[%t]=-",
			"%t - song title, %f - file name, %l - song length (MM:SS),<br>%c - current song position (MM:SS), %p - percents of played song");
		ConfigDialog::addLineEdit(config, "XMMS", "XMMS", QT_TRANSLATE_NOOP("@default", "%xmms% status description tag format"), "statusTagString", "%t",
			QT_TRANSLATE_NOOP("@default", "Here specify format which will replace '%xmms%' tag when you will use it in description while setting status."));

		ConfigDialog::addVGroupBox("XMMS", "XMMS", QT_TRANSLATE_NOOP("@default", "Pseudo-OSD"));
			ConfigDialog::addCheckBox(config, "XMMS", "Pseudo-OSD", QT_TRANSLATE_NOOP("@default", "Enable pseudo-OSD"), "osd", TRUE,
				QT_TRANSLATE_NOOP("@default", "Enables pseudo-OSD done by Kadu hints. Shows currently played title in hint."));
			ConfigDialog::addSelectFont(config, "XMMS", "Pseudo-OSD", QT_TRANSLATE_NOOP("@default", "OSD font"),
				"OsdFont", "helvetica");
			ConfigDialog::addColorButton(config, "XMMS", "Pseudo-OSD", QT_TRANSLATE_NOOP("@default", "OSD font color"), "OsdFgColor",
				QColor("yellow"));
			ConfigDialog::addColorButton(config, "XMMS", "Pseudo-OSD", QT_TRANSLATE_NOOP("@default", "OSD background color"), "OsdBgColor",
				QColor("#000099"));
			ConfigDialog::addSpinBox(config, "XMMS", "Pseudo-OSD", QT_TRANSLATE_NOOP("@default", "OSD timeout"), "OsdTimeout",
				1, 20, 1, 3, QT_TRANSLATE_NOOP("@default", "Time after which OSD disappears (in seconds)"));

		ConfigDialog::addVGroupBox("XMMS", "XMMS", QT_TRANSLATE_NOOP("@default","Cut signatures"));
			ConfigDialog::addCheckBox(config, "XMMS", "Cut signatures", QT_TRANSLATE_NOOP("@default", "Enable signatures cutting"), "signature", TRUE, QT_TRANSLATE_NOOP("@default", "Shell I cut these craps?"));
			ConfigDialog::addTextEdit(config, "XMMS", "Cut signatures", QT_TRANSLATE_NOOP("@default", "Signatures:"), "signatures", DEFAULT_SIGNATURES);

		ConfigDialog::addCheckBox(config, "XMMS", "XMMS", QT_TRANSLATE_NOOP("@default", "Enable XMMS keyboard shortcuts in chat window"), "chatShortcuts", TRUE,
			QT_TRANSLATE_NOOP("@default", "With this option enabled you'll be able to control\nyour XMMS in chat window by keyboard shortcuts:\nWin+ Enter/Backspace/Left/Right/Up/Down."));

		ConfigDialog::addCheckBox(config, "XMMS", "XMMS", QT_TRANSLATE_NOOP("@default", "Put statuses switch to dock menu"), "dockMenu", false,
			QT_TRANSLATE_NOOP("@default","Enable this to put \"XMMS statuses switch\"\ninto docked icon menu."));

	// Initial values of some object variables
	winKeyPressed = FALSE;

	// Button
	Action *action = new Action(icons_manager->loadIcon("Unmute"), "XMMS", "xmms_button", Action::TypeChat);
	connect(action, SIGNAL(activated(const UserGroup*, const QWidget*, bool)),
		this, SLOT(xmmsActionActivated(const UserGroup*, const QWidget*)));
	KaduActions.insert("xmms_button", action);
}

Xmms::~Xmms()
{
	kdebugf();

	KaduActions.remove("xmms_button");

	// Removing configuration
	ConfigDialog::removeControl("XMMS", "Put statuses switch to dock menu");
	ConfigDialog::removeControl("XMMS", "Enable XMMS keyboard shortcuts in chat window");

	ConfigDialog::removeControl("XMMS", "Signatures:");
	ConfigDialog::removeControl("XMMS", "Enable signatures cutting");
	ConfigDialog::removeControl("XMMS", "Cut signatures");

	ConfigDialog::removeControl("XMMS", "OSD timeout");
	ConfigDialog::removeControl("XMMS", "OSD font color");
	ConfigDialog::removeControl("XMMS", "OSD background color");
	ConfigDialog::removeControl("XMMS", "OSD font");
	ConfigDialog::removeControl("XMMS", "Enable pseudo-OSD");
	ConfigDialog::removeControl("XMMS", "Pseudo-OSD");

	ConfigDialog::removeControl("XMMS", "%xmms% status description tag format");
	ConfigDialog::removeControl("XMMS", "XMMS message format for chat windows");
	ConfigDialog::removeControl("XMMS", "XMMS status format");

	ConfigDialog::unregisterSlotOnCreateTab("XMMS" ,this, SLOT(onCreateConfig()));
	ConfigDialog::unregisterSlotOnApplyTab("XMMS" ,this, SLOT(onUpdateConfig()));
	ConfigDialog::unregisterSlotOnCloseTab("XMMS" ,this, SLOT(onDestroyConfig()));

	ConfigDialog::removeTab("XMMS");
	// Stop timer for checking titles
	timer->stop();

	// Disconnect all slots
	disconnect(timer, SIGNAL(timeout()), this, SLOT(checkTitle()));
	disconnect(kadu, SIGNAL(changingStatus(UserStatus&, bool&)), this, SLOT(checkStatusDesc(UserStatus&, bool&)));
	disconnect(chat_manager, SIGNAL(chatCreated(const UserGroup*)), this, SLOT(chatCreated(const UserGroup*)));
	disconnect(chat_manager, SIGNAL(chatDestroying(const UserGroup*)), this, SLOT(chatDestroying(const UserGroup*)));
	ChatList::ConstIterator it;
	for ( it = chat_manager->chats().begin(); it != chat_manager->chats().end(); it++ )
		handleDestroyingChat(*it);

	// Saving configuration
	config->sync();

	// Delete own objects
	delete menu;
	delete timer;
	delete config;

	// Remove menu item (statuses)
	int idx = dockMenu->indexOf(popups[5]);
	if (idx == -1)
		kadu->mainMenu()->removeItem(popups[5]);
	else
		dockMenu->removeItem(popups[5]);
}

void Xmms::xmmsActionActivated(const UserGroup* grp, const QWidget* source)
{
	kdebugf();

	menu->popup(source->mapToGlobal(QPoint(0,20)));
}

void Xmms::chatCreated(const UserGroup* grp)
{
	kdebugf();
	Chat* chat = chat_manager->findChat(grp);
	handleCreatedChat(chat);
}

void Xmms::chatDestroying(const UserGroup* grp)
{
	kdebugf();
	Chat* chat = chat_manager->findChat(grp);
	handleDestroyingChat(chat);
}

void Xmms::handleCreatedChat(Chat* chat)
{
	kdebugf();

	// Key bindings
	connect(chat->edit(), SIGNAL(keyPressed(QKeyEvent*, CustomInput*, bool&)), this, SLOT(chatKeyPressed(QKeyEvent*, CustomInput*, bool&)));
	connect(chat->edit(), SIGNAL(keyReleased(QKeyEvent*, CustomInput*, bool&)), this, SLOT(chatKeyReleased(QKeyEvent*, CustomInput*, bool&)));
}

void Xmms::handleDestroyingChat(Chat* chat)
{
	kdebugf();

	// Key bindings
	disconnect(chat->edit(), SIGNAL(keyPressed(QKeyEvent*, CustomInput*, bool&)), this, SLOT(chatKeyPressed(QKeyEvent*, CustomInput*, bool&)));
	disconnect(chat->edit(), SIGNAL(keyReleased(QKeyEvent*, CustomInput*, bool&)), this, SLOT(chatKeyReleased(QKeyEvent*, CustomInput*, bool&)));
}

void Xmms::chatKeyPressed(QKeyEvent* e, CustomInput* k, bool &handled)
{
	kdebugf();

	if ( !config->readBoolEntry("XMMS","chatShortcuts",TRUE) )
		return;

	if (e->key() == Qt::Key_Super_L)

		// We want to handle LeftWinKey pressed state
		winKeyPressed = TRUE;

	else if (!winKeyPressed || !xmms_remote_is_running(0))

		// If LeftWinKey isn't pressed then break function.
		// If XMMS isn't running then break function too.
		return;

	handled = true;
	switch (e->key())
	{
		case Qt::Key_Left:
			prevTrack();
			break;
		case Qt::Key_Right:
			nextTrack();
			break;
		case Qt::Key_Up:
			incrVolume();
			break;
		case Qt::Key_Down:
			decrVolume();
			break;
		case Qt::Key_Return:
		case Qt::Key_Enter:
			if (isPlaying())
				pause();
			else
				play();
			break;
		case Qt::Key_Backspace:
			stop();
			break;
		default:
			handled = false;
	}
}

void Xmms::chatKeyReleased(QKeyEvent* e, CustomInput* k, bool &handled)
{
	if (e->key() == Qt::Key_Super_L)

		// We want to handle LeftWinKey pressed state
		winKeyPressed = FALSE;
}

void Xmms::putSongTitle(int ident)
{
	kdebugf();
	if (!xmms_remote_is_running(0))
	{
		MessageBox::msg(tr("XMMS isn't running!"));
		return;
	}
	Chat* chat = getCurrentChat();
	QString title;

	// This code tells us which item from XMMS menu button was selected
	int id = 0;
	for ( int i = 0; i < 3; i++ )
	{
		if (popups[i] == ident)
		{
			id = i;
			break;
		}
	}

	// Sets title variable to proper value
	switch (id)
	{
		case 0:
			title = parse(config->readEntry("XMMS", "chatString"));
			break;
		case 1:
			title = getTitle();
			break;
		case 2:
			title = getFile();
			break;
	}

	int x, y;

	HtmlDocument doc, doc2, doc3;
	chat->edit()->getCursorPosition(&y, &x);
	chat->edit()->insertAt(title, y, x); // Here we paste the title
	doc.parseHtml(chat->edit()->text());

	// Parsing inserted element as HTML
	for ( int i = 0; i < doc.countElements(); i++ )
	{
		if (i == 7)
		{
			doc2.parseHtml(doc.elementText(i));
			for ( int j = 0; j < doc2.countElements(); j++ )
			{
				if (doc2.isTagElement(j))
					doc3.addTag(doc2.elementText(j));
				else
					doc3.addText(doc2.elementText(j));
			}
		}
		else
		{
			if (doc.isTagElement(i))
				doc3.addTag(doc.elementText(i));
			else
				doc3.addText(doc.elementText(i));
		}
	}
	chat->edit()->setText(doc3.generateHtml());
	chat->edit()->moveCursor(QTextEdit::MoveEnd, FALSE);
}

void Xmms::putPlayList(int ident)
{
	kdebugf();
	if (!xmms_remote_is_running(0))
	{
		MessageBox::msg(tr("XMMS isn't running!"));
		return;
	}

	Chat* chat = getCurrentChat();
	int id = 0, x, y;
	QString str;
	QStringList list;

	// This code tells us which item from XMMS menu button was selected
	for ( int i = 3; i < 5; i++ )
	{
		if (popups[i] == ident)
		{
			id = i;
			break;
		}
	}

	uint lgt = getPlayListLength();
	if (lgt == 0)
		return;

	// Creating list of elements to paste/send
	switch (id)
	{
		case 3:
		{
			list = getPlayListTitles();
			break;
		}
		case 4:
		{
			list = getPlayListFiles();
			break;
		}
	}


	// Calculating playlist length as characters
	uint chars = 0, emptyEntries = 0;
	for ( uint cnt = 0; cnt < lgt; cnt++ )
	{
		if (list[cnt].length() == 0)
			emptyEntries++;

		chars += list[cnt].length();
		chars += formatLength(getLength(cnt)).length();
		chars += QString::number(cnt).length();
		chars += 12; /*
						 The white-space betwean entry and length (1),
						 dot and whitespace after number of track on the begin of line (2)
						 left and right brace for track length (2),
						 and a </p><p> tags as a new line (7).
					 */

		if (chars >= 2000)
			break;
	}
	chars += 135; // Additional chars added by Kadu chat input.

	if (emptyEntries > (lgt/10))
	{
		if (!MessageBox::ask(tr("More than 1/10 of titles you're trying to send are empty.<br>Perhaps XMMS havn't read all ID3 tags yet, give its some more time.<br>Do you want to send playlist anyway?")) )
			return;
	}
	if (chars >= 2000)
	{
		if (!MessageBox::ask(tr("You're trying to send %1 entries of XMMS playlist.<br>It will be splitted and sent in few messages<br>Are you sure to do that?")
			.arg(QString::number(lgt))) )
			return;
	}

	chat->edit()->moveCursor(QTextEdit::MoveEnd, FALSE);
	for ( uint cnt = 0; cnt < lgt; cnt++ )
	{
		str = QString::number((cnt+1)) + ". " + list[cnt] + " (" + formatLength(getLength(cnt)) + ")\n";
		if ((chat->edit()->text().length() + str.length()) >= 2000)
			chat->sendMessage();

		chat->edit()->getCursorPosition(&y, &x);
		chat->edit()->insertAt(str, y, x); // Here we paste XMMS playlist item
		chat->edit()->moveCursor(QTextEdit::MoveEnd, FALSE);
	}
	chat->sendMessage();
}

QStringList Xmms::getPlayListTitles()
{
	kdebugf();
	QStringList list;

	// Firstly, how many items are in playlist?
	uint lgt = getPlayListLength();
	if (lgt == 0)
		return list;

	// Secondary, get all titles
	for ( uint cnt = 0; cnt < lgt; cnt++ )
		list.append(getTitle(cnt));

	return list;
}

QStringList Xmms::getPlayListFiles()
{
	kdebugf();
	QStringList list;

	// Firstly, how many items are in playlist?
	uint lgt = getPlayListLength();
	if (lgt == 0)
		return list;

	// Secondary, get all file names
	for ( uint cnt = 0; cnt < lgt; cnt++ )
		list.append(getFile(cnt));

	return list;
}

uint Xmms::getPlayListLength()
{
	kdebugf();
	return xmms_remote_get_playlist_length(0);
}

QString Xmms::parse(QString str)
{
	/*
		Code of this function is partialy
		borrown from Kadu. Thanks to Kadu Team! :)
	*/
	kdebugf();
	uint sl = str.length();
	QString r;

	for ( uint i = 0; i < sl; i++ )
	{
		while (str[i] != '%' && i < sl)
		{
			r += str[i];
			i++;
		}

		if (str[i] == '%')
		{
			i++;
			switch(str[i].latin1())
			{
				case 't':
				{
					r += getTitle();
					break;
				}
				case 'f':
				{
					r += getFile();
					break;
				}
				case 'l':
				{
					r += formatLength(getLength());
					break;
				}
				case 'c':
				{
					r += formatLength(getCurrentPos());
					break;
				}
				case 'p':
				{
					QString tmp;
					int perc = getCurrentPos() / getLength();
					tmp = QString::number(perc) + "%";
					r += tmp;
					break;
				}
				default:
					r += str[i];
			}
		}
	}
	return r;
}

Chat* Xmms::getCurrentChat()
{
	kdebugf();

	// Getting all chat windows
	ChatList cs = chat_manager->chats();

	// Now for each chat window we check,
	// if it's an active one.
	uint i;
	for ( i = 0; i < cs.count(); i++ )
	{
		if (cs[i]->isActiveWindow())
		{
			break;
		}
	}
	if (i == cs.count())
		return 0;

	return cs[i];
}

QString Xmms::getTitle(int position)
{
	kdebugf();
	if (!xmms_remote_is_running(0))
		return "";

	// Do we want current track title, or given track title?
	if (position == -1)
		position = pos();

	// Gets the title and converts CP1250 to ISO8859-2 on the fly
	QString title(convert(xmms_remote_get_playlist_title(0, position)));

	// Now, if we want to cut nasty signatures, we do it!
	if (config->readBoolEntry("XMMS", "signature", TRUE))
	{
		QStringList sigList(QStringList::split('\n', config->readEntry("XMMS", "signatures", DEFAULT_SIGNATURES)));
		for (unsigned int i = 0; i < sigList.count(); i++)
			title.remove(sigList[i]);
	}
	return title;
}

QString Xmms::getFile(int position)
{
	kdebugf();

	// Do we want current track file name, or given track file name?
	if (position == -1)
		position = pos();

	// Converts CP1250 to ISO8859-2 on the fly while returning file name
	return convert(xmms_remote_get_playlist_file(0, position));
}

int Xmms::getLength(int position)
{
	kdebugf();

	// Do we want current track file name, or given track file name?
	if (position == -1)
		position = pos();

	return xmms_remote_get_playlist_time(0, position);
}

QString Xmms::formatLength(int length)
{
	kdebugf();
	QString ms;
	int lgt = length / 1000, m, s;
	m = lgt / 60;
	s = lgt % 60;
	ms = QString::number(m) + ":";
	if (s < 10)
		ms += "0";

	ms += QString::number(s);

	return ms;
}

int Xmms::getCurrentPos()
{
	kdebugf();
	return xmms_remote_get_output_time(0);
}

int Xmms::pos()
{
	return xmms_remote_get_playlist_pos(0);
}

void Xmms::toggleStatuses(int id)
{
	kdebugf();

	bool enabled;
	QPopupMenu* menu;
	int idx = dockMenu->indexOf(popups[5]);
	if (idx == -1)
		menu = kadu->mainMenu();
	else
		menu = dockMenu;

	enabled = menu->isItemChecked(popups[5]);

	if (!xmms_remote_is_running(0) && !enabled)
	{
		MessageBox::msg(tr("XMMS isn't running!"));
		return;
	}

	menu->setItemChecked(popups[5], !enabled);
}

void Xmms::checkTitle()
{
	QString title = getTitle();
	int pos = getCurrentPos();

	// If OSD is enabled and current track position is betwean 0 and 1000 ms, then shows OSD
	if (config->readBoolEntry("XMMS", "osd", TRUE) && pos < 1000 && pos > 0)
		putTitleHint(title);

	QPopupMenu* menu;
	int idx = dockMenu->indexOf(popups[5]);
	if (idx == -1)
		menu = kadu->mainMenu();
	else
		menu = dockMenu;

	if (!gadu->status().isOffline() && menu->isItemChecked(popups[5]))
	{
		if (title != currentTitle || !gadu->status().hasDescription())
		{
			currentTitle = title;
			gadu->status().setDescription(parse(config->readEntry("XMMS", "statusString")));
		}
	}
}

void Xmms::checkStatusDesc(UserStatus& status, bool &stop)
{
	QString str = config->readEntry("XMMS", "statusTagString");
	if (status.description().find("%xmms%") > -1)
	{
		QString newDesc = status.description().replace("%xmms%", parse(str));
		status.setDescription(newDesc);
		gadu->status().setStatus(status);
		stop = TRUE;
	}
}

void Xmms::putTitleHint(QString title)
{
	kdebugf();
	QColor fg("yellow"), bg("#000099");
	QFont font("helvetica", 14, QFont::Bold);
	QMap<QString, QVariant> parms;
	parms["Pixmap"] = icons_manager->loadIcon("Unmute");
	parms["Foreground color"] = config->readColorEntry("XMMS", "OsdFgColor", &fg);
	parms["Background color"] = config->readColorEntry("XMMS", "OsdBgColor", &bg);
	parms["Font"] = config->readFontEntry("XMMS", "OsdFont", &font);
	parms["Timeout"] = config->readNumEntry("XMMS", "OsdTimeout", 3);
	parms["ShowSource"] = (bool)FALSE;

	notify->emitMessage("XMMS", "Hints", title, &parms);
}

void Xmms::nextTrack()
{
	kdebugf();
	xmms_remote_playlist_next(0);
}

void Xmms::prevTrack()
{
	kdebugf();
	xmms_remote_playlist_prev(0);
}

void Xmms::play()
{
	kdebugf();
	xmms_remote_play(0);
}

void Xmms::stop()
{
	kdebugf();
	xmms_remote_stop(0);
}

void Xmms::pause()
{
	kdebugf();
	xmms_remote_pause(0);
}

void Xmms::setVolume(int vol)
{
	kdebugf();
	xmms_remote_set_main_volume(0, vol);
}

void Xmms::incrVolume()
{
	kdebugf();
	int vol = xmms_remote_get_main_volume(0);
	vol += 2;
	if (vol > 100)
		vol = 100;

	setVolume(vol);
}

void Xmms::decrVolume()
{
	kdebugf();
	int vol = xmms_remote_get_main_volume(0);
	vol -= 2;
	if (vol < 0)
		vol = 0;

	setVolume(vol);
}

bool Xmms::isPlaying()
{
	kdebugf();
	return xmms_remote_is_playing(0);
}

QString Xmms::convert(QString str)
{
	for (unsigned int i = 0; i < sizeof(CODE_CP1250); i++)
	{
		str.replace(CODE_CP1250[i],CODE_ISO8859_2[i]);
	}
	return QString::fromLocal8Bit(str,-1);
}

void Xmms::onCreateConfig()
{
	kdebugf();
	modules_manager->moduleIncUsageCount("xmms");
}

void Xmms::onUpdateConfig()
{
	kdebugf();

	// Statuses switch
	bool enabled;

	int idx = dockMenu->indexOf(popups[5]);
	if (idx == -1)
	{
		enabled = kadu->mainMenu()->isItemChecked(popups[5]);
		kadu->mainMenu()->removeItem(popups[5]);
	}
	else
	{
		enabled = dockMenu->isItemChecked(popups[5]);
		dockMenu->removeItem(popups[5]);
	}

	bool menuPos = config->readBoolEntry("XMMS", "dockMenu", false);
	if (menuPos)
	{
		popups[5] = dockMenu->insertItem(tr("Enable XMMS statuses"), this, SLOT(toggleStatuses(int)), 0, -1, 10);
		dockMenu->setItemChecked(popups[5], enabled);
	}
	else
	{
		popups[5] = kadu->mainMenu()->insertItem(tr("Enable XMMS statuses"), this, SLOT(toggleStatuses(int)), 0, -1, 0);
		kadu->mainMenu()->setItemChecked(popups[5], enabled);
	}

	// Chat windows XMMS button
	ChatList cs = chat_manager->chats();
	for (uint i = 0; i < cs.count(); i++)
	{
		UserListElements u = cs[i]->users()->toUserListElements();
		if (u.count() > 1)
			continue;
		QValueList<ToolButton*> buttons = KaduActions["xmms_button"]->toolButtonsForUserListElements(u);
		for (QValueList<ToolButton*>::iterator i = buttons.begin(); i != buttons.end(); i++)
		{
			(*i)->setShown(config->readBoolEntry("XMMS", "chatButton", true));
		}
	}

	config->sync();
}

void Xmms::onDestroyConfig()
{
	kdebugf();
	onUpdateConfig();
	modules_manager->moduleDecUsageCount("xmms");
}

