/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * autor
 * Micha Podsiadlik
 * michal at gov.one.pl
 *
 * obsluga maildir:
 * Konrad Klimaszewski
 * klimasz at if.pw.edu.pl>
 */

#include <qlineedit.h>
#include <qprocess.h>
#include <qfontdialog.h> 
#include <qfiledialog.h>
#include <qvgroupbox.h>
#include <qlistbox.h>

#include <debug.h>
#include <config_dialog.h>
#include <misc.h>
#ifdef EXTERNAL
#include <modules/hint_manager.h>
#else
#include "../hints/hint_manager.h"
#endif
#include <modules.h>
#include <icons_manager.h>

#include <stdlib.h>

#include "mail.h"
#include "accountdialog.h"

// dziwne nazewnictwo :|
#define addHintError(x) connectionError(NULL,x)

extern "C" int mail_init()
{
	mail=new Mail();
	return 0;
}

extern "C" void mail_close()
{
	delete mail;
}

Mail::Mail() : QObject()
{
	timer=new QTimer(this);
	connect(timer       , SIGNAL(timeout())  , SLOT(checkmail()));

	config=new ConfigFile(ggPath("mail.conf"));

	SSL_load_error_strings();
	SSL_library_init();

	for(int i=0;;i++)
	{
		QString config_section;
		config_section.sprintf("Account_%i", i);
		if(config->readEntry(config_section, "Name").length()){
			Pop3Proto* acc=new Pop3Proto(
					config->readEntry(config_section, "Name"),
					config->readEntry(config_section, "Server"),
					config->readNumEntry(config_section, "ServerPort", 110),
					config->readEntry(config_section, "User"),
					pwHash(config->readEntry(config_section, "Password", "")),
					(SecureType)config->readNumEntry(config_section, "Encryption"));
			kdebugm(KDEBUG_INFO,
			"Account %i: Name: %s, Server: %s, ServerPort: %i, User: %s, Encryption: %i\n", i,
					config->readEntry(config_section, "Name").latin1(),
					config->readEntry(config_section, "Server").latin1(),
					config->readNumEntry(config_section, "ServerPort", 110),
					config->readEntry(config_section, "User").latin1(),
					config->readNumEntry(config_section, "Encryption"));
			
			connect(acc      , SIGNAL(done(int,int,int,QString)),
					SLOT(printstat(int,int,int,QString)));
			acc->setLastmails(config->readNumEntry(config_section, "Last"));
			accounts.append(acc);
		}
		else
			break;
	}

	ConfigDialog::registerSlotOnCreateTab("Mail", this, SLOT(onCreateConfig()));
	ConfigDialog::registerSlotOnCloseTab("Mail", this, SLOT(onCloseConfig()));
	ConfigDialog::registerSlotOnApplyTab("Mail", this, SLOT(onApplyConfig()));
		

	ConfigDialog::addTab(QT_TRANSLATE_NOOP("@default", "Mail"), "MailTab");

	ConfigDialog::addVGroupBox("Mail", "Mail",
			QT_TRANSLATE_NOOP("@default", "Checking options"));
	ConfigDialog::addLineEdit(config, "Mail", "Checking options",
			QT_TRANSLATE_NOOP("@default", "Interval in seconds"), "Interval", "60");
	ConfigDialog::addLineEdit(config, "Mail", "Checking options",
			QT_TRANSLATE_NOOP("@default", "Message format"), "Format",
			tr("You have %n new mail(s) on %a and total you have %t mail(s) witch total size %s"),
			QT_TRANSLATE_NOOP("@default",
			"\%n new messages\n"
			"\%t total on server\n"
			"\%s total size\n"
			"\%a account"));
	
	ConfigDialog::addCheckBox(config, "Mail", "Checking options",
			QT_TRANSLATE_NOOP("@default", "Run mail client"), "RunClient");
	ConfigDialog::addLineEdit(config, "Mail", "Checking options",
			QT_TRANSLATE_NOOP("@default", "Mail client"), "MailClient", "kmail");
	ConfigDialog::addLineEdit(config, "Mail", "Checking options",
			QT_TRANSLATE_NOOP("@default", "Hint timeout"), "HintTimeout", "10");

	ConfigDialog::addHGroupBox("Mail", "Mail",
			QT_TRANSLATE_NOOP("@default", "Accounts"));
	ConfigDialog::addListBox("Mail", "Accounts", "account");
        ConfigDialog::addVBox("Mail", "Accounts", "buttons");
	ConfigDialog::addPushButton("Mail", "buttons", QT_TRANSLATE_NOOP("@default", "Add"),
			"AddSelectPathDialogButton");
	ConfigDialog::addPushButton("Mail", "buttons", QT_TRANSLATE_NOOP("@default", "Remove"),
			"RemoveSelectPathDialogButton");
	ConfigDialog::addPushButton("Mail", "buttons", QT_TRANSLATE_NOOP("@default", "Edit"),
			"ChangeSelectPathDialogButton");
	

	ConfigDialog::addVGroupBox("Mail", "Mail",
			QT_TRANSLATE_NOOP("@default", "Maildir"));
	ConfigDialog::addCheckBox(config, "Mail", "Maildir",
			QT_TRANSLATE_NOOP("@default", "Check mail in local Maildir"), "LocalMaildir");
	ConfigDialog::addHBox("Mail", "Maildir", "Maildir1");
	ConfigDialog::addLineEdit(config, "Mail", "Maildir1",
			QT_TRANSLATE_NOOP("@default", "Maildir path"), "MaildirPath", "~/Maildir");
	ConfigDialog::addPushButton("Mail", "Maildir1", "...");

	ConfigDialog::addVGroupBox("Mail", "Mail",
			QT_TRANSLATE_NOOP("@default", "Colors"));

	ConfigDialog::addColorButton(config, "Mail", "Colors", 
			QT_TRANSLATE_NOOP("@default", "Font color"),
			"HintColor", QColor("black"));
	
	ConfigDialog::addColorButton(config, "Mail", "Colors", 
			QT_TRANSLATE_NOOP("@default", "Background color"),
			"HintBackground", QColor("light gray"));
	
	ConfigDialog::addHGroupBox("Mail", "Mail",
			QT_TRANSLATE_NOOP("@default", "Font"));
	ConfigDialog::addLabel("Mail", "Font", "FontSel");
	ConfigDialog::addPushButton("Mail", "Font",
			QT_TRANSLATE_NOOP("@default", "Choose"));

	ConfigDialog::connectSlot("Mail", "Choose", SIGNAL(clicked()), this, SLOT(onFontChange()));
	
	ConfigDialog::connectSlot("Mail", "Run mail client",
			SIGNAL(toggled(bool)),
			this, SLOT(mailclienttoolged(bool)));
	ConfigDialog::connectSlot("Mail", "Check mail in local Maildir",
			SIGNAL(toggled(bool)),
			this, SLOT(maildirtogled(bool)));
	
	ConfigDialog::connectSlot("Mail", "...", SIGNAL(clicked()),
			this, SLOT(onSelectMaildir()));

	ConfigDialog::connectSlot("Mail", "Add", SIGNAL(clicked()),
			this, SLOT(onAddButton()));
	ConfigDialog::connectSlot("Mail", "Remove", SIGNAL(clicked()),
			this, SLOT(onRemoveButton()));
	ConfigDialog::connectSlot("Mail", "Edit", SIGNAL(clicked()),
			this, SLOT(onEditButton()));

	selectedfont=config->readFontEntry("Mail", "Font", new QFont("Luxi Sant"));
	timer->start(config->readNumEntry("Mail", "Interval", 60)*1000);
	checkmail();
	kdebugm(KDEBUG_INFO, "Mail started\n");
}

Mail::~Mail()
{
	kdebugf();
	ConfigDialog::disconnectSlot("Mail", "Add", SIGNAL(clicked()),	this, SLOT(onAddButton()));
	ConfigDialog::disconnectSlot("Mail", "Remove", SIGNAL(clicked()), this, SLOT(onRemoveButton()));
	ConfigDialog::disconnectSlot("Mail", "Edit", SIGNAL(clicked()),	this, SLOT(onEditButton()));
	ConfigDialog::disconnectSlot("Mail", "Choose", SIGNAL(clicked()), this, SLOT(onFontChange()));
	ConfigDialog::disconnectSlot("Mail", "Run mail client", SIGNAL(toggled(bool)),
			this, SLOT(mailclienttoolged(bool)));
	ConfigDialog::disconnectSlot("Mail", "Check mail in local Maildir", SIGNAL(toggled(bool)),
			this, SLOT(maildirtogled(bool)));
	ConfigDialog::disconnectSlot("Mail", "...", SIGNAL(clicked()),
			this, SLOT(onSelectMaildir()));
	
	ConfigDialog::unregisterSlotOnApplyTab("Mail", this, SLOT(onApplyConfig()));
	ConfigDialog::unregisterSlotOnCreateTab("Mail", this, SLOT(onCreateConfig()));
	ConfigDialog::unregisterSlotOnCloseTab("Mail", this, SLOT(onCloseConfig()));

	ConfigDialog::removeControl("Mail", "Choose");
	ConfigDialog::removeControl("Mail", "FontSel");
	ConfigDialog::removeControl("Mail", "Font");
	ConfigDialog::removeControl("Mail", "Background color");
	ConfigDialog::removeControl("Mail", "Font color");
	ConfigDialog::removeControl("Mail", "Colors");
	ConfigDialog::removeControl("Mail", "...");
	ConfigDialog::removeControl("Mail", "Maildir path");
	ConfigDialog::removeControl("Mail", "Maildir1");
	ConfigDialog::removeControl("Mail", "Check mail in local Maildir");
	ConfigDialog::removeControl("Mail", "Maildir");
	
	ConfigDialog::removeControl("Mail", "Add");
	ConfigDialog::removeControl("Mail", "Remove");
	ConfigDialog::removeControl("Mail", "Edit");
	ConfigDialog::removeControl("Mail", "buttons");
	ConfigDialog::removeControl("Mail", "account");
	ConfigDialog::removeControl("Mail", "Accounts");
	ConfigDialog::removeControl("Mail", "Hint timeout");
	ConfigDialog::removeControl("Mail", "Mail client");
	ConfigDialog::removeControl("Mail", "Run mail client");
	ConfigDialog::removeControl("Mail", "Message format");
	ConfigDialog::removeControl("Mail", "Interval in seconds");
	ConfigDialog::removeControl("Mail", "Checking options");
	
	ConfigDialog::removeTab("Mail");

	// i zapisz wszystko na koniec
	onApplyConfig();

	config->sync();
	delete config;
	delete timer;
	kdebugm(KDEBUG_INFO, "Mail stopped\n");
}

void Mail::checkmail()
{
	kdebugf();
	if(config->readBoolEntry("Mail", "LocalMaildir"))
		maildir();

	Pop3Proto* acc;
	for(acc=accounts.first();acc;acc=accounts.next())
		acc->getStats();
}

void Mail::printstat(int last, int total, int size, QString name)
{
	kdebugf();
	
	if(total>last){
		addHint(icons_manager->loadIcon("Message"), formatmessage(last, total, size, name));
		/*
		 * uruchom klienta poczty
		 */
		if(config->readBoolEntry("Mail", "RunClient")){
			runmailclient();
		}
	}
}

QString Mail::formatmessage(int last, int total, int size, QString name)
{
	QString msg, msgsize;
	/*
	 * %n liczba nowych wiadomoci
	 * %t liczba wszystkich wiadomoci
	 * %s rozmiar
	 * %a nazwa konta
	 */
	msg=config->readEntry("Mail", "Format");
	//ciekawe kto ma tak wielk poczte ??
	if(size>1073741824)
		msgsize.sprintf("%.2f GB", (float)size/1073741824);
	else if(size>1048576)
		msgsize.sprintf("%.2f MB", (float)size/1048576);
	else if(size>1024)
		msgsize.sprintf("%.2f kB", (float)size/1024);
	else
		msgsize.sprintf("%i B", size);

	msg.replace("%n", QString::number(total-last));
	msg.replace("%t", QString::number(total));
	msg.replace("%s", msgsize);
	msg.replace("%a", name);
	return msg;
}

void Mail::onCreateConfig()
{
	kdebugf();
	
/*	QLineEdit* password;
	password=ConfigDialog::getLineEdit("Mail", "Password");
	password->setEchoMode(QLineEdit::Password);*/
	
	ConfigDialog::getColorButton("Mail", "Font color")->setColor( 
			config->readColorEntry("Mail", "HintColor",
				new QColor("black")));
	ConfigDialog::getColorButton("Mail", "Background color")->setColor(
			config->readColorEntry("Mail", "HintBackground",
				new QColor("light gray")));

	fontlabel=ConfigDialog::getLabel("Mail", "FontSel");
	fontlabel->setText(config->readEntry("Mail", "Format"));
	fontlabel->setFont(
			QFont(config->readFontEntry("Mail", "Font",
				new QFont("Luxi Sant"))));

	maildirtogled(config->readBoolEntry("Mail", "MaildirPath", false));
	mailclienttoolged(config->readBoolEntry("Mail", "RunClient", true));

	updateList();
	
	modules_manager->moduleIncUsageCount("mail");
}

void Mail::onCloseConfig()
{
	kdebugf();

	modules_manager->moduleDecUsageCount("mail");
}

void Mail::onApplyConfig()
{
	int i=0;
	Pop3Proto* acc;
	QString config_section;

	for(acc=accounts.first();acc;acc=accounts.next())
	{
		config_section.sprintf("Account_%i", i);
		
		config->writeEntry(config_section, "Name", acc->getName());
		config->writeEntry(config_section, "Server", acc->getHost());
		config->writeEntry(config_section, "ServerPort", acc->getPort());
		config->writeEntry(config_section, "User", acc->getUser());
		config->writeEntry(config_section, "Password", pwHash(acc->getPassword()));
		config->writeEntry(config_section, "Last", acc->getLastmails());	
		config->writeEntry(config_section, "Encryption", acc->getEncryption());

		i++;
	}
	// usuwa (narazie) sie nie da ale puste mona ustawi
	config_section.sprintf("Account_%i", i);
	config->writeEntry(config_section, "Name", "");	
}

void Mail::runmailclient()
{
	kdebugf();
	QProcess *browser;
	QString cmd;
	QStringList args;

	cmd = QString(config->readEntry("Mail","MailClient"));
	
	args = QStringList::split(" ", cmd);
	
	browser = new QProcess();
	browser->setArguments(args);
	
	if (!browser->start())
		hint_manager->addHintError(tr("Could not run your mail client check your setting"));
	delete browser;
}

void Mail::addHint(const QPixmap& pixmap, const QString& text)
{
	hint_manager->addHint(text, 
		pixmap,
		config->readFontEntry("Mail", "Font", new QFont("Luxi Sant")),
		config->readColorEntry("Mail", "HintColor", new QColor("Black")),
		config->readColorEntry("Mail", "HintBackground", new QColor("Gray")),
		config->readNumEntry("Mail", "HintTimeout", 10));
}

void Mail::onFontChange()
{
	kdebugf();
	bool ok;
	QFont font = QFontDialog::getFont(&ok, config->readFontEntry("Mail", "Font"));
	if(ok){
		selectedfont=font;
		fontlabel->setFont(font);
		config->writeEntry("Mail", "Font", selectedfont);
	}
}

void Mail::maildir()
{
	QString path;
	unsigned int totalsize = 0;
	int last;

	path = config->readEntry("Mail", "MaildirPath");
	last = config->readNumEntry("Mail", "LastMailDir", 0);

	path += "/new";
	if(path[0] == '~')
		path.replace(0, 1, QDir::homeDirPath());
	path = QDir::cleanDirPath(path);

	QDir dir(path);

	if(!dir.exists())
		hint_manager->addHintError(tr("Maildir not found!"));
	else if(!dir.isReadable())
		hint_manager->addHintError(tr("Maildir is not readable!"));
	else {
		const QFileInfoList* filelist = dir.entryInfoList();
		if ( filelist ) {
			QFileInfoListIterator iter( *filelist );
			QFileInfo* file;
				while( (file=iter.current()) != 0 ) {
					++iter;
					if ( file->fileName() == "." || file->fileName() == ".." )
						; // nothing
					else
						totalsize += file->size();
					}
		}
		config->writeEntry("Mail", "LastMailDir", last);
		printstat(last, dir.count()-2, totalsize, "MailDir");
	}
}

void Mail::maildirtogled(bool on)
{
	kdebugf();
//	ConfigDialog::getVGroupBox("Mail", "Server options")->setEnabled(!on);
	ConfigDialog::getLineEdit("Mail", "Maildir path")->setEnabled(on);
	ConfigDialog::getPushButton("Mail", "...")->setEnabled(on);

}

void Mail::mailclienttoolged(bool on)
{
	kdebugf();
	ConfigDialog::getLineEdit("Mail", "Mail client")->setEnabled(on);
}

void Mail::onSelectMaildir()
{
	kdebugf();
	QString dir = QFileDialog::getExistingDirectory();
	QLineEdit* lineedit=ConfigDialog::getLineEdit("Mail", "Maildir path");
	if(dir!=QString::null)
		lineedit->setText(dir);
}

void Mail::onAddButton()
{
	kdebugf();
	Pop3Proto* acc=new Pop3Proto("New", "", 0, "", "");
	AccountDialog* dlg=new AccountDialog(acc);
	if(dlg->exec()==QDialog::Accepted){
		connect(acc      , SIGNAL(done(int,int,int,QString)),
				SLOT(printstat(int,int,int,QString)));
		accounts.append(acc);
		updateList();
	}
	else
		delete acc;
}

void Mail::onEditButton()
{
	kdebugf();
	QListBox* list=ConfigDialog::getListBox("Mail", "account");
	Pop3Proto* acc;

	for(acc=accounts.first();acc;acc=accounts.next())
		if(acc->getName()==list->currentText()){
			AccountDialog* dlg=new AccountDialog(acc);
			dlg->exec();
			updateList();
		}
}

void Mail::onRemoveButton()
{
	kdebugf();
	QListBox* list=ConfigDialog::getListBox("Mail", "account");
	Pop3Proto* acc;
	
	for(acc=accounts.first();acc;acc=accounts.next())
		if(acc->getName()==list->currentText()){
			kdebugm(KDEBUG_INFO, "removing %s\n", acc->getName().latin1());
			acc->disconnect(this);
			accounts.remove(acc);
			updateList();
		}

}

void Mail::updateList()
{
	QListBox* list=ConfigDialog::getListBox("Mail", "account");
	list->clear();

	Pop3Proto* acc;
	for(acc=accounts.first();acc;acc=accounts.next())
		list->insertItem(acc->getName());
}

Mail* mail;
