package GCDisplay;

###################################################
#
#  Copyright 2005 Tian
#
#  This file is part of GCfilms.
#
#  GCfilms is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCfilms is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCfilms; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;
use GCUtils;

my $firstYear = 1895;

{
    package GCFilterSearch;

    sub new
    {
        my ($proto, $info) = @_;
        my $class = ref($proto) || $proto;
        my $self  = {};

        bless ($self, $class);

        $self->{info} = $info if $info;
        $self->clear if !$info;

        return $self;
    }

    sub clear
    {
        my $self = shift;
        
        $self->{cleared} = 1;
        
        $self->{info} = {
                        seen => -1,
                        title => '',
                        actors => '',
                        director => '',
                        type => '',
                        borrower => '',
                        borrowed => -1,
                        rating => 0,
                        age => -1,
                        audio => '',
                        subt => '',
                        yearMin => $firstYear,
                        yearMax => 9999
                        };
    }

    sub setOptions
    {
        my ($self, $info) = @_;
        $self->{cleared} = 0;
        $self->{info} = $info;
    }

    sub setSeen
    {
        my ($self, $seen) = @_;
        
        $self->{cleared} = 0;
        $self->{info}->{seen} = $seen;
    }
    
    sub setRating
    {
        my ($self, $rating) = @_;
        
        $self->{cleared} = 0;
        $self->{info}->{rating} = $rating;
    }
    
    sub setAge
    {
        my ($self, $rating) = @_;
        
        $self->{cleared} = 0;
        $self->{info}->{age} = $rating;
    }
    
    sub setBorrowed
    {
        my ($self, $borrowing) = @_;
        
        $self->{cleared} = 0;
        $self->{info}->{borrowed} = $borrowing;
    }
    
    sub setType
    {
        my ($self, $type) = @_;
        
        $self->{cleared} = 0;
        $self->{info}->{type} = $type;
    }

    sub test
    {
        my ($self, $info) = @_;

        return 1 if $self->{cleared};

        return 0 if ($self->{info}->{seen} != -1) && ($info->{seen} != $self->{info}->{seen});
        return 0 if $self->{info}->{title} && ($info->{title} !~ m/$self->{info}->{title}/i);
        return 0 if $self->{info}->{director} && ($info->{director} !~ m/$self->{info}->{director}/i);
        return 0 if $self->{info}->{type} && ($info->{type} !~ m/$self->{info}->{type}(,|$)/i);

        return 0 if $self->{info}->{audio} && ($info->{audio} !~ m/$self->{info}->{audio}/i);
        return 0 if $self->{info}->{subt} && ($info->{subt} !~ m/$self->{info}->{subt}/i);

        return 0 if $self->{info}->{borrower} && ($info->{borrower} ne $self->{info}->{borrower});
        return 0 if (($info->{borrower} eq 'none') || !$info->{borrower})
                 && ($self->{info}->{borrowed} == 1);
        return 0 if ($info->{borrower} ne 'none') && ($self->{info}->{borrowed} == 0);

        return 0 if $self->{info}->{rating} && ($info->{rating} < $self->{info}->{rating});

        return 0 if (($self->{info}->{age} > 1)
                 && (($info->{age} > $self->{info}->{age}) || ($info->{age} < 2)));

        if ($self->{info}->{actors})
        {
            my @actors = split m/,/, $self->{info}->{actors};
            foreach (@actors)
            {
                s/^\s*//;
                s/\s*$//;
                return 0 if $info->{actors} !~ /$_/i;
            }
        }
        my $year = GCUtils::extractYear($info->{date});
        return 0 if !$year && $self->{info}->{yearMin} && ($self->{info}->{yearMin} != $firstYear);
        return 0 if ($year) &&
                (($year < $self->{info}->{yearMin})
              || ($year > $self->{info}->{yearMax}));

        return 1;
    }
}

{
    package GCFilterBorrowing;

    sub new
    {
        my ($proto, $borrowed) = @_;
        my $class = ref($proto) || $proto;
        my $self  = {};
        
        $self->{borrowed} = $borrowed;
        
        bless ($self, $class);
        return $self;
    }

    sub test
    {
        my ($self, $info) = @_;

        return (($info->{borrower} eq 'none') || !$info->{borrower}) 
            if ! $self->{borrowed};
        return (($info->{borrower} ne 'none') && $info->{borrower}) 
    }
}


use Gtk2;

{
    package GCRatingDialog;
    use base "Gtk2::Dialog";

    sub getUserValue
    {
        my $self = shift;
        my $value = -1;
        my $code = $self->run;
        $value = $self->{rating}->get_value if ($code eq 'ok');
        $self->hide;
        return $value;
    }

    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent->{lang}->{RatingTitle},
                              $parent,
                              [qw/modal destroy-with-parent/],
                              @GCDialogs::okCancelButtons
                              );

        my $label = Gtk2::Label->new($parent->{lang}->{RatingLabel});
        $label->set_line_wrap(1);
        $label->set_padding(5,0);
        my  $adj = Gtk2::Adjustment->new(5, 0, 10, 1, 1, 0) ;
        $self->{rating} = Gtk2::SpinButton->new($adj, 0, 0);

        my $hboxRating = new Gtk2::HBox(1,10);

        $self->vbox->set_homogeneous(0);
        $self->vbox->set_spacing(20);
        $self->vbox->pack_start($label, 0, 0, 5);
        $hboxRating->pack_start($self->{rating}, 0, 0, 5);
        $self->vbox->pack_start($hboxRating, 0, 0, 5);
        $self->vbox->show_all;

        bless ($self, $class);
        return $self;
    }
}

use Gtk2;

{
    package GCAgeDialog;
    use base "Gtk2::Dialog";

    sub getUserValue
    {
        my $self = shift;
        my $value = -1;
        my $code = $self->run;
        $value = $self->{rating}->get_value if ($code eq 'ok');
        $self->hide;
        return $value;
    }

    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent->{lang}->{AgeTitle},
                              $parent,
                              [qw/modal destroy-with-parent/],
                              @GCDialogs::okCancelButtons
                              );

        my $label = Gtk2::Label->new($parent->{lang}->{AgeLabel});
        $label->set_line_wrap(1);
        $label->set_padding(5,0);
        my  $adj = Gtk2::Adjustment->new(8, 8, 25, 1, 1, 0) ;
        $self->{rating} = Gtk2::SpinButton->new($adj, 0, 0);

        my $hboxRating = new Gtk2::HBox(1,10);

        $self->vbox->set_homogeneous(0);
        $self->vbox->set_spacing(20);
        $self->vbox->pack_start($label, 0, 0, 5);
        $hboxRating->pack_start($self->{rating}, 0, 0, 5);
        $self->vbox->pack_start($hboxRating, 0, 0, 5);
        $self->vbox->show_all;

        bless ($self, $class);
        return $self;
    }
}


{
    package GCSearchDialog;
    
    use GCGraphicComponents;
    
    use base "Gtk2::Dialog";
    
    sub initValues
    {
        my $self = shift;
        
        my $info = $self->{parent}->{movies}->{filterSearch}->{info};
        
        $self->{rating}->set_value($info->{rating});
        $self->{seen}->[$info->{seen} + 1]->set_active(1);
        
        ($self->{type}->get_children)[0]->set_text($info->{type});
        
        $self->{borrower}->set_active($info->{borrowed} + 1) if $info->{borrowed} < 1;
        $self->{borrowerOption}->remove_menu;
        $self->{borrowerOption}->set_menu($self->{borrower});
                
        $self->{age}->set_text($info->{age});
    }
    
    sub show
    {
        my $self = shift;
        
        $self->setTypes($self->{parent}->{movies}->getHistory('type'));

        $self->setAudios($self->{parent}->{movies}->getHistory('audio'));
        $self->setSubts($self->{parent}->{movies}->getHistory('subt'));
        
        my @borrowers = split m/\|/, $self->{parent}->{options}->borrowers;
        $self->setBorrowers(\@borrowers);
        
        $self->initValues;
        
        $self->SUPER::show();
        $self->show_all;
        $self->{search} = undef;
        my $ended = 0;
        while (!$ended)
        {
            my $response = $self->run;
            if ($response eq 'ok')
            {
                my %info;
            
                $info{title} = $self->{title}->get_text;
                $info{director} = $self->{director}->get_text;
                $info{type} = ($self->{type}->get_children)[0]->get_text;
                $info{actors} = $self->{actors}->get_text;
                $info{age} = $self->{age}->get_text;
                $info{yearMin} = $self->{yearMin}->get_value;
                $info{yearMax} = $self->{yearMax}->get_value;

                $info{audio} = $self->{audioOption}->child->get_label;
                $info{subt} = $self->{subtOption}->child->get_label;
                $self->{previousAudio} = $info{audio};
                $self->{previousSubt} = $info{subt};
            
                my $seen = -1;
                foreach (@{$self->{seen}})
                {
                    last if $_->get_active;
                    $seen++;
                }
                $info{seen} = $seen;
                $info{rating} = $self->{rating}->get_value;
                
                $info{borrower} = $self->{borrowerOption}->child->get_label;
                $info{borrower} =  'none' if $self->{borrowerOption}->child->get_label eq $self->{parent}->{lang}->{PanelNobody};
                $info{borrowed} = -1;

                $self->{parent}->{menubar}->initFilters(
                                                        $info{seen},
                                                        $info{borrower},
                                                        $info{type},
                                                        $info{rating},
                                                        $info{age},
                                                        );

                $self->{search} = \%info;
            }
            $ended = 1 if ($response eq 'ok') || ($response eq 'cancel') || ($response eq 'delete-event');
            $self->clear if ($response eq 'reject');
        }
        $self->hide;
    }
    
    sub clear
    {
        my $self = shift;
        
        $self->{title}->set_text('');
        $self->{director}->set_text('');
        ($self->{type}->get_children)[0]->set_text('');
        $self->{actors}->set_text('');
        $self->{age}->set_text(0);
        $self->{yearMin}->set_value($self->{firstYear});
        $self->{yearMax}->set_value($self->{currentYear});

        $self->{audioOption}->remove_menu;
        $self->{audio}->set_active(0) if $self->{audio};
        $self->{audioOption}->set_menu($self->{audio}) if $self->{audio};
        $self->{subtOption}->remove_menu;
        $self->{subt}->set_active(0) if $self->{subt};
        $self->{subtOption}->set_menu($self->{subt}) if $self->{subt};

        $self->{seen}[0]->set_active(1);
        $self->{rating}->set_value(0);

        $self->{borrowerOption}->remove_menu;
        $self->{borrower}->set_active(0) if $self->{borrower};
        $self->{borrowerOption}->set_menu($self->{borrower}) if $self->{borrower};
    }
    
    sub search
    {
        my $self = shift;
        
        return $self->{search};
    }

    sub setTypes
    {
        my ($self, $values) = @_;

        my $current = ($self->{type}->get_children)[0]->get_text;
        $self->{type}->set_popdown_strings(@$values);
        ($self->{type}->get_children)[0]->set_text($current);
    }
    
    sub setBorrowers
    {
        my ($self, $values) = @_;
        
        my @borrowers = ('', $self->{parent}->{lang}->{PanelNobody}, @{$values}, $self->{parent}->{lang}->{PanelUnknown});

        $self->{borrowerOption}->remove_menu;
        $self->{borrower}->destroy if $self->{borrower};
        $self->{borrower} = new Gtk2::Menu();

        foreach (@borrowers)
        {
            my $item = Gtk2::MenuItem->new_with_label($_);
		    $self->{borrower}->append($item);
        }
        $self->{borrowers} = \@borrowers;
		$self->{borrowerOption}->set_menu($self->{borrower});
        
        $self->{borrowerOption}->show_all;
    }
    
    sub setAudios
    {
        my ($self, $values) = @_;
        
        $self->{audioOption}->remove_menu;
        $self->{audio}->destroy if $self->{audio};
        $self->{audio} = new Gtk2::Menu();

        my $i = 0;
        my $idx = 0;

        foreach (@$values)
        {
            my $item = Gtk2::MenuItem->new_with_label($_);
            $idx = $i if ($_ eq $self->{previousAudio});
		    $self->{audio}->append($item);
		    $i++;
        }
        $self->{audio}->set_active($idx);        
		$self->{audioOption}->set_menu($self->{audio});
                
        $self->{audioOption}->show_all;
    }
    
    sub setSubts
    {
        my ($self, $values) = @_;
        
        $self->{subtOption}->remove_menu;
        $self->{subt}->destroy if $self->{subt};
        $self->{subt} = new Gtk2::Menu();

        my $i = 0;
        my $idx = 0;

        foreach (@$values)
        {
            my $item = Gtk2::MenuItem->new_with_label($_);
            $idx = $i if ($_ eq $self->{previousSubt});
		    $self->{subt}->append($item);
		    $i++;
        }
        $self->{subt}->set_active($idx);        
		$self->{subtOption}->set_menu($self->{subt});
        
        $self->{subtOption}->show_all;
    }

    
    sub new
    {
        my ($proto, $parent) = @_;
        my $class = ref($proto) || $proto;
        my $self  = $class->SUPER::new($parent->{lang}->{SearchTitle},
                              $parent,
                              [qw/modal destroy-with-parent/],
                              'gtk-clear' => 'reject',
                              @GCDialogs::okCancelButtons
                              );

        $self->{parent} = $parent;

        #my $header = new GCDialogHeader($parent->{lang}->{SearchTitle}, 'find', $parent->{logosDir});
        #$self->vbox->pack_start($header, 0, 0, 0);

        my $table = new Gtk2::Table(14,2,0);
        $table->set_row_spacings(5);
        $table->set_col_spacings(10);
        $table->set_border_width(20);

        my $row = 0;

        my $titleLabel = new Gtk2::Label($parent->{lang}->{SearchMovieTitle});
        $titleLabel->set_alignment(0,0.5);
        $self->{title} = new Gtk2::Entry;
        $self->{title}->signal_connect('activate' => sub {$self->response('ok')} );
        $table->attach($titleLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{title}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);

        $row++;

        my $directorLabel = new Gtk2::Label($parent->{lang}->{SearchDirector});
        $directorLabel->set_alignment(0,0.5);
        $self->{director} = new Gtk2::Entry;
        $self->{director}->signal_connect('activate' => sub {$self->response('ok')} );
        $table->attach($directorLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{director}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);

        $row++;

        my $typeLabel = new Gtk2::Label($parent->{lang}->{SearchType});
        $typeLabel->set_alignment(0,0.5);
        $self->{type} = Gtk2::Combo->new;
        ($self->{type}->get_children)[0]->set_editable(0);
        $table->attach($typeLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{type}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);
        
        $row++;
        
        my $actorsLabel = new Gtk2::Label($parent->{lang}->{SearchActors});
        $actorsLabel->set_alignment(0,0.5);
        $self->{actors} = new Gtk2::Entry;
        $self->{actors}->signal_connect('activate' => sub {$self->response('ok')} );
        $table->attach($actorsLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{actors}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);

        $row++;

        my $ageLabel = new Gtk2::Label($parent->{lang}->{SearchAge});
        $ageLabel->set_alignment(0,0.5);
        $self->{age} = new GCAgeChooser($parent->{lang}, 'public',150);
        $table->attach($ageLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{age}, 1, 2, $row, $row + 1, 'fill', 'fill', 0, 0);

        $row++;

        my $yearLabel = new Gtk2::Label($parent->{lang}->{SearchYear});
        $yearLabel->set_alignment(0,0.5);
        my $betweenYearLabel = new Gtk2::Label($parent->{lang}->{SearchYearBetween});
        $self->{firstYear} = $firstYear;
        my $adjYear1 = Gtk2::Adjustment->new($self->{firstYear}, $self->{firstYear}, 9999, 1, 10, 0) ;
        $self->{yearMin} = Gtk2::SpinButton->new($adjYear1, 0, 0);
        $self->{yearMin}->set_numeric(1);
        $self->{yearMin}->signal_connect('activate' => sub {$self->response('ok')} );
        my ($secondes, $minutes, $hours, $day, $month, $year) = localtime;
        $self->{currentYear} = $year + 1900;
        my $adjYear2 = Gtk2::Adjustment->new($self->{currentYear}, $self->{firstYear}, 9999, 1, 10, 0) ;
        $self->{yearMax} = Gtk2::SpinButton->new($adjYear2, 0, 0);
        $self->{yearMax}->set_numeric(1);
        $self->{yearMax}->signal_connect('activate' => sub {$self->response('ok')} );
        $self->{yearBox} = new Gtk2::HBox(0,0);
        $self->{yearBox}->pack_start($self->{yearMin},1,1,0);
        $self->{yearBox}->pack_start($betweenYearLabel,1,1,20);
        $self->{yearBox}->pack_start($self->{yearMax},1,1,0);
        $table->attach($yearLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{yearBox}, 1, 2, $row, $row + 1, 'fill', 'fill', 0, 0);

        $row++;

        $table->attach(Gtk2::HSeparator->new, 0, 2, $row, $row + 1, 'fill', 'fill', 0, 0);

        $row++;

        my $audioLabel = new Gtk2::Label($parent->{lang}->{Displayed}->{audio});
        $audioLabel->set_alignment(0,0.5);
        $self->{audioOption} = new Gtk2::OptionMenu();
        $self->{audioOption}->signal_connect('activate' => sub {$self->response('ok')} );
        $table->attach($audioLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{audioOption}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);

        $row++;

        my $subLabel = new Gtk2::Label($parent->{lang}->{Displayed}->{subt});
        $subLabel->set_alignment(0,0.5);
        $self->{subtOption} = new Gtk2::OptionMenu();
        $self->{subtOption}->signal_connect('activate' => sub {$self->response('ok')} );
        $table->attach($subLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{subtOption}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);

        $row++;

        $table->attach(Gtk2::HSeparator->new, 0, 2, $row, $row + 1, 'fill', 'fill', 0, 0);
        
        $row++;
        
        my $hboxSeen = new Gtk2::HBox(0,0);
        my $labelSeen = new Gtk2::Label($parent->{lang}->{SearchSeen});
        $labelSeen->set_alignment(0,0.5);
		$self->{seen}[0] = new  Gtk2::RadioButton(undef,$parent->{lang}->{SearchUndef});
		my $seenGroup = $self->{seen}[0]->get_group;
        $self->{seen}[1] = new  Gtk2::RadioButton($seenGroup,$parent->{lang}->{SearchSeenNo});
        $self->{seen}[2] = new  Gtk2::RadioButton($seenGroup,$parent->{lang}->{SearchSeenYes});
 		$hboxSeen->pack_start($self->{seen}[0],0,0,10);
		$hboxSeen->pack_start($self->{seen}[1],0,0,10);
		$hboxSeen->pack_start($self->{seen}[2],0,0,10);
        $table->attach($labelSeen, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($hboxSeen, 1, 2, $row, $row + 1, 'fill', 'fill', 0, 0);
        
        $row++;
        
        my $ratingLabel = new Gtk2::Label($parent->{lang}->{SearchRating});
        $ratingLabel->set_alignment(0,0.5);
        my  $adj = Gtk2::Adjustment->new(0, 0, 10, 1, 1, 0) ;
        $self->{rating} = Gtk2::SpinButton->new($adj, 0, 0);
        $self->{rating}->set_numeric(1);
        $self->{rating}->signal_connect('activate' => sub {$self->response('ok')} );
        $table->attach($ratingLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{rating}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);

        $row++;

        $table->attach(Gtk2::HSeparator->new, 0, 2, $row, $row + 1, 'fill', 'fill', 0, 0);
        
        $row++;
        
        my $borrowerLabel = new Gtk2::Label($parent->{lang}->{Displayed}->{borrower});
        $borrowerLabel->set_alignment(0,0.5);
        $self->{borrowerOption} = new Gtk2::OptionMenu();
        $self->{borrowerOption}->signal_connect('activate' => sub {$self->response('ok')} );
        $table->attach($borrowerLabel, 0, 1, $row, $row + 1, 'fill', 'fill', 0, 0);
        $table->attach($self->{borrowerOption}, 1, 2, $row, $row + 1, 'fill', 'expand', 0, 0);
        
        $self->vbox->pack_start($table, 1, 1, 0);
        $self->vbox->show_all;

        bless ($self, $class);
        return $self;
    }
}


1;